package com.twentyfouri.tvlauncher.adapters

import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartPageSection
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.data.AppListItem
import com.twentyfouri.tvlauncher.widgets.RowItemView
import com.twentyfouri.tvlauncher.data.channel.ExternalRowItem

class RowItemAdapter(section: SmartPageSection)
    : ListAdapter<Any, RowItemAdapter.RowItemViewHolder>(SmartMediaItemDiffCallback()) {
    enum class RowItemType {
        EVENT,
        CHANNEL,
        APP,
        GOOGLE_PLAY,
        EXTERNAL_CHANNEL,
        GRID
    }

    private val rowItemType: RowItemType = when (Flavor().getPlaylistType(section)) {
        PlaylistType.EXTERNAL_CHANNEL -> RowItemType.EXTERNAL_CHANNEL
        PlaylistType.PLAY_STORE -> RowItemType.GOOGLE_PLAY
        PlaylistType.ALL_APPS,
        PlaylistType.ALL_GAMES,
        PlaylistType.FAVORITE_APPS,
        PlaylistType.RECOMMENDED_APPS -> RowItemType.APP
        PlaylistType.ALL_CHANNELS,
        PlaylistType.ALL_CHANNELS_PAGED -> RowItemType.CHANNEL
        PlaylistType.GRID -> RowItemType.GRID
        else -> RowItemType.EVENT
    }

//    private val debugLabel = section.label

    val isChannelRow = rowItemType == RowItemType.CHANNEL

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RowItemViewHolder {
        val rowItemView = LayoutInflater.from(parent.context).inflate(
            when (rowItemType) {
                RowItemType.GOOGLE_PLAY -> R.layout.row_item_google_play_for_viewholder
                RowItemType.APP,
                RowItemType.CHANNEL,
                RowItemType.GRID,
                RowItemType.EVENT -> R.layout.row_item_for_viewholder
                RowItemType.EXTERNAL_CHANNEL ->
                    if (viewType == FIRST_ITEM_VIEW_TYPE) {
                        R.layout.row_item_icon_for_viewholder
                    } else {
                        R.layout.row_item_for_viewholder
                    }
            },
            parent,
            false
        ) as RowItemView
        return RowItemViewHolder(rowItemView)
    }

    override fun onBindViewHolder(holder: RowItemViewHolder, position: Int) {
//        Log.d("Rows3", "bind $debugLabel $position")
        holder.view.apply {
            when (rowItemType) {
                RowItemType.GOOGLE_PLAY -> bindGooglePlay(getItem(position) as AppListItem, position)
                RowItemType.APP -> bind(getItem(position) as AppListItem, position)
                RowItemType.EXTERNAL_CHANNEL -> bindAppChannelProgram(getItem(position) as ExternalRowItem, position)
                RowItemType.GRID -> bind(getItem(position) as SmartMediaItem, position, true)
                else -> bind(getItem(position) as SmartMediaItem, position)
            }
        }
    }

    override fun onBindViewHolder(holder: RowItemViewHolder, position: Int, payloads: List<Any>) {
        // just redirect to regular bind
        onBindViewHolder(holder, position)
    }

    override fun onViewDetachedFromWindow(holder: RowItemViewHolder) {
//        Log.d("Rows3", "detach $debugLabel ${holder.adapterPosition}")
        super.onViewDetachedFromWindow(holder)
        holder.view.onViewDetachedFromWindow()
    }

    override fun onViewAttachedToWindow(holder: RowItemViewHolder) {
//        Log.d("Rows3", "attach $debugLabel ${holder.adapterPosition}")
        super.onViewAttachedToWindow(holder)
        holder.view.onViewAttachedToWindow()
    }

    override fun getItemId(position: Int): Long {
        return position.toLong()
    }

    class RowItemViewHolder(
        val view : RowItemView
    ) : RecyclerView.ViewHolder(view)

    class SmartMediaItemDiffCallback : DiffUtil.ItemCallback<Any>() {

        override fun areItemsTheSame(oldItem: Any, newItem: Any): Boolean {
            when (oldItem) {
                is AppListItem -> {
                    if (newItem is AppListItem) {
                        return oldItem.getLabelAsSafeString() == newItem.getLabelAsSafeString()
                    }
                    return false
                }
                is SmartMediaItem -> {
                    if (newItem is SmartMediaItem) {
                        return oldItem.reference == newItem.reference
                    }
                    return false
                }
                is ExternalRowItem -> {
                    if (newItem is ExternalRowItem) {
                        return oldItem.id == newItem.id
                    }
                    return false
                }
            }
            return oldItem == newItem
        }

        override fun areContentsTheSame(oldItem: Any, newItem: Any): Boolean {
            //This method is called when "areItemsTheSame" return true to check if some value of item is changed
            //unfortunately our design depends also on position in the list (first/last) and on selection (enlarged) which is not available
            //here. Seems that returning FALSE generally solve problem of holes in the list and also duplicated items (which is just not updated
            //item) but do not have visible impact on performance.
            //TODO If this is problem we would need to provide here more info (position/selection)
            return false
        }

        override fun getChangePayload(oldItem: Any, newItem: Any): Any? {
            return newItem
        }
    }

    override fun getItemViewType(position: Int): Int {
        //To identify first item
        return if (position == 0) {
            FIRST_ITEM_VIEW_TYPE
        } else super.getItemViewType(position)
    }

    companion object{
        private const val FIRST_ITEM_VIEW_TYPE = 1
    }
}