package com.twentyfouri.tvlauncher.data

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.ApplicationInfo
import android.content.pm.PackageManager
import android.content.pm.ResolveInfo
import android.provider.Settings
import android.telephony.SignalStrength
import android.text.TextUtils
import android.util.ArraySet
import android.util.Log
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.data.SpecialAppLists.getHighPriorityApps
import com.twentyfouri.tvlauncher.data.SpecialAppLists.getHiddenAppsList
import com.twentyfouri.tvlauncher.data.SpecialAppLists.getRecommendedAppList
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.utils.SharedPreferencesUtils
import com.twentyfouri.tvlauncher.ui.EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE
import com.twentyfouri.tvlauncher.ui.FAVORITE_APP_PICKER
import com.twentyfouri.tvlauncher.ui.FavoriteAppsActivity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import timber.log.Timber

private fun AppListItem.getPackageName() = this.intent.component?.packageName ?: ""
internal fun ResolveInfo.getPackageName() = this.activityInfo.packageName

open class AppListDao(
    c: Context,
    private val appUsageDatabase: AppUsageDatabase
) {

    private val context = c.applicationContext
    private var recommended = mutableListOf<String>()

    //recommendedAppsList property is accessed many times but in case of install/uninstall we need to refresh the list
    //because of that cache and refresh properties are introduced
    private var recommendedAppsListNeedRefresh = true
    private var recommendedAppsListCache: List<String> = emptyList()
    private val recommendedAppsList: List<String>
        get() {
            if (recommendedAppsListNeedRefresh) {
                recommendedAppsListNeedRefresh = false
                recommendedAppsListCache = filterReplacedRecommendedAppList(context, getRecommendedAppList(context))
            }
            return recommendedAppsListCache
        }

    private val hiddenAppsSet: Set<String> = getHiddenAppsList(context).toHashSet()
    private val highPriorityAppsOrderIndices =
        getHighPriorityApps(context).withIndex().associateBy({ it.value }, { it.index })
    private val appsComparator: Comparator<AppListItem> = compareBy({
        highPriorityAppsOrderIndices.getOrDefault(it.getPackageName(), getHighPriorityApps(context).size)
    }, { it.timeOfUse * (-1) })

    private fun filterReplacedRecommendedAppList(context: Context, recommendedApps: List<String>): List<String> {
        val intentActivities = context.packageManager.queryIntentActivities(
            Intent(Intent.ACTION_MAIN).addCategory(Intent.CATEGORY_LEANBACK_LAUNCHER),
            0
        )
        val filtered = recommendedApps.toMutableList()
        Flavor().getReplacedRecommendedApps(context).forEach { pair ->
            val oldApp = intentActivities.find { it.getPackageName() == pair.first }
            val newApp = intentActivities.find { it.getPackageName() == pair.second }
            if(oldApp != null && newApp != null) {
                filtered.remove(pair.first)
            }
        }
        return filtered
    }

    fun getAppList(): MutableList<AppListItem> = getAppList { resolveInfo: ResolveInfo ->
        !isGame(resolveInfo)
                && (!isRecommended(resolveInfo) || highPriorityAppsOrderIndices.keys.contains(resolveInfo.getPackageName()))
    }

    fun getRecommendedAppList(recommendedApps: List<String>?, lookingForFavorites: Boolean = false, refreshNeeded: Boolean = false): MutableList<AppListItem> {
        if(refreshNeeded) recommendedAppsListNeedRefresh = true
        return if (recommendedApps?.isNotEmpty() == true) {
            recommended = mutableListOf()
            recommended.addAll(recommendedApps)
            checkDefaultRecommendedAppsChanged(recommendedApps)
            getRecommendedAppItemList(recommendedApps, lookingForFavorites)
        } else {
            checkDefaultRecommendedAppsChanged(recommendedAppsList)
            getRecommendedAppItemList(recommendedAppsList, lookingForFavorites)
        }
    }

    // This is responsible for resetting Favorite Apps in case of any change in recommended apps
    // either by server or local configuration
    private fun checkDefaultRecommendedAppsChanged(recommendedApps: List<String>){
        val defaultRecommendedAppsPersistent = SharedPreferencesUtils.getDefaultRecommendedApps()
        if (recommendedApps.toSet() != defaultRecommendedAppsPersistent) {
            SharedPreferencesUtils.putDefaultRecommendedApps(null)
        }
    }

    fun getFavoriteAppList(recommendedApps: List<String>?): MutableList<AppListItem> {
        val recommendedAppListNonEmpty =
            if (recommendedApps?.isNotEmpty() == true) recommendedApps else recommendedAppsList
        if (SharedPreferencesUtils.getDefaultRecommendedApps() == null) {
            // Setup favorite apps row with recommendation apps content
            SharedPreferencesUtils.putDefaultRecommendedApps(recommendedAppListNonEmpty.toSet())
            val recommendedAppsAsFavorites = getRecommendedAppList(
                recommendedApps = recommendedApps,
                lookingForFavorites = true
            )
            recommendedAppsAsFavorites.forEachIndexed { index, appListItem ->
                // Initial populate of Favorite apps persistence
                storeFavoriteApp((appListItem), index + 1)
            }
            recommendedAppsAsFavorites.add(getAddFavouriteAppItem())
            return recommendedAppsAsFavorites
        } else {
            // Get favorite apps row content using FavoriteAppsDatabase
            val allApps = getFavoriteAppSelectionList(lookingForFavorites = true, favorites = null)
            val favoriteAppsFromDatabase = getFavoriteAppsFromDatabase()
            val favoriteApps = favoriteAppsFromDatabase.flatMap { app ->
                allApps.filter {
                    app.packageName == (it.intent.component?.packageName ?: it.intent.`package`)
                }
            }.toMutableList()
            GlobalScope.launch(Dispatchers.IO) { removeDeletedFavoriteApps(favoriteAppsFromDatabase, favoriteApps.toMutableList()) }
            favoriteApps.add(getAddFavouriteAppItem())
            return favoriteApps
        }
    }

    fun getGameList(): MutableList<AppListItem> = getAppList { resolveInfo: ResolveInfo ->
        isGame(resolveInfo)
    }

    fun getFavoriteAppSelectionList(lookingForFavorites: Boolean = false, favorites: List<AppListItem>?): MutableList<AppListItem> =
        getAppList(lookingForFavorites = lookingForFavorites) { resolveInfo: ResolveInfo ->
            favorites?.firstOrNull {
                resolveInfo.activityInfo.packageName == it.intent.component?.packageName ?: it.intent.`package` ?: ""
            } == null
        }

    @Suppress("DEPRECATION")
    private fun isGame(resolveInfo: ResolveInfo): Boolean {
        val appInfo = resolveInfo.activityInfo.applicationInfo
        // category is -1 = CATEGORY_UNDEFINED for some games (especially older ones), we need to use FLAG_IS_GAME instead, even though it is deprecated
        return (appInfo.category == ApplicationInfo.CATEGORY_GAME || (appInfo.flags and ApplicationInfo.FLAG_IS_GAME == ApplicationInfo.FLAG_IS_GAME))
    }

    private fun isRecommended(resolveInfo: ResolveInfo): Boolean {
        return if (recommended.isNotEmpty())
            recommended.contains(resolveInfo.getPackageName())
        else
            recommendedAppsList.contains(resolveInfo.getPackageName())
    }

    private fun getAppList( lookingForFavorites: Boolean = false, filterCheck: (ResolveInfo) -> Boolean): MutableList<AppListItem> {
        Timber.tag("AppListRepository").d("getAppList/getGameList")
        val packageManager = context.packageManager
        val intentActivities = packageManager.queryIntentActivities(
            Intent(Intent.ACTION_MAIN).addCategory(Intent.CATEGORY_LEANBACK_LAUNCHER),
            0
        )
        val appListItems = ArrayList<AppListItem>(intentActivities.size)
        for (info in intentActivities) {
            if (filterCheck(info) && !hiddenAppsSet.contains(info.getPackageName())) {
                val intent = if (lookingForFavorites) {
                    Flavor().getAppChannelsDelegate()?.getFavoriteAppIntent(info)
                } else {
                    Flavor().getAppChannelsDelegate()?.getAppIntent(info)
                }
                intent?.let { appIntent ->
                    val appListItem = AppListItem(
                        intent = appIntent,
                        banner = info.activityInfo.loadBanner(packageManager),
                        icon = info.loadIcon(packageManager),
                        label = info.loadLabel(packageManager),
                        timeOfUse = getPackagePriority(info),
                        isFavouriteItem = lookingForFavorites
                    )
                    appListItems.add(appListItem)
                }
            }
        }

        appListItems.sortWith(appsComparator)
        return appListItems
    }

    private fun getRecommendedAppItemList(recommendedApps: List<String>, lookingForFavorites: Boolean = false): MutableList<AppListItem> {
        val packageManager = context.packageManager
        val intentActivities = packageManager.queryIntentActivities(
            Intent(Intent.ACTION_MAIN).addCategory(Intent.CATEGORY_LEANBACK_LAUNCHER),
            0
        )
        val appListItems = ArrayList<AppListItem>(intentActivities.size)
        // for recommended we ignore last used priority, instead show order as from original list
        var priorityIndex = 1L
        for (app in recommendedApps) {
            for (info in intentActivities) {
                if (info.getPackageName() == app && !hiddenAppsSet.contains(info.getPackageName())) {
                    val intent = Flavor().getAppChannelsDelegate()?.getRecommendedAppIntent(info)
                    intent?.let { appIntent ->
                        val appListItem = AppListItem(
                            intent = appIntent,
                            banner = info.activityInfo.loadBanner(packageManager),
                            icon = info.loadIcon(packageManager),
                            label = info.loadLabel(packageManager),
                            timeOfUse = priorityIndex++,
                            isFavouriteItem = lookingForFavorites
                        )
                        appListItems.add(appListItem)
                    }
                }
            }
        }
        return if (appListItems.isNotEmpty())
            appListItems
        else {
            recommended = recommendedAppsList.toMutableList()
            if (recommendedApps == recommended) arrayListOf() //prevent infinite loop
            else getRecommendedAppItemList(recommendedAppsList)
        }
    }

    // Hardcoded Favorite App Item that opens App Selection fragment
    fun getAddFavouriteAppItem(): AppListItem{
        return AppListItem(
            intent = Intent(context, FavoriteAppsActivity::class.java).apply {
                putExtra(EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE, FAVORITE_APP_PICKER)
            },
            banner = context.resources.getDrawable(R.drawable.add_favorite_app_background, null),
            label = context.getString(com.twentyfouri.tvlauncher.R.string.add_app_to_favorites)
        )
    }

    fun getSettingsList(signalStrength: SignalStrength?): ArraySet<AppListItem> {
        val packageManager = context.packageManager
        val networkIntents = packageManager.queryIntentActivities(
            Intent(Settings.ACTION_WIFI_SETTINGS)
                .addCategory(CATEGORY_LEANBACK_SETTINGS), 0
        )
        val intentActivities = packageManager.queryIntentActivities(
            Intent(Intent.ACTION_MAIN)
                .addCategory(CATEGORY_LEANBACK_SETTINGS),
            PackageManager.GET_RESOLVED_FILTER
        )
        val settingsItems = ArraySet<AppListItem>(intentActivities.size)
        for (info in intentActivities) {
            if (info.activityInfo == null) {
                continue
            }
            var isNetwork = false
            for (networkInfo in networkIntents) {
                if (networkInfo.activityInfo == null) {
                    continue
                }
                if (TextUtils.equals(
                        networkInfo.activityInfo.name,
                        info.activityInfo.name
                    ) && TextUtils.equals(
                        networkInfo.activityInfo.packageName,
                        info.activityInfo.packageName
                    )
                ) {
                    isNetwork = true
                    break
                }
            }
            val priority = info.priority

            val appListItem: AppListItem
            val intent = Intent.makeMainActivity(
                ComponentName(
                    info.activityInfo.packageName,
                    info.activityInfo.name
                )
            )
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
            if (isNetwork) {
                appListItem = AppListItemNetwork(
                    context = context,
                    intent = intent,
                    signalStrength = signalStrength,
                    priority = priority.toLong()
                )
            } else {
                appListItem = AppListItem(
                    intent = intent,
                    banner = info.activityInfo.loadBanner(packageManager),
                    icon = info.loadIcon(packageManager),
                    label = info.loadLabel(packageManager),
                    timeOfUse = priority.toLong()
                )
            }
            settingsItems.add(appListItem)
        }

        val notificationsAppListItem = AppListItemNotifications(
            context = context
        )

        settingsItems.add(notificationsAppListItem)

        return settingsItems
    }

    private fun getPackagePriority(info: ResolveInfo): Long =
        appUsageDatabase.appUsageDao().getTimeOfUseByPackageName(info.activityInfo.packageName)

    fun getPlayStoreList(): MutableList<AppListItem> {
        Timber.tag("AppListRepository").d("getPlayStoreList")
        val playStoreItems = ArrayList<AppListItem>()
        try {
            val playStore = getAppList { resolveInfo: ResolveInfo ->
                resolveInfo.activityInfo.packageName == PLAY_STORE_PACKAGE_NAME
            }.first().apply {
                label = context.getString(R.string.app_list_more_apps)
                timeOfUse = 1
            }
            playStoreItems.add(playStore)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        try {
            val googleGames = getAppList { resolveInfo: ResolveInfo ->
                resolveInfo.activityInfo.packageName == PLAY_GAMES_PACKAGE_NAME
            }.first().apply {
                label = context.getString(R.string.app_list_more_games)
                timeOfUse = 0
            }
            playStoreItems.add(googleGames)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return playStoreItems
    }

    fun getFavoriteAppsFromDatabase():List<AppUsage> = appUsageDatabase.appUsageDao().getAllFavoriteApps()

    fun storeFavoriteApp(appItem: AppListItem, favoritePriority: Int) {
        appUsageDatabase.appUsageDao().insertFavoriteApps(
           listOf(
               AppUsage(
                   packageName = appItem.intent.component?.packageName ?: appItem.intent.`package` ?: "",
                   timeOfUse = appItem.timeOfUse,
                   favoritePriority = favoritePriority
               )
           )
        )
    }

    fun insertFavoriteApp(favoriteApps: List<AppUsage>) {
        appUsageDatabase.appUsageDao().insertFavoriteApps(favoriteApps)
    }

    // Function updates favorite apps with new order after user change
    fun updateFavoriteAppsPriority(appList: List<Any>) {
        val favoriteApps = appList.mapIndexedNotNull { index, appListItem ->
            val appItem = (appListItem as? AppListItem) ?: return@mapIndexedNotNull null
            val packageName = appItem.intent.component?.packageName ?: appItem.intent.`package` ?: return@mapIndexedNotNull null
            AppUsage(
                packageName,
                appItem.timeOfUse,
                index + 1
            )
        }
        if (favoriteApps.isNullOrEmpty().not()) appUsageDatabase.appUsageDao().insertFavoriteApps(favoriteApps)
    }

    // In case app was uninstalled by user remove it from FavoriteAppsDatabase
    private fun removeDeletedFavoriteApps(
        favoriteAppsFromDatabase: List<AppUsage>,
        favoriteAppsCurrent: MutableList<AppListItem?>
    ) {
        val appsToDelete = favoriteAppsFromDatabase.filterNot { favoriteAppPersistent ->
            favoriteAppsCurrent.toList().any {
                favoriteAppPersistent.packageName == (it?.intent?.component?.packageName ?: it?.intent?.`package`)
            }
        }.map { it.packageName }
        appsToDelete.ifEmpty { return }
        // Remove deleted app from AppUsage database
        appUsageDatabase.appUsageDao().deleteFavoriteApps(appsToDelete)
        val favoriteAppsToUpdateOrder = appsToDelete.flatMap { deletedApp ->
            favoriteAppsFromDatabase.filter { deletedApp != it.packageName }
        }.mapIndexed { index, appUsage ->
            appUsage.apply { favoritePriority = index + 1 }
        }
        // Update priority of favorite apps in AppUsage database
        appUsageDatabase.appUsageDao().insertFavoriteApps(favoriteAppsToUpdateOrder)
    }

    companion object {
        /**
         * Intent category for launching TV Settings. This was copied from the system API for sample
         * purposes. If building using a system SDK, use Intent.CATEGORY_LEANBACK_SETTINGS instead.
         */
        const val CATEGORY_LEANBACK_SETTINGS = "android.intent.category.LEANBACK_SETTINGS"
        const val TVRECOMMENDATIONS_PACKAGE_NAME = "com.google.android.tvrecommendations"
        const val PLAY_STORE_PACKAGE_NAME = "com.android.vending"
        const val PLAY_GAMES_PACKAGE_NAME = "com.google.android.play.games"
    }
}