package com.twentyfouri.tvlauncher.data

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.database.Cursor
import android.media.tv.TvContract
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MediatorLiveData
import androidx.lifecycle.MutableLiveData
import androidx.tvprovider.media.tv.TvContractCompat
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.data.channel.ExternalRow
import com.twentyfouri.tvlauncher.data.channel.ExternalRowItem
import com.twentyfouri.tvlauncher.notifications.NotificationsContract
import com.twentyfouri.tvlauncher.notifications.ReadNotifications
import com.twentyfouri.tvlauncher.notifications.TvNotification
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import timber.log.Timber
import kotlinx.coroutines.*
import kotlin.time.Duration.Companion.milliseconds

class ContentResolverRepository (
    val context: Context,
    readNotifications: ReadNotifications
) {
    private val innerText = MutableLiveData<String>().apply { value = DEFAULT_INNER_TEXT }
    private val enabled = MutableLiveData<Boolean>().apply { value = false }
    private val hasUnread: LiveData<Boolean>
    private val notificationKeys = MutableLiveData<Set<String>>()

    private val contentResolver = context.contentResolver
    private var loadNotifsCountJob: Job? = null
    private var loadNotifsUnreadJob: Job? = null

    private var notificationReceiver: NotificationReceiver? = null

    init {
        hasUnread = MediatorLiveData<Boolean>().apply {
            value = false
            addSource(readNotifications) { notificationKeys.value?.also { keys -> value = readNotifications.hasUnread(keys)} }
            addSource(notificationKeys) { value = readNotifications.hasUnread(it) }
        }
    }

    fun registerNotificationReceiverOn(context: Context) {
        notificationReceiver = NotificationReceiver()
        val filter = IntentFilter()
        filter.addAction(NOTIFICATION_UPDATE_INTENT_ACTION)
        context.registerReceiver(notificationReceiver, filter)
    }

    fun unregisterNotificationReceiverOn(context: Context) {
        try {
            context.unregisterReceiver(notificationReceiver)
        } catch (e: IllegalArgumentException) {
            Timber.d("UNregisterNotificationReceiverOn ... ${e.message}")
        }
    }

    private fun startLoadNotifsCountJob() {
        if (loadNotifsCountJob?.isActive != true) {
            loadNotifsCountJob = CoroutineScope(Dispatchers.Default).launch {
                delay(DELAY)
                loadNotifsCount()
            }
        }
    }

    private suspend fun loadNotifsCount() {
        try {
            withContext(Dispatchers.IO) {
                contentResolver.query(
                    NotificationsContract.NOTIFS_COUNT_URI,
                    null,
                    null,
                    null,
                    null
                )
            }?.use { cursor ->
                if (cursor.moveToFirst()) {
                    val columnIndex = cursor.getColumnIndex(NotificationsContract.COLUMN_COUNT)
                    val count = cursor.getInt(columnIndex)
                    innerText.postValue(count.toString())
                    enabled.postValue(count > 0)
                } else {
                    innerText.postValue(DEFAULT_INNER_TEXT)
                }
            }
        } catch (t: Throwable) {
            innerText.postValue(DEFAULT_INNER_TEXT)
        }
    }

    private fun startLoadNotifsUnreadJob() {
        if (loadNotifsUnreadJob?.isActive != true) {
            loadNotifsUnreadJob = CoroutineScope(Dispatchers.Default).launch {
                delay(DELAY)
                loadNotifsUnread()
            }
        }
    }

    private suspend fun loadNotifsUnread() {
        try {
            withContext(Dispatchers.IO) {
                contentResolver.query(
                    NotificationsContract.CONTENT_URI,
                    TvNotification.PROJECTION,
                    null,
                    null,
                    null
                )
            }?.use { cursor ->
                notificationKeys.postValue(ReadNotifications.getKeysFromCursor(cursor))
            }
        } catch (t: Throwable) {
            notificationKeys.postValue(emptySet())
        }
    }

    fun getInnerTextLD(): LiveData<String> {
        startLoadNotifsCountJob()
        return innerText
    }

    fun getEnabledLD(): LiveData<Boolean> {
        startLoadNotifsCountJob()
        return enabled
    }

    fun getHasUnreadLD(): LiveData<Boolean> {
        startLoadNotifsUnreadJob()
        return hasUnread
    }

    fun updateNotificationLD(){
        startLoadNotifsCountJob()
        startLoadNotifsUnreadJob()
    }

    @Suppress("RedundantSuspendModifier")
    suspend fun loadChannels(): List<ExternalRow> {
        var cursor: Cursor? = null
        var channels: ArrayList<ExternalRow> = ArrayList()
        var throwable: Throwable? = null
        try {
            cursor = contentResolver.query(
                TvContractCompat.Channels.CONTENT_URI,
                ExternalRow.PROJECTION,
                null,
                null,
                null
            )

            if (cursor != null) {
                try {
                    while (cursor.moveToNext()) {
                        val homeChannel = ExternalRow.fromCursor(cursor)
                        homeChannel.channelLogoUri = TvContract
                            .buildChannelLogoUri(homeChannel.id)
                            .buildUpon()
                            .appendQueryParameter("t", System.currentTimeMillis().toString())
                            .build()
                        channels.add(homeChannel)
                        Timber.tag("CHANNEL_TEST").d("channels.add(homeChannel) ${homeChannel.displayName}")
                    }
                } catch (t: Throwable) {
                    if (throwable != null) throwable.addSuppressed(t) else throwable = t
                }

            } else {
                Timber.tag("CHANNEL_TEST").e("error loading home channels, cursor is null")
            }
            try {
                cursor?.close()
            } catch (t: Throwable) {
                if (throwable != null) throwable.addSuppressed(t) else throwable = t
            }
//            channels.forEach {
//                Log.d("CHANNEL_TEST", "channel - $it")
//            }
        } catch (t: Throwable) {
            cursor?.close()
            if (throwable != null) throwable.addSuppressed(t) else throwable = t
        }
        throwable?.printStackTrace()
        return channels
    }

    @Suppress("RedundantSuspendModifier")
    suspend fun loadChannelPrograms(channelId: Long, sectionIconUri: String): List<ExternalRowItem> {
        var cursor: Cursor? = null
        val externalRowItems: ArrayList<ExternalRowItem> = ArrayList()
        var throwable: Throwable? = null
        try {
            cursor = contentResolver.query(
                TvContractCompat.PreviewPrograms.CONTENT_URI,
                ExternalRowItem.PROJECTION,
                null,
                null,
                "weight DESC"
            )
            try {
                var skipAddingDummyBannerItem = Flavor().disableAddBannerToAppChannelRow
                while (cursor != null && cursor.moveToNext()) {
//                    Log.d("CHANNEL_TEST", "process cursor row - ${cursor.getType(0)}")
                    val channelProgram = ExternalRowItem.fromCursor(cursor)
                    if (channelProgram.channelId == channelId) {
                        if (!skipAddingDummyBannerItem) {
                            //Add posibility to open apps homescreen by inserting dummy item into row
                            val packageName = cursor.getString(38)
                            val launchUri =
                                context.packageManager.getLaunchIntentForPackage(packageName)
                                    ?.toUri(Intent.URI_INTENT_SCHEME)
                                    ?: context.packageManager.getLeanbackLaunchIntentForPackage(packageName)
                                        ?.toUri(Intent.URI_INTENT_SCHEME) ?: ""
                            externalRowItems.add(
                                ExternalRowItem(
                                    packageName = packageName,
                                    actionUri = launchUri,
                                    logoUri = sectionIconUri
                                )
                            )
                            skipAddingDummyBannerItem = true
                        }
                        externalRowItems.add(channelProgram)
                    } //TODO find some way to bypass TvProviders "selection not allowed" to load only specific channel data
                }
            } catch (t: Throwable) {
                if (throwable != null) throwable.addSuppressed(t) else throwable = t
            }

//            programs.forEach {
//                Log.d("CHANNEL_TEST", "program - $it")
//            }
        } catch (t: Throwable) {
            cursor?.close()
            if (throwable != null) throwable.addSuppressed(t) else throwable = t
        }
        throwable?.printStackTrace()
        return externalRowItems
    }

    fun onCleared() {
        loadNotifsCountJob?.cancel()
        loadNotifsCountJob = null
        loadNotifsUnreadJob?.cancel()
        loadNotifsUnreadJob = null
    }

    companion object {
        private const val DEFAULT_INNER_TEXT = "0"

        private const val HOME_CHANNELS_SELECTION = "browsable=1 AND type='TYPE_PREVIEW'"
        private const val PROGRAMS_BY_CHANNEL_ID_SELECTION = "channel_id=? AND browsable=1"

        const val NOTIFICATION_UPDATE_INTENT_ACTION = "com.twentyfouri.tvlauncher.NOTIFICATION"

        private val DELAY = 33.milliseconds // Because of CTS certification :(
    }

    internal inner class NotificationReceiver : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            updateNotificationLD()
        }
    }
}