package com.twentyfouri.tvlauncher.data

import android.content.Context
import android.text.format.DateUtils
import android.text.format.DateUtils.*
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import org.joda.time.DateTime
import org.joda.time.DateTimeZone
import java.util.*
import org.joda.time.format.DateTimeFormat

class DateTimeRepository(
    c: Context
) {

    private val context = c.applicationContext

    fun formatDateRange(start: DateTime?, end: DateTime?): String {
        start ?: return ""
        end ?: return ""
        return DateUtils
            .formatDateRange(
                context,
                Formatter(StringBuilder(50), Locale.getDefault()),
                start.millis,
                end.millis,
                FORMAT_SHOW_TIME or FORMAT_24HOUR,
                Flavor().getTimezoneID())
            .toString()
            .replace("\u2013", " \u2013 ") //put space around dash; 2013 is hexa code for dec code 8211
    }

    fun formatCatchupDate(date: DateTime?): String {
        if (date == null) return ""
        val dateString = DateUtils.formatDateTime(
            context,
            date.millis,
            FORMAT_SHOW_TIME
                    or FORMAT_24HOUR
                    or FORMAT_SHOW_WEEKDAY
                    or FORMAT_SHOW_DATE
                    or FORMAT_ABBREV_ALL
                    or FORMAT_NO_YEAR
        ).toString()

        //for some reason NL format do not have first letter capital but it is requested
        return dateString.substring(0, 1).toUpperCase() + dateString.substring(1)
    }

    fun formatDateTime(start: DateTime?, end: DateTime?): String {
        if (start == null || end == null) return ""
        return customFormatDateTime(
            start.millis
        )
    }

    fun isLive(start: DateTime?, end: DateTime?): Boolean {
        return end != null
                && end.millis / MILLIS_IN_MINUTE >= TimeProvider.now().millis / MILLIS_IN_MINUTE //end time accuracy to minutes, needed for live flag
                && start != null
                && start.millis < TimeProvider.now().millis
    }

    fun isToday(start: DateTime?): Boolean {
        val now = TimeProvider.now()
        return start != null
                && start.year() == now.year()
                && start.dayOfYear() == now.dayOfYear()
    }

    fun hasStartedInHistory(start: DateTime?): Boolean {
        return start != null && start.millis > TimeProvider.now().millis
    }

    fun formatClockTime(millis: Long): String {
        return formatDateTime(millis, FORMAT_SHOW_TIME or FORMAT_24HOUR)
    }

    fun formatRemainingTime(millis: Long): String {
        return "-" + formatTime(millis)
    }

    fun formatTime(millis: Long): String {
        val dateTime = DateTime(millis, DateTimeZone.UTC)
        var result = ""
        if (dateTime.hourOfDay().get() > 0) result += String.format("%d:", dateTime.hourOfDay().get())
        result += String.format("%02d:", dateTime.minuteOfHour().get())
        result += String.format("%02d", dateTime.secondOfMinute().get())
        return result
    }

    private fun customFormatDateTime(millis: Long): String {
        val someDate = DateTime(java.lang.Long.valueOf(millis), DateTimeZone.UTC)
        val date = DateTimeFormat.forPattern("dd-MM-yyyy")
        return date.print(someDate)
    }

    private fun formatDateTime(millis: Long, flags: Int): String {
        val formatter = Formatter(StringBuilder(50), Locale.getDefault())
        return DateUtils.formatDateRange(
            context,
            formatter,
            millis,
            millis,
            flags,
            Flavor().getTimezoneID()
        ).toString()
    }

    fun formatDuration(start: DateTime?, end: DateTime?): String {
        if (start == null || end == null) return ""
        val diff = end.millis - start.millis
        return context.getString(
            R.string.recordings_minutes,
            when {
                diff < 2 * MILLIS_IN_HOUR -> (diff / MILLIS_IN_MINUTE).toString()
                diff < 2 * MILLIS_IN_DAY -> (diff / MILLIS_IN_HOUR).toString()
                else -> (diff / MILLIS_IN_DAY).toString()
            }
        )
    }

    companion object {
        const val MILLIS_IN_MINUTE = 60 * 1000
        private const val MILLIS_IN_HOUR = 60 * MILLIS_IN_MINUTE
        private const val MILLIS_IN_DAY = 24 * MILLIS_IN_HOUR
    }
}