package com.twentyfouri.tvlauncher.data

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import org.joda.time.DateTime
import org.joda.time.DateTimeConstants.MILLIS_PER_DAY
import kotlin.collections.ArrayList
import kotlin.collections.HashMap

class EpgRepository(
    private val smartApi: FlowSmartApi,
    private val apiHandler: ApiHandler,
    resourceRepository: ResourceRepository
) {
    private val channelListLD = MutableLiveData<List<SmartMediaItem>>()

    private val loadingEventInfoString = resourceRepository.getString(R.string.player_loading_channel)
    val noDataSmartMediaItem = DummySmartMediaItem(resourceRepository.getString(R.string.player_no_data))

    suspend fun getAllChannelsFlow(): Flow<List<SmartMediaItem>> {
//        Log.d("EpgRepository","getAllChannels")
        return smartApi.getChannels(Flavor().getPageReference(PageType.EPG, Flavor().getLauncherMenu(smartApi)))

        //DEBUG option: emit only selected channels
//        return flow {
//            emit(smartApi.getChannels(Flavor().getPageReference(PageType.EPG, Flavor().getLauncherMenu(smartApi)))
//                .last()
//                .take(10))
//        }
    }

    private suspend fun getListOfListOfEpgEventsFlow(channels: List<SmartMediaItem>, fromTime: DateTime, toTime: DateTime): Flow<List<List<SmartMediaItem>>> =
        smartApi.getChannelPrograms(channels.map { it.reference }, fromTime, toTime)
                .map { epgData -> epgData.groupBy { it.channelReference } }
                .map { mapChannelIdToEpgData ->
                    channels.map { mapChannelIdToEpgData[it.reference] ?: emptyList() }
                }
                .catch {
                    emit(
                            // leave <SmartMediaItem>, even though AS show this as not necessary
                            // -> without it error will be thrown by build process
                            channels.map { ArrayList<SmartMediaItem>() })
                }

    fun loadListOfListOfEpgEventsForEpg(
            channels: List<SmartMediaItem>,
            fromTime: DateTime,
            toTime: DateTime,
            collector: suspend (List<List<SmartMediaItem>>) -> Unit
    ) {
//        Log.d("maxiEpg", "${print(channels, fromTime, toTime)} loadListOfListOfEpgEventsForEpg start")
        apiHandler.joinPreviousOrLaunchNew(
                synchronizedJobName = "channels: ${channels.hashCode()} size: ${channels.size} start: ${fromTime.millis} end: ${toTime.millis}",
                block = {
//                    Log.d("maxiEpg", "${print(channels, fromTime, toTime)} loadListOfListOfEpgEventsForEpg block start")
                    getListOfListOfEpgEventsFlow(channels, fromTime, toTime)
                            .conflate() //skips epg updates while the collector is still processing the previous update, last is always used
                            .collect { collector(it) }
//                            .last().also { collector(it) }
//                    Log.d("maxiEpg", "${print(channels, fromTime, toTime)} loadListOfListOfEpgEventsForEpg block end")
                }
        )
    }

    private fun print(channels: List<SmartMediaItem>, fromTime: DateTime, toTime: DateTime): String
    = "ch:${channels.joinToString(separator = ",") { it.channelNumber.toString() }}:s:${fromTime.hourOfDay}:${fromTime.minuteOfHour}:e:${toTime.hourOfDay}:${toTime.minuteOfHour}"

    fun loadChannelsForEpg(
            loadingDelay: Long,
            collector: (List<SmartMediaItem>) -> Unit
    ) {
        apiHandler.launchWithLoading(allowProgressHide = true, delay = loadingDelay) {
            //getAllChannelsFlow().collect { collector(it) }
            getAllChannelsFlow().last().also(collector)
        }
    }

    private fun MutableLiveData<List<SmartMediaItem>>.postValueIfChanged(channels: List<SmartMediaItem>) {
        if (value == null || value!!.size != channels.size) {
            postValue(channels)
            return
        }
        channels.forEachIndexed { i, channel ->
            if (channel.reference != value?.get(i)?.reference) {
                postValue(channels)
                return
            }
        }
    }

    fun getAllChannelsLD(): LiveData<List<SmartMediaItem>> {
//        Log.d("EpgRepository","getAllChannelsLD")
        apiHandler.joinPreviousOrLaunchNew(
                synchronizedJobName = "getAllChannelsLD",
                block = {
                    getAllChannelsFlow().last().also { channelListLD.postValueIfChanged(it) }
                    delay(10000) //prevent reloading of channels for 10 seconds
                }
        )
        return channelListLD
    }

    fun getNextEventLD(event: SmartMediaItem?, callerID: String): LiveData<SmartMediaItem?> {
        if(event == null) return MutableLiveData(null)
        val nextEvent = MutableLiveData<SmartMediaItem>(DummySmartMediaItem(loadingEventInfoString))
        val maxTime = TimeProvider.nowMs() + Flavor().getEpgConfigValues.EPG_DAYS_INTO_FUTURE * MILLIS_PER_DAY
        val eventEnd = event.endDate
        if(eventEnd == null || eventEnd.millis > maxTime) {
            //Log.d("miniEpg", "DO NOT LOAD NEXT")
            return MutableLiveData(null)
        }
        apiHandler.cancelPreviousAndLaunchNew(
                synchronizedJobName = "getNextEventLD_$callerID",
                block = {
                    channelFlow<SmartMediaItem?> {
                        smartApi.getMediaNext(event.reference).collect { nextItem ->
                            when (nextItem) {
                                event -> {
                                    //maybe inform about same event
                                    //nextEvent.postValue(DummySmartMediaItem("no more events?"))
                                    send(null)
                                }
                                null -> {
                                    send(DummySmartMediaItem(loadingEventInfoString))
                                }
                                else -> {
                                    launch {
                                        smartApi.getMediaDetail(nextItem.reference).collect { nextItemDetail ->
                                            //ignore dummy detail, without start/end date adjacent events may not be found and confuse GUI
                                            if(nextItemDetail.startDate != null && nextItemDetail.endDate != null) {
                                                send(nextItemDetail)
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }.collect { nextEvent.postValue(it) }
                }
        )
        return nextEvent
    }

    fun getPrevEventLD(event: SmartMediaItem?, callerID: String): LiveData<SmartMediaItem?> {
        if(event == null) return MutableLiveData(null)
        val prevEvent = MutableLiveData<SmartMediaItem>(DummySmartMediaItem(loadingEventInfoString))
        val minTime = TimeProvider.nowMs() - Flavor().getEpgConfigValues.EPG_DAYS_INTO_PAST * MILLIS_PER_DAY
        val eventStart = event.startDate
        if(eventStart == null ||  eventStart.millis < minTime) {
            //Log.d("miniEpg", "DO NOT LOAD PREV")
            return MutableLiveData(null)
        }
        apiHandler.cancelPreviousAndLaunchNew(
                synchronizedJobName = "getPrevEventLD_$callerID",
                block = {
                    channelFlow {
                        smartApi.getMediaPrevious(event.reference).collect { prevItem ->
                            when (prevItem) {
                                event -> {
                                    //maybe inform about same event
                                    //prevEvent.postValue(DummySmartMediaItem("no more events?"))
                                    send(null)
                                }
                                null -> {
                                    send(prevItem)
                                }
                                else -> {
                                    launch {
                                        smartApi.getMediaDetail(prevItem.reference).collect { prevItemDetail ->
                                            //ignore dummy detail, without start/end date adjacent events may not be found and confuse GUI
                                            if(prevItemDetail.startDate != null && prevItemDetail.endDate != null) {
                                                if (event.reference == prevItem.reference) {
                                                    //This can happen in case of data inconsistency (event start/end changed)
                                                    send(null)
                                                } else {
                                                    send(prevItemDetail)
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }.collect { prevEvent.postValue(it) }
                }
        )
        return prevEvent
    }

    fun getDetailLD(item: SmartMediaItem?, callerID: String): LiveData<SmartMediaDetail?> {
        val detail = MutableLiveData<SmartMediaDetail>()
        item ?: return detail
        apiHandler.cancelPreviousAndLaunchNew(
                synchronizedJobName = "getDetailLD_$callerID",
                block = {
                    smartApi.getMediaDetail(item.reference)
//                            .collect { detailItem -> detail.postValue(detailItem) }
                            .last().also { detailItem -> detail.postValue(detailItem) }
                }
        )
        return detail
    }

    private val channelNumberToEventAndDetailLDs: HashMap<Int, Pair<MutableLiveData<SmartMediaItem?>, MutableLiveData<SmartMediaDetail?>>> = HashMap()

    fun getNowEventAndDetailFromChannelLDs(channel: SmartMediaItem?): Pair<LiveData<SmartMediaItem?>, LiveData<SmartMediaDetail?>> {
        val channelNumber = channel?.channelNumber ?: 0
        val nowEventAndDetail = channelNumberToEventAndDetailLDs.getOrPut(channelNumber) {
            Pair(MutableLiveData<SmartMediaItem?>(DummySmartMediaItem(loadingEventInfoString)), MutableLiveData<SmartMediaDetail?>())
        }
        channel ?: return nowEventAndDetail
        val jobPrefix = "getNowFromChannel"
        apiHandler.cancelFilteredJobsAndJoinPreviousOrLaunchNew(
                catchBlock = {
                    channelNumberToEventAndDetailLDs.remove(channelNumber)
                    false
                },
                shouldCancelJobBlock = {
                    it.startsWith(jobPrefix) && it.endsWith(channelNumber.toString()).not()
                },
                nameOfJobToBeJoined = "${jobPrefix}_$channelNumber",
                block = {
                    if (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(channel.reference.toString()) == true) {
                        nowEventAndDetail.first.postValue(DummySmartMediaItem(channel.title ?: ""))
                        return@cancelFilteredJobsAndJoinPreviousOrLaunchNew
                    }
                    val nowEvent = smartApi.getChannelPrograms(listOf(channel.reference), TimeProvider.now(), TimeProvider.now()).last().lastOrNull()
                    nowEvent ?: run {
                        nowEventAndDetail.first.postValue(noDataSmartMediaItem)
                        nowEventAndDetail.second.postValue(null)
                        return@cancelFilteredJobsAndJoinPreviousOrLaunchNew
                    }
                    nowEventAndDetail.first.postValue(nowEvent)
                    val nowDetail = smartApi.getMediaDetail(nowEvent.reference).last()
                    nowEventAndDetail.second.postValue(nowDetail)
                    channelNumberToEventAndDetailLDs.remove(channelNumber)
                }
        )
        return nowEventAndDetail
    }

    companion object {
        const val PLAYER_CONTROLS_EPG_WINDOW = 2
        const val PLAYER_CONTROLS_EPG_WINDOW_EXTENDED = 8
    }
}

class DummySmartMediaItem(title: String = ""): SmartMediaItem(DummySmartMediaReference()) {
    init {
        this.title = title
    }
}

class DummySmartMediaReference: SmartMediaReference() {
    override fun equals(other: Any?): Boolean = true
    override fun hashCode(): Int = -1
}