package com.twentyfouri.tvlauncher.data

import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.epg.EpgDao
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.media.*
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.smartmodel.model.payment.SmartSubscriptionReference
import com.twentyfouri.smartmodel.model.watchlist.SmartContinueWatchingItem
import com.twentyfouri.smartmodel.serialization.SmartDataPrimitive
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.withContext
import timber.log.Timber

class MetadataRepository(
    private val smartApi: FlowSmartApi?,
    private val apiHandler: ApiHandler,
    private val resourceRepository: ResourceRepository,
    private val youboraAnalytics: YouboraAnalytics,
    private val epgDao: EpgDao
) {
    fun getMediaDetailLD(
        reference: SmartMediaReference?,
        channelReference: SmartMediaReference?,
        shouldLoadFullDetail: Boolean
    ): LiveData<SmartMediaDetail> {
        val mediaDetailLD = MutableLiveData<SmartMediaDetail>()
        if (reference != null)
            updateMediaDetailLD(reference, channelReference, mediaDetailLD, shouldLoadFullDetail)
        return mediaDetailLD
    }

    private fun updateMediaDetailLD(
        reference: SmartMediaReference,
        channelReference: SmartMediaReference?,
        mediaDetailLD: MutableLiveData<SmartMediaDetail>,
        shouldLoadFullDetail: Boolean
    ) {
        Timber.tag("FlowApi").d("updateMediaDetailLD channelRef: $channelReference")
        val fallback = Flavor().convertMediaReferenceToDetail(reference)
        channelReference?.also { fallback.channelReference = it }
        if (smartApi == null) {
            mediaDetailLD.postValue(fallback)
            return
        }
        if ((fallback.extras["no-data"] as? SmartDataPrimitive)?.getBoolean(false) == true
            || fallback.type == SmartMediaType.UNKNOWN
            || reference.hashCode() == 0
        ) {
            mediaDetailLD.postValue(
                SmartMediaDetail(reference, fallback.type).apply {
                    title = resourceRepository.getString(R.string.epg_catchup_unavailable)
                    this.channelReference = fallback.channelReference
                }
            )
            return
        }

        if (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(reference.toString()) == true) {
            apiHandler.joinPreviousOrLaunchNew(
                synchronizedJobName = "mediaReference: $reference detail: ${mediaDetailLD.hashCode()}",
                block = {
                    smartApi.getChannels(Flavor().getPageReference(PageType.EPG, Flavor().getLauncherMenu(smartApi)))
                            .mapNotNull { channels -> channels.find { it.reference == reference } }
                            //.collect { mediaDetailLD.postValue(Flavor().getAppChannelsDelegate()?.getAppChannelMediaDetail(it, resourceRepository)) }
                            .last().also { mediaDetailLD.postValue(Flavor().getAppChannelsDelegate()?.getAppChannelMediaDetail(it, resourceRepository)) }
                },
                catchBlock = {
                    mediaDetailLD.postValue(fallback)
                    false
                }
            )
        } else {
            apiHandler.joinPreviousOrLaunchNew(
                synchronizedJobName = "mediaReference: $reference detail: ${mediaDetailLD.hashCode()}",
                block = {
                    mediaDetailLD.postValue(
                        with (smartApi.getMediaDetail(reference)) {
                            if (shouldLoadFullDetail) {
                                last()
                            } else {
                                first { it.description?.isNotBlank() == true && it.description != it.title }
                            }
                        }
                    )
                },
                catchBlock = {
                    mediaDetailLD.postValue(fallback)
                    if (fallback.type == SmartMediaType.LIVE_EVENT)
                        resetLiveChannelAndPostLiveEvent(fallback, mediaDetailLD)
                    false
                }
            )
        }
    }

    private suspend fun resetLiveChannelAndPostLiveEvent(
        fallback: SmartMediaDetail,
        mediaDetailLD: MutableLiveData<SmartMediaDetail>
    ) {
        fallback.channelReference ?: run {
            withContext(Dispatchers.IO) { epgDao.deleteAllEpgEvents() }
            return
        }
        smartApi ?: return
        try {
            withContext(Dispatchers.IO) { epgDao.deleteEpgEventsOnChannel(fallback.channelReference!!.roomId()) }
            TimeProvider.now()
                .let { smartApi.getChannelPrograms(listOf(fallback.channelReference!!), it, it).last() }
                .firstOrNull()
                ?.let { smartApi.getMediaDetail(it.reference).last() }
                ?.also { mediaDetailLD.postValue(it) }
        } catch (e: Exception) {}
    }

    fun getMediaStreamLD(
        reference: SmartMediaReference,
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean
    ): LiveData<SmartMediaStream> {
        val mediaStreamLD = MutableLiveData<SmartMediaStream>()
        if (smartApi == null) return mediaStreamLD
        apiHandler.launchNew(
            block = {
//                Log.d("FlowApi", "getMediaStreamLD by mediaReference")
//                Log.d("MediaDetail", "getMediaDetail launcher from MetadataRepository getMediaStreamLD")
                smartApi.getMediaDetail(reference)
                        .last().also { detail ->
                            val stream = detail.editions.firstOrNull()?.streams?.firstOrNull()
                                    ?.let {
                                        //(it as SmartVideoStreamReference).url = ""
                                        smartApi.getMediaStream(it, SmartMediaStreamOptions(SmartMediaStreamOptions.Builder())).last() }
                            if (stream == null) {
                                youboraAnalytics.plugin.fireError(resourceRepository.getString(R.string.missing_stream), resourceRepository.getString(R.string.missing_stream_code), "")
                                throw IllegalStateException(resourceRepository.getString(R.string.missing_stream))
                            }
                            mediaStreamLD.postValue(stream!!)
                        }
//                        .flatMapLatest { detail ->
//                            val stream = detail.editions.firstOrNull()?.streams?.firstOrNull()
//                                    ?.let { smartApi.getMediaStream(it, SmartMediaStreamOptions(SmartMediaStreamOptions.Builder())) }
//                            if (stream == null) {
//                                youboraAnalytics.plugin.fireError(resourceRepository.getString(R.string.missing_stream), resourceRepository.getString(R.string.missing_stream_code), "")
//                                throw IllegalStateException(resourceRepository.getString(R.string.missing_stream))
//                            }
//                            stream
//                        }
//                        .first()
//                        .also { mediaStreamLD.postValue(it) }
            },
            catchBlock = catchBlock
        )
        return mediaStreamLD
    }

    fun getMediaStreamLD(
        streamReference: SmartMediaStreamReference,
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean
    ): LiveData<SmartMediaStream> {
        val mediaStreamLD = MutableLiveData<SmartMediaStream>()
        if (smartApi == null)
            return mediaStreamLD
        apiHandler.launchNew(
            block = {
//                Log.d("FlowApi", "getMediaStreamLD by streamReference")
                smartApi.getMediaStream(streamReference, SmartMediaStreamOptions(SmartMediaStreamOptions.Builder()))
//                        .first()
                        .last()
                        .also { stream ->
                            stream.primaryUrl.isBlank()
                                    .ifTrue {
                                        youboraAnalytics.plugin.fireError(resourceRepository.getString(R.string.missing_stream), resourceRepository.getString(R.string.missing_stream_code), "")
                                        throw IllegalStateException(resourceRepository.getString(R.string.missing_stream)) }
                                    .ifFalse { mediaStreamLD.postValue(stream) }
                        }
            },
            catchBlock = catchBlock
        )
        return mediaStreamLD
    }

    fun getSubscriptionsListLD(reference: SmartMediaReference?): LiveData<List<SmartSubscription>> {
        val subscriptionsList = MutableLiveData<List<SmartSubscription>>()
        if (smartApi == null || reference == null)
            return subscriptionsList
        apiHandler.launchNew {
            val list = Flavor().getSubscriptions(smartApi, reference)
            subscriptionsList.postValue(list.filter {
                it.canBePurchasedInApp
            })
        }
        return subscriptionsList
    }

    fun subscribeProduct(productReference: SmartSubscriptionReference?, isAutoRenew: Boolean?, subscriptionProduct: SmartSubscription?): MutableLiveData<SmartPurchase> {
        val purchaseResult = MutableLiveData<SmartPurchase>()
        if (smartApi == null)
            return purchaseResult
        if (productReference != null && isAutoRenew != null) {
            //todo: handle PurchaseResult
            apiHandler.launchNew {
                purchaseResult.postValue(Flavor().subscribeProduct(smartApi, productReference, isAutoRenew, subscriptionProduct))
            }
        }
        return purchaseResult
    }

    fun getEpgImagePlaceholderId(): Int {
        return R.drawable.metadata_placeholder
    }

    fun getSubscriptionChannelsList(subscription: SmartSubscription?): LiveData<List<SmartMediaItem>> {
        val subscriptionChannels = MutableLiveData<List<SmartMediaItem>>()
        if (subscription == null) return subscriptionChannels
        if (smartApi == null) return subscriptionChannels

        apiHandler.launchNew {
            //TODO getChannels should be called only from EpgRepository
            smartApi.getChannels(Flavor().getPageReference(PageType.EPG, Flavor().getLauncherMenu(smartApi)))
                    .map { channelItems ->
                        channelItems.filter { channelItem ->
                            channelItem.extras["subscription"].getArray()
                                    .find { it.getString() == subscription.name } != null
                        }
                    }
//                    .collect {
                    .last().also {
                        filteredChannelsList ->
                        subscriptionChannels.postValue(
                                if (filteredChannelsList.isNotEmpty()) {
                                    //subscription channels found locally
                                    filteredChannelsList
                                } else {
                                    //fallback to find subscription channels via network call
                                    Flavor().getSubscriptionChannels(smartApi, subscription)
                                }
                        )
                    }
        }
        return subscriptionChannels
    }

    fun sendPlayerEvent(event: SmartPlayerEvent, bookmark: SmartContinueWatchingItem?) {
        if (smartApi == null) return
        apiHandler.launchNew {
            smartApi.postPlayerEvent(event, bookmark).last()
        }
    }
}