package com.twentyfouri.tvlauncher.data

import com.twentyfouri.androidcore.utils.Log
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistOptions
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistSelectedOptions
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.SearchGenre
import kotlinx.coroutines.flow.first
import timber.log.Timber
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutionException

class RecommendationsRepository(
    private val smartApi: FlowSmartApi,
    private val apiHandler: ApiHandler
) {

    private suspend fun getSearchResult(query: String, genre: SearchGenre): List<SmartMediaItem> {
        val playlistOptions = SmartPlaylistOptions().apply {
            Flavor().getSearchSorting(genre)?.let { sortings.add(it) }
            pagings.setupFixedMode(50)
        }

        val options = SmartPlaylistSelectedOptions(playlistOptions).apply {
            Flavor().getSearchFilter(genre)?.let { f->
                setFilters(listOf(f))
            }
        }

        return try {
            smartApi.getSearchResults(query, options).first().items
        } catch (e: Exception) {
            Timber.e(e,"Failed to perform search for $query on $genre: ${e.message}")
            emptyList()
        }
    }

    fun searchByGenre(query: String, searchGenre: SearchGenre): List<SmartMediaItem> {
        val future = CompletableFuture<List<SmartMediaItem>>()
        apiHandler.launchNew {
            try {
                val results = getSearchResult(query, searchGenre)
                    .filter { !Flavor().isChannelAdult(it) }

                future.complete(results)
            } catch (e: Exception) {
                future.completeExceptionally(e)
            }
        }

        return try {
            future.get()
        } catch (e: ExecutionException) {
            Timber.e(e, "Failed to perform search")
            emptyList()
        }
    }
}