package com.twentyfouri.tvlauncher.data

import android.content.Context
import android.content.pm.PackageManager
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.error.SmartApiException
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStatus
import com.twentyfouri.smartmodel.model.user.SmartSessionState
import com.twentyfouri.smartmodel.serialization.SmartDataObject
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.common.data.SmartApiRepository
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import timber.log.Timber
import com.twentyfouri.tvlauncher.common.extensions.postAddAll

class RowPageRepository(
    private val smartApi: FlowSmartApi?,
    private val appListRepository: AppListRepository,
    private val contentResolverRepository: ContentResolverRepository,
    private val apiHandler: ApiHandler,
    private val context: Context
) {

    fun getPageSectionsLD(reference: SmartPageReference): LiveData<List<SmartPageSection>> {
        val sections = MutableLiveData<List<SmartPageSection>>()
        apiHandler.launchWithLoading {
            val r = getSmartPage(reference).sections
            // uncomment to artificially multiply rows - to test high number of rows
//            val r = getSmartPage(reference).sections.toMutableList()
//            r.addAll(r)
//            r.addAll(r)

             sections.postValue(r)
        }
        return sections
    }

    private suspend fun getSmartPage(reference: SmartPageReference): SmartPage {
        val page = HardcodedPageMapper.convertPage(
            Flavor().getLauncherPage(smartApi, reference) ?: SmartPage(reference),
            smartApi == null,
            NetworkConnectionState.instance.state != NetworkConnectionState.State.OFFLINE,
            SmartApiRepository.isServerBusy
        )

        //add rows from 3rd party apps
        if (!Flavor().isAppsPage(page.reference) && canAddRowsFrom3rdPartyApps(reference)) {
            // MAIN detected -> insert
            val appChannels = contentResolverRepository.loadChannels()
            val sections = page.sections.toMutableList()
            appChannels.forEach { externalRow ->
                if (Flavor().isAppAllowedToAddRow(externalRow, context) ) {
                    val packageManager = context.packageManager
                    val channelName = packageManager.getApplicationLabel(packageManager.getApplicationInfo(externalRow.packageName ?: "", PackageManager.GET_META_DATA))
                    val section = SmartPageSection(
                        id = "${externalRow.id}-${externalRow.packageName}",
                        label = "${channelName ?: ""}: ${externalRow.displayName ?: ""}",
                        refreshTimeMs = EXTERNAL_CHANNEL_REFRESH_INTERVAL,
                        playlistReference = ExternalRowPlaylistReference(externalRow)
                    ).apply { extras = SmartDataObject("channel_icon_uri" to externalRow.channelLogoUri.toString()) }
                    sections.add(Flavor().externalRowsHardcodedPosition(sections), section)
                }
            }
            page.sections = sections
        }
        return page
    }

    private suspend fun canAddRowsFrom3rdPartyApps(reference: SmartPageReference): Boolean {
        if (Flavor().allowDisplayOfExternalRows().not()) return false
        if (smartApi?.sessionState != SmartSessionState.LOGGED_IN) return false
        return try {
            reference == Flavor().getPageReference(PageType.MAIN, Flavor().getLauncherMenu(smartApi))
        } catch (e: SmartApiException) {
            false
        }
    }

    // PAGING METHODS SECTION
    private suspend fun getSectionContent(section: SmartPageSection): List<Any> {
        smartApi ?: return emptyList()

        if (section.nothingWasLoadedYet()) { //nothing was loaded yet
            return getSectionContentNextPage(section) //-> load first page
        }

        //some pages were already loaded -> refresh all
        val sectionToRefreshAllPages = section.cloneToZeroOffset()
        val refreshedItems = mutableListOf<Any>()
        Timber.tag("Flow")
            .d("old section ${section.selectedOptions?.pagingOffset} ${section.selectedOptions?.pagingCount} " +
                    "${section.selectedOptions?.totalCount} new section ${sectionToRefreshAllPages.selectedOptions?.pagingOffset} " +
                    "${sectionToRefreshAllPages.selectedOptions?.pagingCount} ${sectionToRefreshAllPages.selectedOptions?.totalCount}")
        while (sectionToRefreshAllPages.selectedOptions!!.pagingOffset < section.selectedOptions!!.pagingOffset
                && sectionToRefreshAllPages.selectedOptions!!.pagingOffset != section.selectedOptions!!.totalCount
        ) {
            val itemsOnPage = getSectionContentNextPage(sectionToRefreshAllPages)
            if (itemsOnPage.isEmpty()) return refreshedItems
            refreshedItems.addAll(itemsOnPage)
        }
        return refreshedItems
    }

    private fun SmartPageSection.nothingWasLoadedYet() = (selectedOptions?.pagingOffset ?: 0) == 0

    private fun SmartPageSection.cloneToZeroOffset() = SmartPageSection(id, label, playlistReference, refreshTimeMs).also {
        it.sectionStyle = sectionStyle
        it.aspectRatio = aspectRatio
        it.selectedOptions = selectedOptions?.clone()?.setPaging(
            0,
            selectedOptions?.pagingCount ?: 0,
            selectedOptions?.totalCount ?: 0
        )
        it.extras = extras
    }

    private suspend fun getSectionContentNextPage(section: SmartPageSection): List<Any> {
        smartApi ?: return emptyList()
        val playlistOptions =
            section.selectedOptions ?: getFirstPageSelectedOptions(smartApi, section)
        Timber.tag("Flow")
            .d("getSectionContentNextPage offset: ${playlistOptions.pagingOffset} pageSize: ${playlistOptions.pagingCount} " +
                    "total: ${playlistOptions.totalCount}")
        lateinit var playlist: SmartPlaylist
        //TODO use intermediate results from the Flow
        //smartApi.getPlaylistItems(section.playlistReference, playlistOptions).collect { playlist = it }
        smartApi.getPlaylistItems(section.playlistReference, playlistOptions).last().also {
            playlist = it
        }
        section.selectedOptions!!.setPaging(playlist.totalItemsMin,
            section.selectedOptions!!.pagingCount,
            playlist.totalItemsMax)
        return filterResults(section, playlist.items)
    }

    private suspend fun getFirstPageSelectedOptions(_smartApi: FlowSmartApi, section: SmartPageSection)
            = _smartApi.getPlaylistOptions(section.playlistReference).last()
        .createSelection()
        .setPaging(0, Flavor().getPageSize(section))
        .apply {
            when (val type = Flavor().getPlaylistType(section)) {
                PlaylistType.ON_NOW, PlaylistType.DO_NOT_MISS_CHANNELS, PlaylistType.LAST_WATCHED_CHANNELS ->
                    Flavor().getGenreFilterReference("", type).also { filter -> this.setFilters(listOf(filter)) }
                PlaylistType.ALL_CHANNELS_PAGED, PlaylistType.ALL_CHANNELS ->
                    Flavor().getChannelSorting()?.also { sorting -> this.setSorting(sorting.reference, true) }
                else -> {
                }
            }
        }
        .also { section.selectedOptions = it }

    private fun filterResults(section: SmartPageSection, items: List<SmartMediaItem>): List<SmartMediaItem> {
        // Filter out app channels from channel
        return when (Flavor().getPlaylistType(section)) {
            PlaylistType.RECENT_RECORDINGS -> {
                // If it's the recent recordings row, only show successful recordings
                items.filter {
                    it.recordingData?.recordingStatus == SmartRecordingStatus.SUCCESS
                }
            }
            PlaylistType.ALL_CHANNELS,
            PlaylistType.ALL_CHANNELS_PAGED,
            PlaylistType.ON_NOW,
            PlaylistType.DO_NOT_MISS_CHANNELS,
            PlaylistType.FAVORITE_CHANNELS -> {
                // Filter out app channels from channel rows
                val appChannelIds = Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.toSet()
                if (appChannelIds != null) {
                    items.filter {
                        !appChannelIds.contains(Flavor().getChannelId(it))
                    }
                } else items
            }
            else -> items
        }
    }

    fun updateSectionContentNextLD(
        section: SmartPageSection,
        rowData: MutableLiveData<List<Any>>,
        rowUpdated: MutableLiveData<Boolean>
    ) {
        apiHandler.launchNew {
            rowData.postAddAll(getSectionContentNextPage(section))
            rowUpdated.postValue(true)
        }
    }

    private suspend fun getRecommendedAppsContent(): List<String>? {
        smartApi?.let { api ->
            try {
                //TODO use intermediate results of the Flow
                lateinit var playlist: SmartPlaylist
                api.getPlaylistItems(
                        Flavor().getPlaylistReference(PlaylistType.RECOMMENDED_APPS),
                        SmartPlaylistOptions()
                                .apply { pagings.setupFixedMode(50) }
                                .createSelection()
                                .apply { Flavor().getGenreFilterReference(RecommendedAppsFilter).let { filter -> this.setFilters(listOf(filter)) } }
                //).collect { playlist = it }
                ).last().also { playlist = it }
                return playlist.items.map { it.description ?: "" }

            } catch (exception: Exception) {
                // We keep the result recommended apps list as empty.
                Timber.tag("RECOMMENDED_APPS")
                    .d("RowPageRepository -> Getting recommended apps content. Exception: $exception.")
            }
        }
        return emptyList()
    }

    private suspend fun getAppChannelContent(section: ExternalRowPlaylistReference, sectionIconUri: String): List<Any>
            = contentResolverRepository.loadChannelPrograms(section.externalRow.id, sectionIconUri)

    fun updateSectionContentLD(
        section: SmartPageSection,
        sectionDataUpdater: Observer<List<Any>>
    ) {
        apiHandler.joinPreviousOrLaunchNew(
            synchronizedJobName = "section: ${section.hashCode()} row: ${sectionDataUpdater.hashCode()}",
            block = {
                when (Flavor().getPlaylistType(section)) {
                    PlaylistType.EXTERNAL_CHANNEL -> sectionDataUpdater.onChanged(
                        getAppChannelContent(
                            section = section.playlistReference as ExternalRowPlaylistReference,
                            sectionIconUri = section.extras["channel_icon_uri"].getString("")).toMutableList())
                    PlaylistType.RECOMMENDED_APPS, PlaylistType.FAVORITE_APPS ->
                        appListRepository.registerSectionContentLDForUpdates(
                            section = section,
                            sectionDataUpdater = sectionDataUpdater,
                            serverListOfApps = getRecommendedAppsContent())
                    PlaylistType.PLAY_STORE,
                    PlaylistType.ALL_APPS,
                    PlaylistType.ALL_GAMES -> appListRepository.registerSectionContentLDForUpdates(
                        section = section,
                        sectionDataUpdater = sectionDataUpdater)
                    PlaylistType.UPCOMING_PROGRAMS -> sectionDataUpdater.onChanged(
                            Flavor().cleanupUpcomingPrograms(getSectionContent(section))
                        )
                    PlaylistType.CATCHUP_PROGRAMS,
                    PlaylistType.MOVIES,
                    PlaylistType.ALL_CHANNELS,
                    PlaylistType.FAVORITE_CHANNELS,
                    PlaylistType.RECENT_RECORDINGS,
                    PlaylistType.ALL_CHANNELS_PAGED,
                    PlaylistType.DO_NOT_MISS_CHANNELS,
                    PlaylistType.LAST_WATCHED_CHANNELS,
                    PlaylistType.ON_NOW,
                    PlaylistType.MOST_WATCHED,
                    PlaylistType.LAST_WATCHED,
                    PlaylistType.GRID,
                    PlaylistType.UNKNOWN -> sectionDataUpdater.onChanged(getSectionContent(section))
                    else -> {
                        //throw IllegalArgumentException("Unsupported PlaylistType.Type")
                        sectionDataUpdater.onChanged(emptyList())
                    }
                }
            }
        )
    }

    fun updateSectionContentLDWithLoading(
        section: SmartPageSection,
        sectionDataUpdater: Observer<List<Any>>
    ) {
        apiHandler.launchWithLoading(
            allowProgressHide = true,
            delay = LOADING_DELAY,
            block = {
                sectionDataUpdater.onChanged(getSectionContent(section)) }
        )
    }

    fun clearSection(section: SmartPageSection, sectionDataUpdater: Observer<List<Any>>) { appListRepository.clearSection(section, sectionDataUpdater) }

    fun updateFavoriteAppsPriority(currentList: List<Any>) {
        appListRepository.updateFavoriteAppsPriority(currentList)
    }

    fun getPlaylistOptionsLD(section: SmartPageSection): LiveData<SmartPlaylistOptions> {
        val optionsLD = MutableLiveData<SmartPlaylistOptions>()
        apiHandler.joinPreviousOrLaunchNew(
            synchronizedJobName = "section options: ${section.hashCode()}",
            block = {
                smartApi?.also { api ->
                    section.playlistReference
                        .let { api.getPlaylistOptions(it) }
                        .last().also { optionsLD.postValue(it) }
                }
            }
        )
        return optionsLD
    }

    companion object {
        private const val RecommendedAppsFilter = "Aanbevolen apps"
        private const val EXTERNAL_CHANNEL_REFRESH_INTERVAL = 5 * 60 * 1000
        private const val LOADING_DELAY = 500L
    }
}