package com.twentyfouri.tvlauncher.extensions

import com.twentyfouri.androidcore.epg.model.*
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.data.EpgChannelExt
import com.twentyfouri.tvlauncher.data.EpgEventExt
import com.twentyfouri.tvlauncher.utils.MutexProvider
import com.twentyfouri.tvlauncher.utils.StringsMapper
import com.twentyfouri.tvlauncher.viewmodels.EventComparator
import kotlinx.coroutines.sync.withLock
import org.joda.time.DateTime
import java.util.*
import kotlin.collections.ArrayList
import kotlin.collections.Iterator
import kotlin.collections.List
import kotlin.collections.indices
import kotlin.collections.isNotEmpty
import kotlin.collections.set
import kotlin.collections.sortedWith
import kotlin.collections.toList

const val NO_DATA_ID = "no-data"
private const val NO_DATA_ENABLED = true
private const val NO_DATA_MINIMAL_TIME_FRAME = 5 * 60 * 1000
//private const val EPG_EVENT_EXPIRATION_MS = 1000 * 60 * 60 * 4 //4 hours

internal suspend fun EpgData.addDataSuperSafely(
        channels: List<SmartMediaItem>,
        events: List<List<SmartMediaItem>>,
        fromTime: Long,
        toTime: Long,
        getRecordingDrawableBlock: (event: SmartMediaItem) -> EpgEventLabel?,
        epgAbsoluteStartTime: DateTime,
        epgAbsoluteEndTime: DateTime
): EpgData {
    for (newIndex in channels.indices) {
        val newChannel = channels[newIndex]
        var existingChannelIndex = -1
        findChannelIndex@for (index in 0..channelCount) {
            if (getChannel(index).id == Flavor().getChannelId(newChannel)) {
                existingChannelIndex = index
                break@findChannelIndex
            }
        }
        if (existingChannelIndex < 0) continue

        //this makes sure that each channel is locked when the block executes, thus no concurrency exception should happen
        MutexProvider.getOrCreateMutexForChannelNumber(existingChannelIndex).withLock {
//            Log.d("maxiEpg", "addDataSuperSafely $existingChannelIndex start: ${DateTime.now().millis}")
            val newEvents = events[newIndex].sortedWith(EventComparator()).toMutableList()
            val oldChannel = getChannel(existingChannelIndex) as EpgChannelExt

            // If channel is AppChannel add app channel event
            if (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(oldChannel.smartMediaItem.reference.toString()) == true) {
                newEvents.add(
                    SmartMediaItem(
                        oldChannel.smartMediaItem.reference,
                        type = SmartMediaType.LIVE_EVENT
                    ).apply {
                       startDate = epgAbsoluteStartTime
                       endDate = epgAbsoluteEndTime
                       title = oldChannel.smartMediaItem.title
                    }
                )
            }
            val oldEvents = getEvents(existingChannelIndex)
            val eventsToAdd = ArrayList<EpgEventExt>()
            val eventsToRemove = ArrayList<EpgEventExt>()
            findEventsToAddAndEventsToRemove(
                newEvents = newEvents,
                oldEvents = oldEvents.toList() /*clone*/,
                eventsToAdd = eventsToAdd,
                eventsToRemove = eventsToRemove,
                getRecordingDrawableBlock = getRecordingDrawableBlock,
                epgAbsoluteStartTime = epgAbsoluteStartTime,
                epgAbsoluteEndTime = epgAbsoluteEndTime,
                channelNumber = newChannel.channelNumber
            )
//        Log.d("maxiEpg", "addDataSuperSafely ch: ${newChannel.channelNumber} eventsToRem: ${eventsToRemove.size}")
//        eventsToRemove.forEach {
//            Log.d("maxiEpg", "addDataSuperSafely ch: ${newChannel.channelNumber} eventToRem: ${EpgChannelExt.print1(it.startTime, it.endTime)}")
//        }
//        Log.d("maxiEpg", "addDataSuperSafely ch: ${newChannel.channelNumber} eventsToAdd: ${eventsToAdd.size}")
//        eventsToAdd.forEach {
//            Log.d("maxiEpg", "addDataSuperSafely ch: ${newChannel.channelNumber} eventToAdd: ${EpgChannelExt.print1(it.startTime, it.endTime)}")
//        }
            if (eventsToRemove.isNotEmpty()) {
                oldEvents.removeAll(eventsToRemove)
            }
            if (eventsToAdd.isNotEmpty()) {
                val map = HashMap<EpgChannel, List<EpgEvent>>()
                map[oldChannel] = eventsToAdd
                addData(map)
            }
            oldChannel.markLoadedRange(fromTime, toTime)
//            Log.d("maxiEpg", "addDataSuperSafely $existingChannelIndex end: ${DateTime.now().millis}")
        }
    }
    return this
}

/**
 * Use this for iterating to prevent [ConcurrentModificationException]!
 */
internal fun EpgDataType.getEventsClone(channelPosition: Int) = getEvents(channelPosition)?.toList()

private fun findEventsToAddAndEventsToRemove(
        newEvents: List<SmartMediaItem>,
        oldEvents: List<EpgEvent>,
        eventsToAdd: ArrayList<EpgEventExt>,
        eventsToRemove: ArrayList<EpgEventExt>,
        getRecordingDrawableBlock: (event: SmartMediaItem) -> EpgEventLabel?,
        epgAbsoluteStartTime: DateTime,
        epgAbsoluteEndTime: DateTime,
        channelNumber: Int
) {
    val newEventsIterator = newEvents.iterator()
    val oldEventsIterator = oldEvents.iterator()
    var newEvent: SmartMediaItem? = null
    var oldEvent: EpgEventExt? = null
    var lastEndTimeMs = epgAbsoluteStartTime.millis
    //the infinite loop is intentional, the inner code is responsible for breaking it
    while(true) {
        oldEvent = oldEvent ?: oldEventsIterator.nextEpgEventExtOrNull()
        newEvent = newEvent ?: newEventsIterator.nextOrNull()
        when {
            oldEvent == null && newEvent == null -> {
                //there are no more events to analyze -> return
                addNoDataIfApplicable(lastEndTimeMs, epgAbsoluteEndTime.millis, channelNumber, eventsToAdd)
                return
            }
            oldEvent == null -> {
                //there are no events from previous set left to analyze.
                //this happens if there are no data yet or if the user scrolls to the FUTURE and there are no data yet
                addNewDataOrNoData(newEvent!!, channelNumber, eventsToAdd, getRecordingDrawableBlock, lastEndTimeMs) { lastEndTimeMs = it }
                newEvent = null
            }
            newEvent == null -> {
                if (oldEvent.startTime < lastEndTimeMs) {
                    removeOldData(channelNumber, oldEvent, eventsToRemove)
                } else {
                    addNoDataIfApplicable(lastEndTimeMs, oldEvent.startTime, channelNumber, eventsToAdd)
                    lastEndTimeMs = oldEvent.endTime
                }
                oldEvent = null
            }
            newEvent.startDate == null || newEvent.endDate == null -> {
                //do not add anything we just need to skip erroneous data
                newEvent = null
            }
            oldEvent.smartMediaItem.reference == newEvent.reference
                    && oldEvent.startTime == newEvent.startDate!!.millis
                    && oldEvent.endTime == newEvent.endDate!!.millis-> {
                //skip over identical items
                //TODO extend the identity check
                lastEndTimeMs = newEvent.endDate!!.millis
                oldEvent = null
                newEvent = null
            }
            oldEvent.endTime <= newEvent.startDate!!.millis -> {
                if (oldEvent.startTime < lastEndTimeMs) {
                    removeOldData(channelNumber, oldEvent, eventsToRemove)
                } else {
                    addNoDataIfApplicable(lastEndTimeMs, oldEvent.startTime, channelNumber, eventsToAdd)
                    lastEndTimeMs = oldEvent.endTime
                }
                oldEvent = null
            }
            oldEvent.startTime < newEvent.endDate!!.millis -> {
                //now we know that they are overlapping
                removeOldData(channelNumber, oldEvent, eventsToRemove)
                addNewDataOrNoData(newEvent, channelNumber, eventsToAdd, getRecordingDrawableBlock, lastEndTimeMs) { lastEndTimeMs = it }
                oldEvent = null
                newEvent = null
            }
            else -> {
                addNewDataOrNoData(newEvent, channelNumber, eventsToAdd, getRecordingDrawableBlock, lastEndTimeMs) { lastEndTimeMs = it }
                newEvent = null
            }
        }
    }
}

private fun removeOldData(
    channelNumber: Int,
    oldEvent: EpgEventExt,
    eventsToRemove: ArrayList<EpgEventExt>
) {
//    if (oldEvent.id == NO_DATA_ID)  Log.d("maxiEpg", "findEvents ch: $channelNumber remNod ${EpgChannelExt.print1(oldEvent.startTime, oldEvent.endTime)}")
//    else                            Log.d("maxiEpg", "findEvents ch: $channelNumber remEve ${EpgChannelExt.print1(oldEvent.startTime, oldEvent.endTime)}")
    eventsToRemove.add(oldEvent)
}

private fun addNewDataOrNoData(
    newEvent: SmartMediaItem,
    channelNumber: Int,
    eventsToAdd: ArrayList<EpgEventExt>,
    getRecordingDrawableBlock: (event: SmartMediaItem) -> EpgEventLabel?,
    lastEndTimeMs: Long,
    updateLastEndTimeBlock: (millis: Long) -> Unit
) {
    val newEventStart = newEvent.startDate!!.millis
    val newEventEnd = newEvent.endDate!!.millis
    addNoDataIfApplicable(lastEndTimeMs, newEventStart, channelNumber, eventsToAdd)
    if (newEventStart >= lastEndTimeMs) {
        addNewData(newEvent, channelNumber, eventsToAdd, getRecordingDrawableBlock, updateLastEndTimeBlock)
    } else if (newEventEnd > lastEndTimeMs + NO_DATA_MINIMAL_TIME_FRAME) {
        newEvent.startDate = DateTime(lastEndTimeMs)
        addNewData(newEvent, channelNumber, eventsToAdd, getRecordingDrawableBlock, updateLastEndTimeBlock)
    } else {
//        Log.d("maxiEpg", "findEvents ch: $channelNumber skpEve ${EpgChannelExt.print1(newEventStart, newEventEnd)}")
    }
}

private fun addNewData(
    newEvent: SmartMediaItem?,
    channelNumber: Int,
    eventsToAdd: ArrayList<EpgEventExt>,
    getRecordingDrawableBlock: (event: SmartMediaItem) -> EpgEventLabel?,
    updateLastEndTimeBlock: (millis: Long) -> Unit
) {
    newEvent?.startDate ?: return
    newEvent.endDate ?: return
    val lastEndTimeMs = newEvent.endDate!!.millis
    updateLastEndTimeBlock(lastEndTimeMs)
//    Log.d("maxiEpg", "findEvents ch: $channelNumber addEve ${EpgChannelExt.print1(newEvent.startDate!!.millis, lastEndTimeMs)}")
    eventsToAdd.add(newEvent.toEpgEventExt(getRecordingDrawableBlock))
}

private fun addNoDataIfApplicable(
    start: Long,
    end: Long,
    channelNumber: Int,
    eventsToAdd: ArrayList<EpgEventExt>,
    label: String? = null
) {
    if (NO_DATA_ENABLED && start + NO_DATA_MINIMAL_TIME_FRAME < end) {
//        Log.d("maxiEpg", "findEvents ch: $channelNumber addNod ${EpgChannelExt.print1(start, end)}")
        eventsToAdd.add(buildNoData(start, end, label))
    }
}

private fun EpgEventExt.isEqualTo(another: SmartMediaItem): Boolean {
    if (startTime != another.startDate!!.millis) return false
    if (endTime != another.endDate!!.millis) return false
    return true
}

private fun SmartMediaItem.toEpgEventExt(getRecordingDrawableBlock: (event: SmartMediaItem) -> EpgEventLabel?): EpgEventExt {
    return EpgEventExt(
            smartMediaItem = this,
            labelDrawables = ArrayList<EpgEventLabel>().also { dList ->
                getRecordingDrawableBlock(this)?.also { dList.add(it) }
            }
    )
}

private fun <E> Iterator<E>.nextOrNull() = if(hasNext()) next() else null

private fun <E> Iterator<E>.nextEpgEventExtOrNull(): EpgEventExt? {
    while (hasNext()) {
        (next() as? EpgEventExt)?.also { return it }
    }
    return null
}

//fun EpgData.addDataAndFix(data: Map<EpgChannel, List<EpgEvent>>, epgFromTime: Long, epgToTime: Long) {
//    addData(data)
//    fixEvents(data, epgFromTime, epgToTime)
//}
//
//private fun EpgEvent.timeContains(event: EpgEvent): Boolean =
//        startTime <= event.startTime && endTime >= event.endTime
//
//fun EpgData.fixEvents(data: Map<EpgChannel, List<EpgEvent>>, epgFromTime: Long, epgToTime: Long) {
//
//    for (epgChannel in data.keys) {
//        var existingChannelIndex = -1
//        findChannelIndex@for (index in 0..channelCount) {
//            if (getChannel(index).id == epgChannel.id) {
//                existingChannelIndex = index
//                break@findChannelIndex
//            }
//        }
//
//        if (existingChannelIndex < 0) continue
//
//        // remove all no data events
//        getEvents(existingChannelIndex).removeAll {
//            it.id == NO_DATA_ID
//        }
//
//        /*
//         * Sometimes, duplicate events are added.
//         * They have the same ID, but their end times are different, leading to glitched EPG
//         * items.
//         *
//         * This checks for items that are completely overlapped by others and removes them if
//         * they have the same ID.
//         */
//        getEvents(existingChannelIndex).let { events ->
//            // Iterate in reverse so that we can remove items safely
//            for (i in (0..events.size).reversed()) {
//                val nextI = i + 1
//
//                if (nextI < events.size && events[nextI].id == events[i].id) {
//                    // Only remove items that overlap completely
//                    if (events[i].timeContains(events[nextI])) {
//                        // Current event covers the next one completely. Remove it.
//                        events.removeAt(i)
//                    } else if (events[nextI].timeContains(events[i])) {
//                        // The next event covers the current one completely. Remove it.
//                        events.removeAt(nextI)
//                    }
//                }
//            }
//        }
//
//        var indexStart = 0
//        var previousEvent = buildNoData(epgFromTime, epgToTime, "<<<")
//        // add initial no data event that is before all regular events
//        if (NO_DATA_ENABLED) {
//            if (getEvents(existingChannelIndex).isEmpty()
//                || getEvents(existingChannelIndex).first().startTime > epgFromTime + NO_DATA_MINIMAL_TIME_FRAME
//            ) {
//                // no data event before all regular events
//                getEvents(existingChannelIndex).add(indexStart, previousEvent)
//                indexStart++
//            }
//        }
//
//        var index = indexStart
//        var skippedCount = 0
//        while (index < getEvents(existingChannelIndex).size) {
//            val it = getEvents(existingChannelIndex)[index]
//            if (previousEvent.id == it.id) {
//                // skip duplicate events
//                index++
//                skippedCount++
//                continue
//            }
//            if (NO_DATA_ENABLED && previousEvent.endTime + NO_DATA_MINIMAL_TIME_FRAME < it.startTime) {
//                // insert no data events in gaps in epg
//                val noDataEvent = buildNoData(previousEvent.endTime, it.startTime)
//                getEvents(existingChannelIndex).add(
//                    index,
//                    noDataEvent)
//                // fixes no data event to not show over areas where loading was not finished yet
//                index += fixNoData(previousEvent, existingChannelIndex, index - skippedCount - 1)
//                previousEvent = noDataEvent
//                index++
//            } else {
//                // fix potential event overlap
//                if (previousEvent.endTime > it.startTime) {
//                    previousEvent.updateTime(previousEvent.startTime, it.startTime)
//                }
//            }
//            // fixes no data event to not show over areas where loading was not finished yet
//            if (NO_DATA_ENABLED) index += fixNoData(previousEvent, existingChannelIndex, index - skippedCount - 1)
//            //
//            previousEvent = it
//            skippedCount = 0
//            index++
//        }
//
//        // add no data event that is after all regular events
//        if (NO_DATA_ENABLED) {
//            if (previousEvent.endTime + NO_DATA_MINIMAL_TIME_FRAME < epgToTime) {
//                previousEvent = buildNoData(previousEvent.endTime, epgToTime)
//                getEvents(existingChannelIndex).add(previousEvent)
//                index++
//            }
//            // fix last event
//            fixNoData(previousEvent, existingChannelIndex, index - 1)
//        }
//    }
//}
//
///**
// * Takes potential no data event and process it, so it is not present in areas of channel
// * that are not loaded yet.
// *
// * @param noDataEvent - event that we need to check for fix
// * @param channelIndex - index of channel, where fix of no data is necessary
// * @param indexOfNoData - index in list of events where no data event is currently placed
// *
// * @return Change in size of list of events in channel. For example when 1 event is removed, returns "-1".
// * In future changes, if 2 new no data events were added, returns 2.
// */
//private fun EpgData.fixNoData(
//    noDataEvent: EpgEvent,
//    channelIndex: Int,
//    indexOfNoData: Int
//): Int {
////    // if fast disable is needed
////    return noDataEvent
//
//    val epgChannelExt = getChannel(channelIndex) as? EpgChannelExt
//    if (noDataEvent.id != NO_DATA_ID) return 0 // ignore regular events
//    if (epgChannelExt == null) return 0 // ignore non EXT channels
//    val loadedRanges = epgChannelExt.getLoadedRanges(noDataEvent.startTime, noDataEvent.endTime)
//        .filter { it.status == EpgChannelExt.Status.LOADED }
//
//    // if no range is loaded - remove no data
//    if (loadedRanges.isEmpty()) {
//        getEvents(channelIndex).remove(noDataEvent)
//        return -1
//    }
//
//    // TODO in future improvements we should implement splitting mechanism
//    // TODO so when middle section of no data event is not loaded, we have to create 2 events
//    // TODO one before not loaded section, and one after such section
//    // TODO warning: this will have to be recursive, since there may be multiple such ranges in original no data event
//
//    // cut no data event to not start before first loaded range
//    loadedRanges.minBy { it.from }?.let {
//        if (noDataEvent.startTime < it.from) {
//            noDataEvent.updateTime(it.from, noDataEvent.endTime)
//        }
//    }
//
//    // cut no data event to not end after last loaded range
//    loadedRanges.maxBy { it.to }?.let {
//        if (noDataEvent.endTime > it.to) {
//            noDataEvent.updateTime(noDataEvent.startTime, it.to)
//        }
//    }
//
//    //  if width 0, then remove it from event list
//    if (noDataEvent.endTime - noDataEvent.startTime <= 0) {
//        getEvents(channelIndex).remove(noDataEvent)
//        return -1
//    }
//
//    return 0
//}
//
fun buildNoData(epgFromTime: Long, epgToTime: Long, label: String? = null): EpgEventExt {
    //Log.d("maxiEpg", "addDataSuperSafely NO_DATA: ${EpgChannelExt.print1(epgFromTime, epgToTime)}")
    return EpgEventExt(
        smartMediaItem = SmartMediaItem(
            Flavor().getSmartMediaReferenceForId(NO_DATA_ID),
            SmartMediaType.LIVE_EVENT
        ).apply {
            startDate = DateTime(epgFromTime)
            endDate = DateTime(epgToTime)
            title = label ?: StringsMapper.translate(StringsMapper.EPG_NO_DATA)
        },
        labelDrawables = ArrayList(),
        noDataId = NO_DATA_ID
    )
}