package com.twentyfouri.tvlauncher.homepagechannels

import android.app.Activity
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannel
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannelItem
import com.twentyfouri.tvlauncher.ui.MainActivity
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import timber.log.Timber

/**
 * Delegates to the correct activity based on how the user entered the app.
 *
 * Supports two options: view and play. The view option will open the channel for the user to be
 * able to view more programs. The play option will start playing the movie.
 */
class AppLinkActivity : Activity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val intent = intent
        val uri = intent.data
        Timber.v(uri.toString())
        if (uri?.pathSegments?.isEmpty() != false) {
            Timber.e("Invalid uri $uri")
            finish()
            return
        }
        val action: AppLinkHelper.AppLinkAction? = AppLinkHelper.extractAction(uri)
        when (action?.getAction()) {
            AppLinkHelper.PLAYBACK -> play(action as AppLinkHelper.PlaybackAction)
            AppLinkHelper.BROWSE -> browse(action as AppLinkHelper.BrowseAction)
            AppLinkHelper.KEY_EVENT -> handleKeyEvent(action as AppLinkHelper.KeyEventAction) //appLink uri scheme: "scheme://host/keyEvent/{keyCode}"
            else -> throw IllegalArgumentException("Invalid Action $action")
        }
    }

    private fun browse(action: AppLinkHelper.BrowseAction) = CoroutineScope(Dispatchers.Main).launch {
        val homepageChannel: HomepageChannel? = withContext(Dispatchers.IO) {
            HomepageChannelDatabaseRepository.findHomepageChannelByName(this@AppLinkActivity, action.channelName)
        }

        if (homepageChannel == null) {
            Timber.e("Invalid subscription ${action.channelName}")
            if (action.defaultNavigationTarget.isNullOrEmpty().not()) {
                navigateTo(action.defaultNavigationTarget)
            }
        } else { // TODO: Open an activity that has the movies for the subscription.
            Toast.makeText(this@AppLinkActivity, action.channelName, Toast.LENGTH_LONG).show()
        }
        finish()
    }

    private fun play(action: AppLinkHelper.PlaybackAction) = CoroutineScope(Dispatchers.Main).launch {
        if (action.position == AppLinkHelper.DEFAULT_POSITION.toLong()) Timber.d("Playing channel item ${action.channelItemId} from channel ${action.channelId}")
        else Timber.d("Continuing channel item ${action.channelItemId} from channel ${action.channelId} at time ${action.position}")
        val homepageChannelItem =
            HomepageChannelDatabaseRepository.findChannelItemById(this@AppLinkActivity,
                action.channelId,
                action.channelItemId)

        if (homepageChannelItem == null) {
            Timber.e("Invalid program ${action.channelItemId}")
            navigateTo(HOME)
        } else {
            startPlaying(action.channelId, homepageChannelItem, action.position)
        }
        finish()
    }

    private fun startPlaying(channelId: Long, homepageChannelItem: HomepageChannelItem, position: Long) {
//        Toast.makeText(this, "AppLinkActivity.startPlaying channelId: $channelId movie: $movie position: $position", Toast.LENGTH_LONG).show()

        val playHomepageChannelItemIntent = Intent(this, MainActivity::class.java).apply {
            action = Intent.ACTION_VIEW
            data = Uri.parse(homepageChannelItem.appLinkUri)
        }

        try {
            startActivity(playHomepageChannelItemIntent)
        } finally {
            finish()
        }
    }

    private fun navigateTo(target: String?){
        if (!target.isNullOrEmpty()) {
            Timber.d("Navigating to $target")
            val navigationIntent = Intent(this, MainActivity::class.java).apply {
                when (target) {
                    HOME -> action = MainActivity.INTENT_ACTION_MAIN
                    EPG -> action = MainActivity.INTENT_ACTION_EPG
                    else -> action = Intent.ACTION_VIEW
                }
            }
            try {
                startActivity(navigationIntent)
            } finally {
                finish()
            }
        }
    }

    private fun handleKeyEvent(action: AppLinkHelper.KeyEventAction){
        val keyEventIntent = Intent().apply {
            this.action = ACTION_APP_LINK_KEY
            this.`package` = packageName
            this.putExtra("keyCode", action.KeyCode)
        }
        try {
           sendBroadcast(keyEventIntent)
        } finally {
            finish()
        }
    }

    companion object {
        private const val TAG = "AppLinkActivity"
        private const val ACTION_APP_LINK_KEY = "com.twentyfouri.tvlauncher.APP_LINK_KEY"
        const val EPG = "EPG"
        const val HOME = "HOME"
    }
}