package com.twentyfouri.tvlauncher.homepagechannels

import android.net.Uri
import android.view.KeyEvent

/** Builds and parses uris for deep linking within the app.
 *
 * For testing purposes deep link can be triggered via adb:
 * $ adb shell am start -a android.intent.action.VIEW -d <APP_LINK_URI> <PACKAGE>
 * */
object AppLinkHelper {

    const val PLAYBACK = "playback"
    const val BROWSE = "browse"
    const val KEY_EVENT = "keyEvent"
    private const val URI_INDEX_OPTION = 0
    private const val URI_INDEX_KEY_CODE = 1
    private const val URI_INDEX_CHANNEL_ID = 2
    private const val URI_INDEX_DEFAULT_TARGET = 2
    private const val URI_INDEX_CHANNEL_NAME = 1
    private const val URI_INDEX_MOVIE_ID = 3
    private const val URI_INDEX_POSITION = 4
    const val DEFAULT_POSITION = -1

    fun buildPlaybackUri(channelId: Long, channelItemId: Long, position: Long, scheme: String, host: String, appLinkJson: String): Uri = Uri
        .withAppendedPath(Uri.parse("$scheme://$host"), PLAYBACK)
        .buildUpon()
        .appendPath(appLinkJson)
        .appendPath(channelId.toString())
        .appendPath(channelItemId.toString())
        .appendPath(position.toString())
        .build()

    /**
     * Builds a [Uri] to deep link into viewing a channel.
     *
     * @param channelName - name of the channel.
     * @return a uri.
     */
    fun buildBrowseUri(channelName: String?, defaultNavigationTarget: String?, scheme: String, host: String): Uri? = Uri
        .withAppendedPath(Uri.parse("$scheme://$host"), BROWSE)
        .buildUpon()
        .appendPath(channelName)
        .appendPath(defaultNavigationTarget)
        .build()

    /**
     * Returns an [AppLinkAction] for the given Uri.
     *
     * @param uri to determine the intended action.
     * @return an action.
     */
    fun extractAction(uri: Uri): AppLinkAction? {
        return when {
            isPlaybackUri(uri) -> PlaybackAction(extractChannelId(uri), extractChannelItemId(uri), extractPosition(uri))
            isBrowseUri(uri) -> BrowseAction(extractChannelName(uri), extractDefaultNavigationTarget(uri))
            isKeyEventUri(uri) -> KeyEventAction(extractKeyCode(uri))
            else -> throw IllegalArgumentException("No action found for uri $uri")
        }
    }

    /**
     * Tests if the [Uri] was built for playing a movie.
     *
     * @param uri to examine.
     * @return true if the uri is for playing a movie.
     */
    private fun isPlaybackUri(uri: Uri): Boolean {
        if (uri.pathSegments.isEmpty()) return false
        val option = uri.pathSegments[URI_INDEX_OPTION]
        return PLAYBACK == option
    }

    /**
     * Tests if a [Uri] was built for browsing a channel.
     *
     * @param uri to examine.
     * @return true if the Uri is for browsing a channel.
     */
    private fun isBrowseUri(uri: Uri): Boolean {
        if (uri.pathSegments.isEmpty()) return false
        val option = uri.pathSegments[URI_INDEX_OPTION]
        return BROWSE == option
    }

    /**
     * Tests if the [Uri] was built for dispatching key event.
     *
     * @param uri to examine.
     * @return true if the uri is for dispatching key event.
     */
    private fun isKeyEventUri(uri: Uri): Boolean {
        if (uri.pathSegments.isEmpty()) return false
        val option = uri.pathSegments[URI_INDEX_OPTION]
        return KEY_EVENT == option
    }

    /**
     * Extracts the channel name from the [Uri].
     *
     * @param uri that contains a channel name.
     * @return the channel name.
     */
    private fun extractChannelName(uri: Uri): String? = extract(uri, URI_INDEX_CHANNEL_NAME)

    /**
     * Extracts the channel id from the [Uri].
     *
     * @param uri that contains a channel id.
     * @return the channel id.
     */
    private fun extractChannelId(uri: Uri): Long = extractLong(uri, URI_INDEX_CHANNEL_ID)

    /**
     * Extracts the movie id from the [Uri].
     *
     * @param uri that contains a movie id.
     * @return the movie id.
     */
    private fun extractChannelItemId(uri: Uri): Long = extractLong(uri, URI_INDEX_MOVIE_ID)

    /**
     * Extracts the playback mPosition from the [Uri].
     *
     * @param uri that contains a playback mPosition.
     * @return the playback mPosition.
     */

    private fun extractKeyCode(uri: Uri): Int = extract(uri, URI_INDEX_KEY_CODE)?.toInt() ?: KeyEvent.KEYCODE_UNKNOWN

    private fun extractDefaultNavigationTarget(uri: Uri): String? = extract(uri, URI_INDEX_DEFAULT_TARGET)

    private fun extractPosition(uri: Uri): Long = extractLong(uri, URI_INDEX_POSITION)

    private fun extractLong(uri: Uri, index: Int): Long = extract(uri, index)?.toLong() ?: 0

    private fun extract(uri: Uri, index: Int): String? = uri.pathSegments.getOrNull(index)

    /** Action for deep linking.  */
    interface AppLinkAction {
        /** Returns an string representation of the action.  */
        fun getAction(): String
    }

    /** Browse a channel.  */
    class BrowseAction(
        val channelName: String?,
        val defaultNavigationTarget: String?
    ): AppLinkAction {
        override fun getAction() = BROWSE
    }

    /** Play a movie.  */
    class PlaybackAction(
        val channelId: Long,
        val channelItemId: Long,
        val position: Long
    ): AppLinkAction {
        override fun getAction() = PLAYBACK
    }

    class KeyEventAction(
        val KeyCode: Int
    ): AppLinkAction {
        override fun getAction() = KEY_EVENT
    }
}