/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.twentyfouri.tvlauncher.notifications

import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.RectF
import android.text.TextUtils
import android.util.AttributeSet
import android.view.View
import android.view.ViewTreeObserver
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView

import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import timber.log.Timber

/**
 * View for a non-dismissible notification displayed in the notifications side panel.
 */
open class NotificationPanelItemView : LinearLayout, ViewTreeObserver.OnGlobalFocusChangeListener {
    private var mProgressBounds: RectF? = null
    private lateinit var mProgressPaint: Paint
    private lateinit var mProgressMaxPaint: Paint
    private var mProgressStrokeWidth: Int = 0
    private var mProgressDiameter: Int = 0
    private var mProgressPaddingStart: Int = 0
    private var mProgressPaddingTop: Int = 0
    private var mProgressColor: Int = 0
    private var mProgressMaxColor: Int = 0
    private var mProgress: Int = 0
    private var mProgressMax: Int = 0
    private var mIsRtl: Boolean = false
    private lateinit var mIcon: ImageView
    private lateinit var mTitle: TextView
    private lateinit var mText: TextView
    private lateinit var mExpandedText: TextView
    protected lateinit var mMainContentText: View
    protected var mNotificationKey: String? = null
    private var mNotification: TvNotification? = null

    private val isContentTextCutOff: Boolean
        get() {
            val layout = mText.layout
            if (layout != null) {
                val lines = layout.lineCount
                if (lines > 0) {
                    val ellipsisCount = layout.getEllipsisCount(lines - 1)
                    if (ellipsisCount > 0) {
                        return true
                    }
                }
            }
            return false
        }

    constructor(context: Context) : super(context) {
        initializeLayoutValues()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        initializeLayoutValues()
    }

    private fun initializeLayoutValues() {
        val config = context.resources.configuration
        mIsRtl = config.layoutDirection == View.LAYOUT_DIRECTION_RTL

        val res = resources
        mProgressStrokeWidth = res.getDimensionPixelSize(
            R.dimen.notification_progress_stroke_width
        )
        mProgressColor = res.getColor(R.color.notification_progress_stroke_color, null)
        mProgressMaxColor = res.getColor(R.color.notification_progress_stroke_max_color, null)
        mProgressDiameter = res.getDimensionPixelSize(R.dimen.notification_progress_circle_size)
        mProgressPaddingTop = res.getDimensionPixelOffset(
            R.dimen.notification_progress_circle_padding_top
        )
        mProgressPaddingStart = res.getDimensionPixelOffset(
            R.dimen.notification_progress_circle_padding_start
        )

        mProgressPaint = Paint().apply {
            isAntiAlias = true
            style = Paint.Style.STROKE
            color = mProgressColor
            strokeWidth = mProgressStrokeWidth.toFloat()
        }

        mProgressMaxPaint = Paint().apply {
            isAntiAlias = true
            style = Paint.Style.STROKE
            color = mProgressMaxColor
            strokeWidth = mProgressStrokeWidth.toFloat()
        }

    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        viewTreeObserver.addOnGlobalFocusChangeListener(this)
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        viewTreeObserver.removeOnGlobalFocusChangeListener(this)
    }

    override fun onFinishInflate() {
        super.onFinishInflate()
        mIcon = findViewById(R.id.notification_icon)
        mTitle = findViewById(R.id.notification_title)
        mText = findViewById(R.id.notification_text)
        mMainContentText = findViewById(R.id.notification_container)
        mExpandedText = findViewById(R.id.notification_expanded_text)

        mMainContentText.setOnClickListener { view ->
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Notification item selected: ${mTitle.text}")
            if (mNotificationKey != null) {
                NotificationsUtils.openNotification(view.context, mNotificationKey!!)
            }
        }
    }

    open fun setNotification(notif: TvNotification) {
        mNotification = notif
        mNotificationKey = notif.notificationKey
        mTitle.text = notif.title
        mText.text = notif.text
        if (!TextUtils.isEmpty(notif.title)) {
            if (!TextUtils.isEmpty(notif.text)) {
                val formatting = resources.getString(
                    R.string.notification_content_description_format
                )
                mMainContentText.contentDescription = String.format(formatting, notif.title, notif.text)
            } else {
                mMainContentText.contentDescription = notif.title
            }
        } else {
            mMainContentText.contentDescription = notif.text
        }
        mExpandedText.text = notif.text
        mIcon.setImageIcon(notif.smallIcon)
        setProgress(notif.progress, notif.progressMax)
        mMainContentText.visibility = View.VISIBLE
    }

    fun setProgress(progress: Int, progressMax: Int) {
        mProgress = progress
        mProgressMax = progressMax
        if (mProgressMax != 0) {
            if (mProgressBounds == null) {
                mProgressBounds = RectF()
            }
            setWillNotDraw(false)
        } else {
            mProgressBounds = null
            setWillNotDraw(true)
        }
        requestLayout()
    }

    override fun onLayout(changed: Boolean, l: Int, t: Int, r: Int, b: Int) {
        super.onLayout(changed, l, t, r, b)
        val left: Int
        val right: Int
        val top = mProgressPaddingTop
        val bottom = top + mProgressDiameter
        if (mIsRtl) {
            right = r - mProgressPaddingStart
            left = right - mProgressDiameter
        } else {
            left = mProgressPaddingStart
            right = left + mProgressDiameter
        }

        mProgressBounds?.set(left.toFloat(), top.toFloat(), right.toFloat(), bottom.toFloat())
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        val mProgressBounds = this.mProgressBounds
        if (mProgressMax != 0 && mProgressBounds != null) {
            val sweepAngle = 360f * mProgress / mProgressMax
            if (mIsRtl) {
                canvas.drawArc(mProgressBounds, -90f, -sweepAngle, false, mProgressPaint)
                canvas.drawArc(
                    mProgressBounds, -90f, 360 - sweepAngle, false,
                    mProgressMaxPaint
                )
            } else {
                canvas.drawArc(mProgressBounds, -90f, sweepAngle, false, mProgressPaint)
                canvas.drawArc(
                    mProgressBounds, sweepAngle - 90, 360 - sweepAngle, false,
                    mProgressMaxPaint
                )
            }
        }
    }

    protected fun expandText() {
        if (isContentTextCutOff) {
            mText.visibility = View.GONE
            mTitle.maxLines = 2
            mExpandedText.visibility = View.VISIBLE
        }
        background = resources.getDrawable(R.drawable.button_background_gradient_60, null)
    }

    protected fun collapseText() {
        mExpandedText.visibility = View.GONE
        mTitle.maxLines = 1
        mText.visibility = View.VISIBLE
        setBackgroundColor(Color.TRANSPARENT)
    }

    override fun onGlobalFocusChanged(oldFocus: View?, newFocus: View?) {
        val currentFocus = focusedChild
        if (currentFocus == null) {
            collapseText()
        } else if ((newFocus === currentFocus || newFocus?.parent === currentFocus)) {
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Notification item focused: ${mTitle.text}")
            expandText()
        }
    }
}
