package com.twentyfouri.tvlauncher.notifications

import android.content.Context
import android.content.SharedPreferences
import android.database.Cursor
import androidx.lifecycle.LiveData

/**
 * Read/unread notification tracking logic.
 */
class ReadNotifications(context: Context) : LiveData<Set<String>>() {
    private val prefs = context.applicationContext.getSharedPreferences(VIEWED_PREFS_FILE, Context.MODE_PRIVATE)

    init {
        value = setOf()
    }

    private val listener = SharedPreferences.OnSharedPreferenceChangeListener { prefs, key ->
        if (key == VIEWED_KEY) {
            postValue(prefs.getStringSet(VIEWED_KEY, null) ?: setOf())
        }
    }

    fun markAsRead(keys: Set<String>?) = prefs.edit()
        .putStringSet(VIEWED_KEY, keys)
        .apply()

    fun markAsRead(cursor: Cursor?) = markAsRead(getKeysFromCursor(cursor))

    fun hasUnread(keys: Set<String>?): Boolean = !(value!!.containsAll(keys ?: setOf()))

    fun hasUnread(cursor: Cursor?): Boolean = hasUnread(getKeysFromCursor(cursor))

    override fun onInactive() {
        prefs.unregisterOnSharedPreferenceChangeListener(listener)
    }

    override fun onActive() {
        postValue(prefs.getStringSet(VIEWED_KEY, null) ?: setOf())
        prefs.registerOnSharedPreferenceChangeListener(listener)
    }

    companion object {
        private const val VIEWED_PREFS_FILE = "viewed_notifications"
        private const val VIEWED_KEY = "viewed"

        fun getKeysFromCursor(cursor: Cursor?):Set<String> {
            val keySet = HashSet<String>()
            cursor?.apply {
                if (!moveToFirst()) return@apply
                val idx = cursor.getColumnIndex(NotificationsContract.COLUMN_SBN_KEY)

                do {
                    keySet.add(cursor.getString(idx))
                } while (moveToNext())
            }
            return keySet
        }
    }
}