package com.twentyfouri.tvlauncher.receiver

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.media.AudioManager
import java.util.*
import kotlin.concurrent.schedule

class HDMIStatusReceiver : BroadcastReceiver() {
    companion object {
        const val HDMI_INTENT = "android.intent.action.HDMI_PLUGGED"
        const val VOLUME_CHANGE_INTENT = "android.media.VOLUME_CHANGED_ACTION"
        private const val SCREEN_BROADCAST_WAIT: Long = 3/*seconds*/ * 1000/*millis*/
    }

    var currentVolume: Int = -1
    var appResumed: Boolean = false

    override fun onReceive(context: Context?, intent: Intent?) {
        when (intent?.action) {
            HDMI_INTENT -> if (intent.getBooleanExtra("state", false)) onHDMIPlugged(context) else onHDMIUnplugged(context)
            VOLUME_CHANGE_INTENT -> onVolumeChanged(context)
        }
    }

    private fun getAudioManager(context: Context?): AudioManager? {
        return context?.getSystemService(Context.AUDIO_SERVICE) as? AudioManager?
    }

    private fun updateCurrentVolume(context: Context?) {
        getAudioManager(context)?.getStreamVolume(AudioManager.STREAM_MUSIC)?.let { streamMusicVolume ->
            currentVolume = streamMusicVolume
        }
    }

    private fun setVolume(context: Context?, volume: Int) {
        getAudioManager(context)?.setStreamVolume(AudioManager.STREAM_MUSIC, volume, 0)
    }

    private fun onVolumeChanged(context: Context?) {
        // If the user changes the device stream music volume value, we update our current value.
        if (ScreenOnOffReceiver.screenWasOff) return //Device in standby, do not change volume
        if (!appResumed){
            // App is paused and maybe going to standby
            awaitForScreenOffBroadcast { updateCurrentVolume(context) }
        } else {
            updateCurrentVolume(context)
        }
    }

    private fun onHDMIPlugged(context: Context?) {
        if (currentVolume == -1) {
            // At start up if our device is plugged will call this method.
            // If the current volume is -1 it means that we didn't set the start up value. Here we set it.
            updateCurrentVolume(context)
        } else {
            // If we plugged the HDMI into the device after start up, we set the volume to the value we set before unplugging or once we unplugged.
            setVolume(context, currentVolume)
        }
    }

    private fun onHDMIUnplugged(context: Context?) {
        // When the HDMI is unplugged by default will set the volume to 15. Here we set it to our last volume value.
        updateCurrentVolume(context)
        setVolume(context, currentVolume)
    }

    private fun awaitForScreenOffBroadcast(callback: () -> Unit) {
        val t0 = System.currentTimeMillis()
        Timer().schedule(1000L, 1000L) {
            if (ScreenOnOffReceiver.screenWasOff){
                cancel()
                return@schedule
            }
            if (System.currentTimeMillis() - t0 > SCREEN_BROADCAST_WAIT) {
                callback.invoke()
                cancel()
                return@schedule
            }
        }
    }
}