package com.twentyfouri.tvlauncher.services

import android.content.Intent
import android.os.IBinder
import android.text.format.DateFormat
import androidx.lifecycle.LifecycleService
import androidx.lifecycle.lifecycleScope
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.common.utils.SharedPreferencesUtils
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_RUN_LOG
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber

class RemoteLoggingService : LifecycleService() {
    private val binder = object: RemoteLoggingInterface.Stub() {
        private val wrapped = RemoteLoggingBridge()

        override fun enableRemoteLogging(time: Int, storage: Int) {
            /*
             * Run it all on the main thread, to avoid potential inconsistencies between
             * prefs and the logger.
             */
            lifecycleScope.launch(Dispatchers.Main) {
                wrapped.enableRemoteLogging(time, storage)
            }
        }

        override fun disableRemoteLogging() {
            lifecycleScope.launch(Dispatchers.Main) {
                wrapped.disableRemoteLogging()
            }
        }

        override fun getRemoteLoggingStatus(): String = runBlocking(Dispatchers.Main) {
            wrapped.getRemoteLoggingStatus()
        }

        override fun getDetailedRemoteLoggingStatus(): String = runBlocking(Dispatchers.Main) {
            wrapped.getDetailedRemoteLoggingStatus()
        }
    }

    override fun onBind(intent: Intent): IBinder {
        super.onBind(intent)
        return binder
    }

    class RemoteLoggingBridge: KoinComponent {
        fun enableRemoteLogging(time: Int, storage: Int) {
            try {
                val loggingStopDateMs: Long = if (time > 0) {
                    TimeProvider.nowMs() + (time * 1000)
                } else 0
                val loggingStopDate = if (loggingStopDateMs > 0) {
                    DateFormat.format("dd.MM.yyyy HH:mm:ss", loggingStopDateMs)
                } else "NEVER"
                SharedPreferencesUtils.putRemoteLoggingEnabled(true)
                SharedPreferencesUtils.putRemoteLoggingStopDate(loggingStopDateMs)
                SharedPreferencesUtils.putRemoteLoggingStorageLimit(storage)
                Timber.tag(TAG_RUN_LOG).i("Remote logging enabled = true")
                Timber.tag(TAG_RUN_LOG).i("Remote logging time limit = %d which means to stop at %s", time, loggingStopDate)
                Timber.tag(TAG_RUN_LOG).i("Remote logging storage limit = %d", storage)
                get<OselToggleableLogger>().toggleLogging(true)
            } catch (e: Exception) {
                Timber.tag(TAG_RUN_LOG).e("Remote logging intercepted -> %s", e.stackTraceToString())
            }
        }

        fun disableRemoteLogging() {
            try {
                Timber.i("Remote logging enabled = false")
                Timber.i("Remote logging time limit = 0")
                Timber.i("Remote logging storage limit = 0")
                SharedPreferencesUtils.putRemoteLoggingEnabled(true)
                SharedPreferencesUtils.putRemoteLoggingStopDate(0)
                SharedPreferencesUtils.putRemoteLoggingStorageLimit(0)
                get<OselToggleableLogger>().toggleLogging(false)
            } catch (e: Exception) {
                Timber.tag(TAG_RUN_LOG).e("Remote logging intercepted -> %s", e.stackTraceToString())
            }
        }

        fun getRemoteLoggingStatus(): String =
            get<OselToggleableLogger>().loggingStatus.name

        fun getDetailedRemoteLoggingStatus(): String =
            get<OselToggleableLogger>().detailedLoggingStatus
    }
}