package com.twentyfouri.tvlauncher.ui

import android.animation.ObjectAnimator
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import androidx.core.os.bundleOf
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.Observer
import com.google.firebase.analytics.FirebaseAnalytics
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.extensions.enlargeFromCenter
import com.twentyfouri.tvlauncher.common.extensions.enlargeFromLeft
import com.twentyfouri.tvlauncher.common.ui.messagedialog.CANCEL
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogAction
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogCodes
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogDismissListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.common.ui.messagedialog.OPTION_A
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.databinding.FragmentDetailBinding
import com.twentyfouri.tvlauncher.extensions.isVisible
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import com.twentyfouri.tvlauncher.viewmodels.DetailViewModel
import org.koin.androidx.viewmodel.ext.android.getViewModel
import timber.log.Timber
import kotlin.math.min

class DetailFragment : BaseFragment() {

    private var nullableBinding: FragmentDetailBinding? = null
    val binding: FragmentDetailBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")
    private var alreadyPaused = false
    private var globalLayoutListener: ViewTreeObserver.OnGlobalLayoutListener? = null
    private var detailFragmentVisible = false
    private var idForLog: String = ""

    private val globalFocusListener = ViewTreeObserver.OnGlobalFocusChangeListener { _, _ ->
        nullableBinding ?: return@OnGlobalFocusChangeListener
        if (binding.rowView.findFocus() != null) {
            val translationY = (-1) * (binding.rowView.y - 16f)
            ObjectAnimator.ofFloat(
                    binding.parentTransitionView,
                    "translationY",
                    translationY
            ).setDuration(250).start()
        } else {
            ObjectAnimator.ofFloat(binding.parentTransitionView, "translationY", 0f).setDuration(250).start()
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentDetailBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@DetailFragment.viewLifecycleOwner
        detailView.binding.lifecycleOwner = lifecycleOwner
        nullableBinding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        detailFragmentVisible = false

        super.onViewCreated(view, savedInstanceState)

        val mediaReference = requireArguments().getSerializable(ARG_MEDIA_REFERENCE) as? SmartMediaReference
        idForLog = Flavor().getLogInfoFromSmartMediaReference(mediaReference)
        val viewModelFromFactory = getViewModel<DetailViewModel>()
        mediaReference?.also { viewModelFromFactory.setMediaReference(it) }

        binding.viewModel = viewModelFromFactory
        binding.detailView.binding.viewModel = viewModelFromFactory

        binding.detailView.setObservers(viewLifecycleOwner, requireContext(), this@DetailFragment.activity)
        setupRowDataObserver()
        setupButtonOnFocusChange()
        setupRecordingButtonObserver()
    }

    override fun onPause() {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page paused: Detail (ID:$idForLog)")
        Timber.tag("leaks").d("DetailFragment onPause")
        super.onPause()
        nullableBinding ?: return
        onPausePrivate()
    }

    private fun onPausePrivate() {
        this.binding.viewModel?.onPause()
        binding.detailView.onPausePrivate()
        binding.rowView.viewTreeObserver.removeOnGlobalFocusChangeListener(globalFocusListener)
        if (this.isVisible.not()) {
            // check for isVisible because when entering detail from recordings screen
            // detail fragment is paused and immediately resumed because of PersonalSettingsActivity
            alreadyPaused = true
        }
        binding.rowView.binding.rowList.viewTreeObserver.removeOnGlobalLayoutListener(globalLayoutListener)
        binding.rowView.onRowAdapterDetached()
        globalLayoutListener = null
        detailFragmentVisible = false
        //TODO make it with less encapsulation breaks
        (activity as? MainActivity)?.getBindingSafe()?.viewModel?.recordingClicked?.removeObservers(viewLifecycleOwner)
    }

    override fun onDestroyView() {
        Timber.tag("leaks").d("DetailFragment onDestroyView")
        onPausePrivate()
        binding.detailView.onDestroyView()
        binding.rowView.onDestroyView()
        super.onDestroyView()
        nullableBinding = null
    }

    override fun onBackPressed(topbarFocused: Boolean): BackPressAction {
        binding.viewModel?.isExpanded?.value?.let {
            if (it) {
                binding.viewModel?.setShownMore(false)
                return BackPressAction.RETURN
            }
        }
        return BackPressAction.POP_BACK_STACK
    }

    private fun setupRowDataObserver() {
        binding.viewModel?.rowViewModelsFromChannelReference?.observeOnce(viewLifecycleOwner) { rowModels ->
            rowModels?.firstOrNull()?.also {
                binding.rowView.bind(it)
                binding.rowView.onAttached()
            }
        }
        binding.viewModel?.scrollDown?.observe(viewLifecycleOwner) {
            if (binding.viewModel?.shownMore?.value == true)
                binding.detailView.binding.descriptionScrollView.smoothScrollBy(0,
                    binding.detailView.binding.descriptionScrollView.maxScrollAmount)
            else if (isRowLoaded()) binding.rowView.requestFocus()
        }
        binding.viewModel?.scrollUp?.observe(viewLifecycleOwner) {
            val scrollBy = min(binding.detailView.binding.descriptionScrollView.maxScrollAmount,
                binding.detailView.binding.descriptionScrollView.scrollY)
            if (scrollBy > 0) binding.detailView.binding.descriptionScrollView.smoothScrollBy(0,
                -1 * scrollBy)
        }

        binding.viewModel?.onGlobalLayout?.observe(viewLifecycleOwner) {
            val view = listOf(
                binding.detailView.binding.detailPlayButton,
                binding.detailView.binding.restartButton,
                binding.detailView.binding.deleteRecordButton,
                binding.detailView.binding.startRecordButton,
                binding.detailView.binding.stopRecordButton
            ).firstOrNull { it.isVisible() }
            view?.requestFocus()
        }
    }

    private fun checkBinding(method: String): Boolean {
        if(nullableBinding == null) {
            Log.e("Detail", "Binding is null in DetailFragment in $method method")
            return false
        }
        return true
    }

    private fun setupButtonOnFocusChange() {
        if(!checkBinding("setupButtonOnFocusChange")) return
        binding.detailView.binding.detailPlayButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) play button focused")
            v.enlargeFromLeft(hasFocus)
        }

        binding.detailView.binding.restartButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) restart button focused")
            if (binding.viewModel?.playButtonVisibility?.value == View.VISIBLE) {
                v.enlargeFromCenter(hasFocus)
            } else {
                v.enlargeFromLeft(hasFocus)
            }
        }

        binding.detailView.binding.deleteRecordButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            nullableBinding ?: return@OnFocusChangeListener
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) delete record button focused")
            if (binding.viewModel?.playButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.restartButtonVisibility?.value == View.VISIBLE
            ) {
                v.enlargeFromCenter(hasFocus)
            } else {
                v.enlargeFromLeft(hasFocus)
            }
        }

        binding.detailView.binding.startRecordButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            nullableBinding ?: return@OnFocusChangeListener
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) start record button focused")
            if (binding.viewModel?.playButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.restartButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.deleteRecordingButtonVisibility?.value == View.VISIBLE
            ) {
                v.enlargeFromCenter(hasFocus)
            } else {
                v.enlargeFromLeft(hasFocus)
            }
        }

        binding.detailView.binding.stopRecordButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            nullableBinding ?: return@OnFocusChangeListener
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) stop record button focused")
            if (binding.viewModel?.playButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.restartButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.deleteRecordingButtonVisibility?.value == View.VISIBLE
                || binding.viewModel?.planRecordingButtonVisibility?.value == View.VISIBLE
            ) {
                v.enlargeFromCenter(hasFocus)
            } else {
                v.enlargeFromLeft(hasFocus)
            }
        }

        binding.detailView.binding.showMoreButton.onFocusChangeListener = View.OnFocusChangeListener { v, hasFocus ->
            if (hasFocus) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:$idForLog) show more button focused")
            v.enlargeFromCenter(hasFocus)
        }
    }

    private fun setupRecordingButtonObserver() {
        (activity as? MainActivity)?.getBindingSafe()?.viewModel?.recordingClicked?.observe(viewLifecycleOwner) {
            if (detailFragmentVisible) binding.detailView.binding.detailDescription.startStopRecording()
        }
    }

    override fun onResume() {
        Timber.tag("leaks").d("DetailFragment onResume")
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Page resumed: Detail (ID:$idForLog)")
        super.onResume()
        FirebaseAnalytics.getInstance(requireContext()).logEvent(FirebaseAnalytics.Event.SCREEN_VIEW, bundleOf(
            FirebaseAnalytics.Param.SCREEN_NAME to "Detail",
            FirebaseAnalytics.Param.SCREEN_CLASS to "DetailFragment"
        ))
        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = viewLifecycleOwner, skipWaiting = !ScreenOnOffReceiver.screenWasOff) {
            nullableBinding ?: return@waitForConnection
            //Attempt to fix "trying to access uninitialized binding" in globalFocusListener
            //I have suspicion that below code may execute between onStop and onDestroyView
            //isResumed.not() should prevent execution
            if (isResumed.not()) return@waitForConnection
            binding.rowView.viewTreeObserver.addOnGlobalFocusChangeListener(globalFocusListener)
            detailFragmentVisible = true
            if (alreadyPaused) {
                globalLayoutListener = object : ViewTreeObserver.OnGlobalLayoutListener {
                    //TODO: find out why 3 cycles of layout are needed to focus will be handled properly
                    //it is probably some child layout finished but can't find which one
                    private var count = 0
                    override fun onGlobalLayout() {
                        if (binding.rowView.binding.sectionViewModel?.sectionLoaded == true) {
                            if (count++ >= 3) {
                                binding.rowView.requestFocus()
                                binding.viewModel?.onGlobalLayout?.call()
                                binding.rowView.binding.rowList.viewTreeObserver.removeOnGlobalLayoutListener(globalLayoutListener)
                            }
                        }
                    }
                }
                binding.rowView.binding.rowList.viewTreeObserver.addOnGlobalLayoutListener(globalLayoutListener)
            }
        }
    }

    //TODO: unify this with "SingleLiveEvent"
    private fun <T> LiveData<T>.observeOnce(lifecycleOwner: LifecycleOwner, observer: Observer<T>) {
        observe(lifecycleOwner, object : Observer<T> {
            override fun onChanged(t: T?) {
                observer.onChanged(t)
                removeObserver(this)
            }
        })
    }

    private fun isRowLoaded() = (binding.viewModel?.rowViewVisibility?.value == View.VISIBLE
            && binding.rowView.binding.rowContainer.isVisible()
            && binding.rowView.binding.sectionViewModel?.sectionLoaded == true
            )

    fun openPlayer(startFromBeginning: Boolean = false) {
        if (binding.viewModel?.checkBroadcastStarted() == true) {
            if (binding.viewModel?.restartButtonVisibility?.value == View.VISIBLE || canBeStartedFromBeginning()) {
                if (startFromBeginning) {
                    showStartOverConfirmationDialog()
                } else {
                    binding.viewModel?.onPlayButtonClicked(binding.detailView.binding.detailPlayButton)
                }
            }
        }
    }

    private fun canBeStartedFromBeginning(): Boolean {
        return binding.viewModel?.canBeStartedFromBeginning?.value ?: false
    }

    private fun showStartOverConfirmationDialog() {
        val messageDialogModel = MessageDialogModel(
                message = getString(R.string.player_ok_longpress_action_dialog_event, binding.viewModel?.getTitle()),
                description = null,
                optionsButtonText = arrayOf(getString(R.string.button_yes)),
                cancelButtonText = getString(R.string.cancel),
                code = MessageDialogCodes.reproduceFromBeginning
        )
        val confirmMessageDialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        confirmMessageDialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {} //do nothing
        }
        confirmMessageDialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when(answer.type) {
                    OPTION_A -> { binding.viewModel?.onRestartButtonClicked(binding.detailView.binding.restartButton)}
                    CANCEL -> { return false } //cancel, do nothing
                    else -> { return false }
                }
                return false
            }
        }
        (view?.context as? FragmentActivity)?.supportFragmentManager?.also {
            confirmMessageDialogFragment.show(
                it,
                "tag_dialog_startover"
            )
        }
    }
    companion object {
        private const val ARG_MEDIA_REFERENCE = "ARG_MEDIA_REFERENCE"

        fun newInstance(mediaReference: SmartMediaReference): DetailFragment {
            val args = Bundle(1)
            args.putSerializable(ARG_MEDIA_REFERENCE, mediaReference)
            val fragment = DetailFragment()
            fragment.arguments = args
            return fragment
        }
    }
}