package com.twentyfouri.tvlauncher.ui

import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.provider.Settings
import android.util.Log
import androidx.databinding.DataBindingUtil
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.androidcore.utils.ResourceImageSpecification
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ApiMissingException
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import com.twentyfouri.tvlauncher.data.*
import com.twentyfouri.tvlauncher.databinding.FavoriteAppsActivityBinding
import com.twentyfouri.tvlauncher.utils.IntentLauncher
import com.twentyfouri.tvlauncher.viewmodels.FavoriteAppsViewModel
import org.koin.androidx.viewmodel.ext.android.getViewModel
import timber.log.Timber

const val FAVORITE_APP_PICKER = 1
const val EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE = "fragment_navigation_code"
const val SELECTED_APP_ID = "selected_app_id"

class FavoriteAppsActivity: BaseActivity() {

    lateinit var binding: FavoriteAppsActivityBinding
    private var menuFragment: ListPickerFragment? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        try {
            binding = DataBindingUtil.setContentView<FavoriteAppsActivityBinding>(
                this,
                R.layout.favorite_apps_activity
            )
                .apply {
                    viewModel = getViewModel<FavoriteAppsViewModel>()
                    lifecycleOwner = this@FavoriteAppsActivity
                }
        } catch (t: ApiMissingException) {
            finish()
        }

        if (intent.getIntExtra(EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE, 0) == FAVORITE_APP_PICKER) {
            moveToFavoriteAppPicker()
            return
        }

        binding.viewModel?.setSelectedAppId(intent.getStringExtra(SELECTED_APP_ID))
        buildMenu()
    }

    private fun buildMenu() {
        val menuItems = ArrayList<ListPickerItem>()
        if (SpecialAppLists.getFixedPriorityApps(this)
                .contains(intent.getStringExtra(SELECTED_APP_ID)).not()
        ) {
            menuItems.add(
                ListPickerItem(
                    id = "remove",
                    title = getString(R.string.remove),
                    subtitle = "",
                    label = "",
                    icon = ResourceImageSpecification(R.drawable.ic_baseline_favorite_24),
                    type = SettingsItemType.TITLE_WITH_ICON
                )
            )

            menuItems.add(
                ListPickerItem(
                    id = "move",
                    title = getString(R.string.move),
                    subtitle = "",
                    label = "",
                    icon = ResourceImageSpecification(R.drawable.ic_baseline_swap_horiz_24),
                    type = SettingsItemType.TITLE_WITH_ICON
                )
            )
        }
        menuItems.add(
            ListPickerItem(
                id = "settings",
                title = getString(R.string.app_settings),
                subtitle = "",
                label = "",
                icon = ResourceImageSpecification(R.drawable.ic_baseline_settings_24),
                type = SettingsItemType.TITLE_WITH_ICON
            )
        )

        menuItems.add(
            ListPickerItem(
                id = "open",
                title = getString(R.string.open_app),
                subtitle = "",
                label = "",
                icon = ResourceImageSpecification(R.drawable.ic_baseline_open_in_new_24),
                type = SettingsItemType.TITLE_WITH_ICON
            )

        )

        menuFragment = ListPickerFragment.newInstance(
            title = getString(R.string.favorite_app_activity_title),
            defaultPosition = ListPickerFragment.DEFAULT_POSITION_TOP,
            data = menuItems,
            closeAfterSelection = false
        )

        menuFragment?.let {
            Timber.tag("FavoriteAppsActivity.buildMenu").d("commit $it")
            FirebaseCrashlytics.getInstance().log("FavoriteAppsActivity.buildMenu: commit $it")
            supportFragmentManager.beginTransaction().add(R.id.favorite_apps_root, it).commit()
        }
    }


    private fun moveToFavoriteAppPicker() {
        val fragment = FavoriteAppPickerFragment()
        Timber.tag("FavoriteAppsActivity.moveToFavoriteAppPicker").d("commit $fragment")
        FirebaseCrashlytics.getInstance()
            .log("FavoriteAppsActivity.moveToFavoriteAppPicker: commit $fragment")
        supportFragmentManager.beginTransaction()
            .add(R.id.content_frame, fragment)
            .addToBackStack(null)
            .commit()
    }

    override fun onActivityReenter(resultCode: Int, data: Intent?) {
        super.onActivityReenter(resultCode, data)
        (data?.extras?.get(ListPickerFragment.BUNDLE_KEY_RESULT) as? ListPickerItem)?.let {
            when (it.id) {
                "remove" -> {
                    binding.viewModel?.removeSelectedItem()
                    setResult(RESULT_REQUEST_ADD_REMOVE)
                    finish()
                }
                "move" -> {
                    setResult(RESULT_REQUEST_MOVE)
                    finish()
                }
                "settings" -> {
                    openSettings()
                    finish()

                }
                "open" -> {
                    openApp()
                    finish()
                }
                else -> {}
            }
        }
    }

    override fun onBackPressed() {
        finish()
        super.onBackPressed()
    }

    override fun onPause() {
        menuFragment?.isAdded.ifTrue {
            menuFragment?.let {
                supportFragmentManager.beginTransaction().remove(it).commitNow()
            }
        }
        super.onPause()
    }

    private fun openSettings(){
        val launchIntent = Intent()
        val uri = Uri.fromParts("package", intent.getStringExtra(SELECTED_APP_ID), null)
        launchIntent.action = Settings.ACTION_APPLICATION_DETAILS_SETTINGS
        launchIntent.data = uri
        IntentLauncher.launchIntent(this, launchIntent)
    }

    private fun openApp(){
        val pm = this.packageManager
        val launchIntent = pm?.getLaunchIntentForPackage(intent.getStringExtra(SELECTED_APP_ID) ?: "")
        if (launchIntent == null) {
            val intent = Intent()
            intent.action = Intent.ACTION_VIEW
            intent.`package` = intent.getStringExtra(SELECTED_APP_ID)
            IntentLauncher.launchIntent(this, launchIntent)
        } else {
            IntentLauncher.launchIntent(this, launchIntent)
        }
    }

    companion object {
        const val RESULT_REQUEST_MOVE = 11475
        const val RESULT_REQUEST_ADD_REMOVE = 11476
    }
}