package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import androidx.core.os.bundleOf
import androidx.fragment.app.setFragmentResult
import androidx.leanback.widget.BaseGridView
import androidx.leanback.widget.VerticalGridView
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.adapters.ListPickerAdapter
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.databinding.FragmentListPickerBinding
import com.twentyfouri.tvlauncher.extensions.putBoolean
import com.twentyfouri.tvlauncher.extensions.putInt
import com.twentyfouri.tvlauncher.extensions.putSerializable
import com.twentyfouri.tvlauncher.extensions.putString
import com.twentyfouri.tvlauncher.viewmodels.ListPickerViewModel
import com.twentyfouri.tvlauncher.widgets.SettingsItemView
import org.koin.androidx.viewmodel.ext.android.getViewModel
import org.koin.core.parameter.parametersOf
import timber.log.Timber

class ListPickerFragment private constructor() : BaseFragment(), ViewTreeObserver.OnGlobalFocusChangeListener {

    private var result: Bundle? = null
    private var hideFragment = true
    lateinit var binding: FragmentListPickerBinding
    lateinit var items: List<ListPickerItem>
    private var lastFocusedItem: ListPickerItem? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentListPickerBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@ListPickerFragment.viewLifecycleOwner
        binding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val viewModelFromFactory = getViewModel<ListPickerViewModel> { parametersOf(requireArguments().getString(BUNDLE_KEY_TITLE, "")) }

        binding.viewModel = viewModelFromFactory

        @Suppress("UNCHECKED_CAST")
        items = requireArguments().getSerializable(BUNDLE_KEY_DATA) as? ArrayList<ListPickerItem> ?: emptyList()

        val adapter = ListPickerAdapter(listener = object : ListPickerAdapter.OnItemPickedListener{
            override fun onItemPicked(position: Int, item: ListPickerItem) {
                val title = if(item.title.isEmpty()) item.label else item.title
                Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Side menu $position selected: $title")
                if (requireArguments().getBoolean(BUNDLE_HAS_SUBDATA)) {
                    showSubList(position)
                } else {
                    hideFragment = false
                    result = bundleOf(
                        BUNDLE_KEY_RESULT to item,
                        BUNDLE_KEY_TITLE to arguments?.getString(BUNDLE_KEY_TITLE)
                    )
                    if (requireArguments().getBoolean(BUNDLE_KEY_CLOSE_AFTER_SELECTION)) {
                        activity?.onBackPressed()
                    } else {
                        sendResult()
                    }
                }
            }
        })
        binding.listPickerRecycler.adapter = adapter
        setScrollDefault()
        adapter.submitList(items)

        val selectedItemIndex = arguments?.getInt(BUNDLE_KEY_DEFAULT_SELECTED_ITEM_INDEX)?.let {
            if (it == DEFAULT_SELECTED_ITEM_INDEX_MIDDLE) {
                items.size / 2
            } else it
        } ?: 0

        binding.listPickerRecycler.apply {
            selectedPosition = selectedItemIndex
            requestFocus()
        }
    }

    override fun onBackPressed(topbarFocused: Boolean): BackPressAction {
        hideFragment = false
        sendResult()
        return BackPressAction.POP_BACK_STACK
    }

    private fun sendResult() {
        setFragmentResult(REQUEST_KEY, result ?: Bundle())
    }

    private fun setScrollDefault() {
        binding.listPickerRecycler.apply {
            itemAlignmentOffset = 0
            itemAlignmentOffsetPercent = VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED
            isItemAlignmentOffsetWithPadding = true
            windowAlignment = BaseGridView.WINDOW_ALIGN_BOTH_EDGE
            windowAlignmentOffset = 0
            windowAlignmentOffsetPercent = if (arguments?.getInt(BUNDLE_KEY_SELECTED_POSITION) == SELECTED_POSITION_MIDDLE) {
                19f //focused item aligned to second item
            } else {
                VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED //focused item aligned to top
            }
        }
    }

    override fun onPause() {
        if (hideFragment) activity?.supportFragmentManager?.popBackStack()
        view?.viewTreeObserver?.removeOnGlobalFocusChangeListener(this)
        super.onPause()
    }

    override fun onResume() {
        super.onResume()
        view?.viewTreeObserver?.addOnGlobalFocusChangeListener(this)
    }

    private fun showSubList(position: Int) {
        @Suppress("UNCHECKED_CAST")
        val sublist = arguments?.getSerializable(BUNDLE_KEY_SUBDATA) as? ArrayList<ListPickerItem>
        val fragment = newInstance(
            title = items[position].title,
            defaultPosition = DEFAULT_POSITION_TOP,
            data = sublist)
        Timber.tag("ListPickerFragment.showSubList").d("commit $fragment")
        FirebaseCrashlytics.getInstance().log("ListPickerFragment.showSubList: commit $fragment")
        activity?.supportFragmentManager?.beginTransaction()
            ?.replace(R.id.content_frame, fragment)
            ?.addToBackStack(null)
            ?.commit()
    }

    companion object {
        const val BUNDLE_KEY_TITLE = "BUNDLE_KEY_TITLE"
        const val BUNDLE_KEY_DATA = "BUNDLE_KEY_DATA"
        const val BUNDLE_KEY_SUBDATA = "BUNDLE_KEY_SUBDATA"
        const val BUNDLE_KEY_RESULT = "BUNDLE_KEY_RESULT"
        const val BUNDLE_HAS_SUBDATA = "BUNDLE_HAS_SUBDATA"
        const val REQUEST_KEY = "ListPickerFragmentRequest"
        const val BUNDLE_KEY_DEFAULT_SELECTED_ITEM_INDEX = "BUNDLE_KEY_DEFAULT_SELECTED_ITEM_INDEX"
        const val BUNDLE_KEY_SELECTED_POSITION = "BUNDLE_KEY_SELECTED_POSITION"
        const val BUNDLE_KEY_CLOSE_AFTER_SELECTION = "BUNDLE_KEY_CLOSE_AFTER_SELECTION"
        const val DEFAULT_POSITION_TOP = 0
        const val DEFAULT_SELECTED_ITEM_INDEX_MIDDLE = -1
        const val SELECTED_POSITION_MIDDLE = -1

        fun newInstance(
            title: String,
            defaultPosition: Int,
            data: ArrayList<ListPickerItem>?,
            hasSubData: Boolean = false,
            subData: ArrayList<ListPickerItem>? = null,
            selectedPosition: Int = DEFAULT_POSITION_TOP,
            closeAfterSelection: Boolean = true
        ) = ListPickerFragment().apply {
            putString(BUNDLE_KEY_TITLE, title)
            putInt(BUNDLE_KEY_DEFAULT_SELECTED_ITEM_INDEX, defaultPosition)
            putBoolean(BUNDLE_HAS_SUBDATA, hasSubData)
            putSerializable(BUNDLE_KEY_DATA, data)
            putSerializable(BUNDLE_KEY_SUBDATA, subData)
            putInt(BUNDLE_KEY_SELECTED_POSITION, selectedPosition)
            putBoolean(BUNDLE_KEY_CLOSE_AFTER_SELECTION, closeAfterSelection)
        }
    }

    override fun onGlobalFocusChanged(oldFocus: View?, newFocus: View?) {
        if(newFocus is SettingsItemView) {
            val currentItem = newFocus.binding.viewModel?.position?.let {
                (binding.listPickerRecycler.adapter as? ListPickerAdapter)?.currentList?.get(it)
            }
            if(currentItem != lastFocusedItem) {
                val title = if(currentItem?.title.isNullOrEmpty()) currentItem?.label else currentItem?.title
                Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Side menu ${newFocus.binding.viewModel?.position} focused: $title")
            }
            lastFocusedItem = currentItem
        }
    }
}