package com.twentyfouri.tvlauncher.ui

import android.Manifest
import android.annotation.SuppressLint
import android.app.NotificationManager
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.ActivityInfo
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.database.sqlite.SQLiteException
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.Environment
import android.os.Handler
import android.os.PowerManager
import android.provider.Settings
import android.view.KeyEvent
import android.view.View
import android.view.animation.Animation
import android.view.animation.AnimationUtils
import android.widget.Toast
import androidx.core.app.ActivityCompat
import androidx.core.content.getSystemService
import androidx.core.os.bundleOf
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.DialogFragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.Observer
import androidx.lifecycle.lifecycleScope
import com.google.firebase.analytics.FirebaseAnalytics
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.google.firebase.remoteconfig.FirebaseRemoteConfig
import com.twentyfouri.androidcore.epg.EpgView
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.epg.DATABASE_NAME
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.dashboard.SmartPageReference
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.tvlauncher.BuildConfig
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.SmartApiRepository
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.receiver.SerialNumberReceiver
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.ui.messagedialog.CANCEL
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogAction
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogCodes
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogDismissListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.common.ui.messagedialog.OPTION_A
import com.twentyfouri.tvlauncher.common.utils.*
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState.State
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_KEY_EVENT_LOG
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_LOG
import com.twentyfouri.tvlauncher.data.AppListRepository
import com.twentyfouri.tvlauncher.data.ContentResolverRepository
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.HardcodedPageReference
import com.twentyfouri.tvlauncher.data.NotificationListenerServiceTVLauncher
import com.twentyfouri.tvlauncher.databinding.ActivityMainBinding
import com.twentyfouri.tvlauncher.di.repositoryModule
import com.twentyfouri.tvlauncher.extensions.beGone
import com.twentyfouri.tvlauncher.extensions.beInVisible
import com.twentyfouri.tvlauncher.extensions.beVisible
import com.twentyfouri.tvlauncher.extensions.initOSEL
import com.twentyfouri.tvlauncher.extensions.isGone
import com.twentyfouri.tvlauncher.extensions.isVisible
import com.twentyfouri.tvlauncher.homepagechannels.TvUtil
import com.twentyfouri.tvlauncher.receiver.GlobalButtonReceiver
import com.twentyfouri.tvlauncher.receiver.PowerOffReceiver
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import com.twentyfouri.tvlauncher.receiver.TimeChangeReceiver
import com.twentyfouri.tvlauncher.ui.actions.ActivityEpgAction
import com.twentyfouri.tvlauncher.ui.actions.ActivityPlayerAction
import com.twentyfouri.tvlauncher.utils.ConnectionMessagesHelper
import com.twentyfouri.tvlauncher.utils.FirebaseRemoteConfigHelper
import com.twentyfouri.tvlauncher.utils.IntentLauncher
import com.twentyfouri.tvlauncher.utils.Navigator
import com.twentyfouri.tvlauncher.utils.RemoteControlState
import com.twentyfouri.tvlauncher.utils.RemoteKeyManager
import com.twentyfouri.tvlauncher.utils.internalNavigate
import com.twentyfouri.tvlauncher.utils.internalNavigateCustom
import com.twentyfouri.tvlauncher.utils.isEpgFragmentDisplayedOverPlayer
import com.twentyfouri.tvlauncher.utils.setSelectedState
import com.twentyfouri.tvlauncher.utils.shouldCallOnBackPressed
import com.twentyfouri.tvlauncher.viewmodels.MainActivityViewModel
import com.twentyfouri.tvlauncher.widgets.RowItemView
import io.github.inflationx.calligraphy3.CalligraphyConfig
import io.github.inflationx.calligraphy3.CalligraphyInterceptor
import io.github.inflationx.viewpump.ViewPump
import io.github.inflationx.viewpump.ViewPumpContextWrapper
import kotlinx.coroutines.*
import org.koin.android.ext.android.get
import org.koin.android.ext.android.getKoin
import org.koin.android.ext.android.inject
import org.koin.core.component.getScopeId
import org.koin.core.context.loadKoinModules
import org.koin.core.context.unloadKoinModules
import java.io.BufferedReader
import java.io.File
import java.io.FileReader
import java.lang.IllegalArgumentException
import kotlin.collections.ArrayList
import timber.log.Timber

interface BackPressListener {
    fun onBackPressed(topbarFocused: Boolean): BackPressAction
}

enum class OnCreateState {
    LOGGED_IN,
    LOGGED_IN_WITH_LIMITATIONS,
    RESTART,
    OFFLINE_AND_SETUP_WIZARD,
    OFFLINE_AND_NETWORK_ERROR,
    OFFLINE_AND_LOGIN_ERROR,
    NOT_ALLOWED,
    OFFLINE_AND_TOO_MANY_REQUESTS_ERROR
}

enum class BackPressAction {
    NOTHING,
    POP_BACK_STACK,
    RETURN
}

enum class OnResumeAction {
    NOTHING,
    GOTO_APPS,
    GOTO_HOME,
    GOTO_EPG,
    DEEP_LINK,
    RESTART
}

class MainActivity : BaseActivity(), Navigator.NavigatorContract, ActivityPlayerAction, MainActivityAction,
    SerialNumberReceiverHelper, ActivityEpgAction, EpgView.EpgFocusedEventStore {

    companion object {
        const val REQUEST_CODE_LOGIN_AND_RESTART = 1
        const val REQUEST_CODE_SETTINGS = 2
        const val RESULT_CODE_PARENTAL_CONTROL_CHANGED = 3
        const val REQUEST_CODE_SETTINGS_RESTART = 4
        const val REQUEST_CODE_FAVORITE_APP_SELECTION = 5
        const val EXTRA_FORCE_RESTART = "EXTRA_FORCE_RESTART"
        const val EXTRA_OPEN_USER_PROFILE = "EXTRA_OPEN_USER_PROFILE"
        @Suppress("SpellCheckingInspection")
        const val SAVED_TOPBAR = "SAVED_TOPBAR"
        const val SAVED_PLAYER = "SAVED_PLAYER"
        private const val PERMISSION_READ_STATE = 24
        private const val REQUEST_INSTALL_PERMISSION = 25
        private const val PERMISSION_STORAGE_READ = 26
        private const val PERMISSION_READ_TV_LISTINGS = 27
        private const val INTENT_ACTION_APPS = "android.intent.action.ALL_APPS"
        const val INTENT_ACTION_MAIN = "android.intent.action.MAIN"
        internal const val INTENT_CATEGORY_HOME = "android.intent.category.HOME"
        const val INTENT_GLOBAL_BUTTON = "android.intent.action.GLOBAL_BUTTON"
        const val INTENT_ACTION_EPG = "com.twentyfouri.tvlauncher.EPG"
        const val INTENT_ACTION_DETAIL = "com.twentyfouri.tvlauncher.DETAIL"
        const val ACTION_APP_LINK_KEY = "com.twentyfouri.tvlauncher.APP_LINK_KEY"
        private const val CONNECTIVITY_CHECK_DELAY: Long = 8000
        private const val RESTART_COUNTER_LIMIT = 1
        @Suppress("unused")
        val TAG: String = MainActivity::class.java.simpleName
        const val TRANSACTION_BREADCRUMB_BUTTONS_HIDE = "TRANSACTION_BREADCRUMB_BUTTONS_HIDE"
        //for upload following file use "adb push firebase.dev /sdcard/"
        //into file put only simple text like name, rather do not use spaces etc. it will be used as a key
        private const val FIREBASE_FILE_NAME = "firebase.dev"
    }

    enum class AppState { ONLINE, OFFLINE, UNINITIALIZED }

    private val firebaseRemoteConfig = FirebaseRemoteConfig.getInstance()
    private var currentMode: AppState = AppState.UNINITIALIZED
    private var onCreateState = OnCreateState.NOT_ALLOWED
    private lateinit var playerFragment: PlayerFragment
    internal lateinit var playerAction: PlayerAction
    internal var lastPlayerFragmentLocation = PlayerLocation.FOREGROUND
    lateinit var remoteKeyManager: RemoteKeyManager
    private val backPressListeners = ArrayList<BackPressListener>()
    @Suppress("SpellCheckingInspection")
    private var topbarFragment: TopbarFragment? = null
    private val timeChangeReceiver = TimeChangeReceiver()
    private val powerOffReceiver = PowerOffReceiver()
    //private val hdmiStatusReceiver = HDMIStatusReceiver() //TODO: uncomment or remove upon QA/customer decision - PRJ1210ENT-2122
    private val globalButtonReceiver = GlobalButtonReceiver(this)
    private var deepLinkingUri: Uri? = null
    private var onResumeAction = OnResumeAction.NOTHING
    private var globalSavedInstanceState: Bundle? = null
    private lateinit var configuration: Configuration
    private var screenOnOffReceiver: ScreenOnOffReceiver? = null
    private var serialNumberReceiver: SerialNumberReceiver? = null

    private var storedEpgChannelPosition: Int = 0
    private var storedEpgEvent: EpgEvent? = null
    private var storedEpgChannelPositionBackupForReturnFromEPG: Int = 0
    private var storedEpgEventBackupForReturnFromEPG: EpgEvent? = null

    private val handler = Handler()
    private val showProgressRunnable = Runnable { binding.viewModel?.showProgress(true) }

    internal val onScreenLogger = OnScreenExtendedLogger(this)

    private val checkConnectivityAgainWithDelay = Runnable {
        // fix of Fatal Exception: java.lang.IllegalStateException Can not perform this action after onSaveInstanceState
        if (lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED)) {
            setProperConnectionMode()
        }
    }

    private var wasEpgPressed = false
    private var shouldSwitchToUserProfile = false

    private var networkQualityCheckTimer: NetworkQualityCheckTimer? = null

    fun getEpgPressed(): Boolean {
        val toReturn = wasEpgPressed
        wasEpgPressed = false
        return toReturn
    }
    
    //this is necessary for custom fonts
    override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(ViewPumpContextWrapper.wrap(newBase))
    }

    override fun storeEventAndChannelPosition(event: EpgEvent?, channelPosition: Int) {
        storedEpgChannelPosition = channelPosition
        storedEpgEvent = event
    }

    override fun getStoredChannelPosition() = storedEpgChannelPosition
    override fun getStoredEvent() = storedEpgEvent

    fun refreshPlayerAfterSubscriptionBought() {
        remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
        playerAction.hideSubscriptions()
        playerAction.retryPlay()
    }

    fun setRemoteControlStateAfterSeekToLive() {
        remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
    }

    internal fun getTagExtFromPageReference(pageReference: SmartPageReference?): String {
        return when (pageReference) {
            is HardcodedPageReference -> pageReference.type.name
            null -> ""
            else -> pageReference.toString()
        }
    }

    override fun handleEpgButtonPress(i: Intent, remoteControlState: String) {
        wasEpgPressed = true
        val handlingEpg = getEpgFragment()?.isVisible == true && Flavor().shouldEpgCloseEpg
        if (handlingEpg && (remoteControlState == RemoteControlState.PLAYER_BACKGROUND.name || remoteControlState == RemoteControlState.PLAYER_FOREGROUND.name)) {
            // handling epg in player
            togglePlayerFrontBack()
            getEpgFragment()?.forceFocusStoredEvent()
            removeSidePanel()
        } else if (handlingEpg && remoteControlState != RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION.name) {
            // closing epg (going to previous screen) when not in player or subscription screen
            super.onBackPressed() //we want to remove fragment, just onBackPressed is doing different things
        } else {
            // opening epg from anywhere but EPG

            if(i.action == INTENT_ACTION_EPG) {
                when {
                    getEpgFragment()?.isVisible != true ||
                    remoteControlState == RemoteControlState.PLAYER_FOREGROUND.name -> {
                        startActivity(i)
                    }
                    getEpgFragment()?.isVisible == true && lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED).not() -> {
                        //try to get PersonalSettingsActivity and finish it in case
                        val personalSettingsActivity: PersonalSettingsActivity? = with(getKoin()) {
                            try {
                                getScope(getProperty(PersonalSettingsActivity.PERSONAL_SETTINGS_ACTIVITY_SCOPE_ID)!!).get()
                            } catch (t: Throwable) {
                                Timber.w(t)
                                null
                            }
                        }
                        personalSettingsActivity?.finish()
                    }
                }
            }
        }
    }

    override fun handleLiveButtonPress() {
        if (getEpgFragment()?.isVisible == true && Flavor().shouldLiveFocusLive)
            getEpgFragment()?.scrollToLive()
        else {
            if (Flavor().shouldLiveOpenPlayer) {
                openLastPlayedChannel()
            }
        }
    }

    override fun reportApiException(exception: java.lang.Exception) {
       YouboraAnalytics.getInstance(this).reportApiException(exception)
    }

    //this one is from ActivityPlayerAction
    override fun playerPlay(
        location: PlayerLocation,
        mediaReference: SmartMediaReference,
        isSoftZap: Boolean
    ) {
        internalNavigatePlayer(location, mediaReference, false, isSoftZap)
    }

    private fun internalNavigatePlayer(
        location: PlayerLocation,
        reference: SmartMediaReference,
        isStartOver: Boolean,
        isSoftZap: Boolean = false,
        isInitialNavigationToPlayer: Boolean = false,
        playerPosition: Int? = null
    ) {
        FirebaseAnalytics.getInstance(this).logEvent(FirebaseAnalytics.Event.SCREEN_VIEW, bundleOf(
            FirebaseAnalytics.Param.SCREEN_NAME to "Player",
            FirebaseAnalytics.Param.SCREEN_CLASS to "PlayerFragment"
        ))
        when (location) {
            PlayerLocation.UNCHANGED -> { /*do not change player location*/
            }
            else -> lastPlayerFragmentLocation = location
        }

        playerAction.playFromReference(
                reference = reference,
                lastPlayerFragmentLocation = lastPlayerFragmentLocation,
                isStartOver = isStartOver,
                isSoftZap = isSoftZap,
                isInitialPlayback = isInitialNavigationToPlayer,
                position = playerPosition
        )
        binding.playerContainer.beVisible()
        binding.playerInBackgroundShadowOverlay.beVisible() //it is Z positioned to cover only if player is in background

        if (lastPlayerFragmentLocation == PlayerLocation.FOREGROUND) {
            (supportFragmentManager.findFragmentById(R.id.content_frame) as? RowPageFragment)?.lastFocusedViewBeforeSubscriptions =
                currentFocus
        }

        when (lastPlayerFragmentLocation) {
            PlayerLocation.FOREGROUND -> remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            else -> remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_BACKGROUND)
        }

        when (lastPlayerFragmentLocation) {
            PlayerLocation.BACKGROUND -> binding.playerContainer.translationZ =
                resources.getDimension(R.dimen.z_order_background)
            PlayerLocation.FOREGROUND,
            PlayerLocation.PIP -> binding.playerContainer.translationZ =
                resources.getDimension(R.dimen.z_order_top_foreground)
            else -> {
            }
        }
        supportFragmentManager.fragments.forEachIndexed { index, fragment ->
            (fragment as? RowPageFragment)?.also {
                Timber.tag("fragments").d("iteration $index ${fragment.isAdded}")
                try {
                    it.onPlayerStart()
                } catch (e: IllegalStateException) {
                    // it very rarely happens that the "java.lang.IllegalStateException: Fragment already added: RowPageFragment"
                    // is thrown. At the moment we don't know why, just that it is coming from here
                    Timber.tag("fragments").w(e, "transaction IllegalStateException catched and ignored")
                    FirebaseCrashlytics.getInstance().recordException(e)
                }
            }
        }
    }

    override fun onBackPressed() {
        FirebaseAnalytics.getInstance(this).logEvent("back", null)
        var backPressAction = BackPressAction.NOTHING
        loop@ for (it in backPressListeners.reversed()) {
            backPressAction = it.onBackPressed(binding.topbarContainer.hasFocus())
            when (backPressAction) {
                BackPressAction.RETURN -> return
                BackPressAction.POP_BACK_STACK -> break@loop
                BackPressAction.NOTHING -> {
                }
            }
        }
        // when pop back stack was requested from inner fragment, then ignore topbar showing logic
        if (backPressAction != BackPressAction.POP_BACK_STACK && !binding.topbarContainer.hasFocus()) {
            showTopBar()
            if (onResumeAction != OnResumeAction.GOTO_EPG) binding.topbarContainer.requestFocus(View.FOCUS_RIGHT)
            return
        }

        if (backPressAction == BackPressAction.POP_BACK_STACK) {
            Timber.tag("fragments").d("pop back stack")
            supportFragmentManager.backStackEntryCount
                    .let { count -> if (count > 0) count-1 else null }
                    ?.let { index -> supportFragmentManager.getBackStackEntryAt(index) }
                    ?.let { backStackEntry -> backStackEntry.breadCrumbShortTitle }
                    ?.also { if (it == TRANSACTION_BREADCRUMB_BUTTONS_HIDE) binding.viewModel?.showDebugButtons(true) }
            if (isEpgFragmentDisplayedOverPlayer() && playerAction.isSideMenuShown().not()) {
                togglePlayerFrontBack()
            }
            requestLastFocusedView()
        }

        if (shouldCallOnBackPressed()) {
            //HOME and APPS cannot be closed by Back
            super.onBackPressed()
        }

        if (shouldSwitchToUserProfile && getDetailFragment() == null && !isForegroundPlaybackActive() ){
            shouldSwitchToUserProfile = false
            navigate(SmartNavigationTarget.toUserRecordings())
        }
    }

    override fun backFromSubscriptionDialog() {
        PurchaseChecker.dismissPinDialog()
        playerStop()
    }

    override fun playerStop(onlyInForeground: Boolean) {
        remoteKeyManager.isInRemoteControlState(RemoteControlState.OFFLINE_MODE).ifTrue { return }
        if ((!onlyInForeground || lastPlayerFragmentLocation == PlayerLocation.FOREGROUND)) {
            if(getBindingSafe()?.playerContainer?.isVisible() != false){
                for(fragment in supportFragmentManager.fragments) {
                    (fragment as? RowPageFragment)?.onPlayerStop()
                }
            }
            playerAction.stop()
            playerAction.resetStreamTimers()
            getBindingSafe()?.playerContainer?.beGone()
            getBindingSafe()?.playerInBackgroundShadowOverlay?.beGone()
            remoteKeyManager.setRemoteControlState(RemoteControlState.NORMAL)
            if (getEpgFragment()?.isVisible == true && Flavor().displayPlayerBehindEPG) {
                getEpgFragment()?.forceFocusStoredEvent()
            }
        }
    }

    override fun checkRemoteStateAfterReturnFromSubscriptions() {
        if(remoteKeyManager.isInRemoteControlState(RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION)){
            remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
        }
    }

    override fun requestFocusAfterReturningFromSubscriptions() {
        (supportFragmentManager.findFragmentById(R.id.content_frame) as BaseFragment).lastFocusedViewBeforeSubscriptions?.requestFocus()
    }

    override fun requestLastFocusedView() {
        (supportFragmentManager.findFragmentById(R.id.content_frame) as BaseFragment).resumeFocus()
    }

    override fun togglePlayerFrontBack() {
        if (binding.playerContainer.visibility == View.VISIBLE) {
            if (lastPlayerFragmentLocation == PlayerLocation.FOREGROUND) {
                lastPlayerFragmentLocation = PlayerLocation.BACKGROUND
                playerAction.cancelSeeking()
                remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_BACKGROUND)
                binding.playerContainer.translationZ = resources.getDimension(R.dimen.z_order_background)
                getEpgFragment()?.onHiddenChanged(false)
                playerAction.hideControls()
                storedEpgChannelPositionBackupForReturnFromEPG = storedEpgChannelPosition
                storedEpgEventBackupForReturnFromEPG = storedEpgEvent
            } else {
                lastPlayerFragmentLocation = PlayerLocation.FOREGROUND
                remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
                binding.playerContainer.translationZ = resources.getDimension(R.dimen.z_order_foreground)
                getEpgFragment()?.onHiddenChanged(true)
                playerAction.showControls()
                storeEventAndChannelPosition(storedEpgEventBackupForReturnFromEPG, storedEpgChannelPositionBackupForReturnFromEPG)
            }
        }
    }

    override fun provideLastPlayerLocation(): PlayerLocation {
        return lastPlayerFragmentLocation
    }

    private var nullableBinding: ActivityMainBinding? = null

    internal val binding: ActivityMainBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")

    fun getBindingSafe() = nullableBinding

    override fun onCreate(savedInstanceState: Bundle?) {
        getKoin().setProperty(ApiHandler.MAIN_ACTIVITY_SCOPE_ID, getScopeId())
        if (Settings.Secure.getInt(contentResolver, "user_setup_complete", 0) != 1) {
            // launcher cannot be started before google login
            finish()
        }

        Thread.setDefaultUncaughtExceptionHandler(CustomExceptionHandler())
        setTheme(R.style.AppTheme)
        super.onCreate(savedInstanceState)
        configuration = resources.configuration
        globalSavedInstanceState = savedInstanceState

        // Reload Koin repositoryModule. New copies of the single dependencies will be created after restart
        unloadKoinModules(repositoryModule)
        loadKoinModules(repositoryModule)

        //Attempt to cap maximum size of EpgDatabase: PRJ1210ENT-3045
        clearEpgDatabaseIfOverMaximumSize()

        // OSEL tool remote logging, enable/disable if needed
        get<OselToggleableLogger>().toggleLogging(
            SharedPreferencesUtils.getRemoteLoggingEnabled(),
            isAutomaticCall = true
        )

        val intentFilter = IntentFilter().also {
            it.addAction(Intent.ACTION_SCREEN_ON)
            it.addAction(Intent.ACTION_SCREEN_OFF)
        }
        screenOnOffReceiver = ScreenOnOffReceiver(this)
        registerReceiver(screenOnOffReceiver, intentFilter)

        val powerOffReceiverIntentFilter = IntentFilter().apply {
            addAction(PowerOffReceiver.SHUTDOWN_INTENT)
        }
        registerReceiver(powerOffReceiver, powerOffReceiverIntentFilter)

        NetworkConnectionState.instance.register(
            context = this,
            fallback = { restartActivity() },
            networkApprove = {
                if (Flavor().shouldCheckNetworkConnectionQuality) Flavor().checkNetworkConnectionQuality(this)
            }
        )

        if(Flavor().shouldCheckNetworkConnectionQuality) networkQualityCheckTimer = NetworkQualityCheckTimer(this)

        //This permission is necessary to get channels from TvProvider
        if (Flavor().allowDisplayOfExternalRows()) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf("android.permission.READ_TV_LISTINGS"),
                PERMISSION_READ_TV_LISTINGS
            )
        }

        nullableBinding = DataBindingUtil.setContentView(this, R.layout.activity_main)
        //getting S/N from custom SDMC broadcast
        if (com.twentyfouri.tvlauncher.common.Flavor().getSerialNumberFromBroadcast()) {
            if(DeviceId.isSerialNumberStored(this)) {
                continueOnCreate1()
            } else {
                val serialNumberIntentFilter = IntentFilter()
                serialNumberIntentFilter.addAction(SerialNumberReceiver.INTENT_FILTER_ACTION)
                registerReceiver(SerialNumberReceiver(this), serialNumberIntentFilter)
                onSerialNumberReceiverPending()
            }
        } else {
            continueOnCreate1()
        }

        if (com.twentyfouri.tvlauncher.common.Flavor().isYouboraEnabled) {
            NetworkConnectionState.instance.stateLD.observe(this) {
                YouboraAnalytics.getInstance(this).resolveConnectionState(it)
            }
        }
    }

    private fun continueOnCreate1() {
        checkAndSetUpPersonalFirebaseDevID()

        playerFragment = globalSavedInstanceState?.let {
            supportFragmentManager.getFragment(
                    it,
                    SAVED_PLAYER
            ) as? PlayerFragment
        } ?: PlayerFragment()
        playerAction = playerFragment
        remoteKeyManager = RemoteKeyManager(playerAction, this, this, this)

        @Suppress("ConstantConditionIf")
        if (BuildConfig.IS_APP) {
            if (checkSelfPermission(Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
                requestPermissions(
                        arrayOf(Manifest.permission.READ_PHONE_STATE),
                        PERMISSION_READ_STATE
                )
            }
            waitForConnection()
            TvUtil.scheduleSyncingChannel(this)
        } else {
            waitForConnection()
        }

        /* Don't remove. This snippet is commented in relation to: PRJ1210ENT-341.
           TODO This should be executed only in an alternative build type with automatic installation enabled for search packages and setup wizard.
           We should ensure that the app can install the packages automatically to accomplish that we check if the app has permissions.

        if (BuildConfig.IS_DEBUG && Flavor().getSearchableDelegate() != null) {
            packageManager.canRequestPackageInstalls().ifFalse {
                val intent = Intent(Settings.ACTION_MANAGE_UNKNOWN_APP_SOURCES, Uri.parse("package:$packageName"))
                ActivityCompat.startActivityForResult(this, intent, REQUEST_INSTALL_PERMISSION, null)
            }
        } */
    }

    private fun verifyNotificationPermission(): Boolean =
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O_MR1) {
            getSystemService<NotificationManager>()?.isNotificationListenerAccessGranted(ComponentName(this, NotificationListenerServiceTVLauncher::class.java)) ?: false
        } else {
            Settings.Secure.getString(contentResolver, "enabled_notification_listeners").contains(
                NotificationListenerServiceTVLauncher::class.java.simpleName
            )
        }

    //continue with creation only when the S/N is received
    override fun onSerialNumberReceived(context: Context?) {
        try {
            get<ApiHandler>().joinPreviousOrLaunchNew(
            block = {
                val dialogFragment = supportFragmentManager.findFragmentByTag("pending_serial_dialog")
                dialogFragment?.let { (it as? DialogFragment)?.dismiss() }
            },
                synchronizedJobName = "pending_serial_dialog"
            )
            context?.unregisterReceiver(serialNumberReceiver)
        } catch (e: IllegalArgumentException) {
            e.printStackTrace()
        }
        serialNumberReceiver = null
        continueOnCreate1()
    }

    @SuppressLint("MissingPermission")
    override fun onSerialNumberReceiverError() {
        get<ApiHandler>().joinPreviousOrLaunchNew(
            block = {
                withContext(Dispatchers.Main) {
                    val serialNumberPendingDialog =
                        supportFragmentManager.findFragmentByTag("pending_serial_dialog")
                    serialNumberPendingDialog?.let { (it as? DialogFragment)?.dismiss() }
                    val messageDialogModel = MessageDialogModel(
                        getString(R.string.serial_number_receiver_error),
                        null,
                        getString(R.string.serial_number_receiver_reboot_button),
                        MessageDialogCodes.serialNumberReceiverException
                    )
                    val serialNumberErrorDialog =
                        MessageDialogFragment.newInstance(messageDialogModel)
                    serialNumberErrorDialog.mListener = MessageDialogFragmentListener.from {
                        (applicationContext.getSystemService(Context.POWER_SERVICE) as? PowerManager)?.reboot(null)
                        true
                    }
                    serialNumberErrorDialog.show(supportFragmentManager, "serial_dialog_error")
                    supportFragmentManager.executePendingTransactions()
                }
            },
            synchronizedJobName = "serial_dialog_error"
        )
    }

    @SuppressLint("MissingPermission")
    private fun onSerialNumberReceiverPending() {
        val messageDialogModel = MessageDialogModel(
            getString(R.string.serial_number_receiver_pending),
            null,
            getString(R.string.serial_number_receiver_reboot_button),
            MessageDialogCodes.serialNumberReceiverPending
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = MessageDialogFragmentListener.from {
            (applicationContext.getSystemService(Context.POWER_SERVICE) as? PowerManager)?.reboot(null)
            true
        }
        get<ApiHandler>().joinPreviousOrLaunchNew(
            block = {
                withContext(Dispatchers.Main) {
                    dialogFragment.show(supportFragmentManager, "pending_serial_dialog")
                    supportFragmentManager.executePendingTransactions()
                }
            },
            synchronizedJobName = "pending_serial_dialog"
        )
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        topbarFragment?.let { topBarFragment ->
            supportFragmentManager.putFragment(outState, SAVED_TOPBAR, topBarFragment)
        }
        ::playerFragment.isInitialized.ifTrue {
            if (playerFragment.isAdded) supportFragmentManager.putFragment(
                outState,
                SAVED_PLAYER,
                playerFragment
            )
        }
    }

    private fun waitForConnection() {
        // - if ONLINE then probe was successfully done and we can continue
        // - if NO_INTERNET or OFFLINE then connecting to WiFi may still running so we will wait
        //binding intentionally moved here but viewModel not, so there is waiting screen but viewModel is not created yet (with wrong conn state)
        NetworkConnectionState.instance.waitForConnection(this) {
            lifecycleScope.launch(Dispatchers.Default) {
                com.twentyfouri.tvlauncher.common.Flavor().preStartActions(
                    get<FlowSmartApi>(),
                    { // Activity can be destroyed during preStartActions. Lifecycle condition prevents continue OnCreate in such case.
                        if (lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED)) continueOnCreate2()
                    },
                    { exception ->
                        //TODO: do this properly with di
                        YouboraAnalytics.getInstance(this@MainActivity).reportApiException(
                            e = exception,
                            isOnnetService = true)
                    }
                )
            }
        }
    }

    private fun continueOnCreate2() {
        initCalligraphy()

        val globalButtonIntentFilter = IntentFilter().apply {
            addAction(INTENT_GLOBAL_BUTTON)
            addAction(ACTION_APP_LINK_KEY)
        }
        registerReceiver(globalButtonReceiver, globalButtonIntentFilter)

        /*val hdmiStatusIntentFilter = IntentFilter().apply {
            addAction(HDMIStatusReceiver.HDMI_INTENT)
            addAction(HDMIStatusReceiver.VOLUME_CHANGE_INTENT)
        }
        registerReceiver(hdmiStatusReceiver, hdmiStatusIntentFilter)*/ //TODO: uncomment or remove upon QA/customer decision - PRJ1210ENT-2122

        registerReceiver(timeChangeReceiver, IntentFilter(Intent.ACTION_TIME_TICK))
        get<AppListRepository>().registerAsReceiverOn(this)
        get<ContentResolverRepository>().registerNotificationReceiverOn(this)

        //MainActivityViewModel is here intentionally because we have to wait for connectivity
        //so that this viewModel is created with non-null smart api (proper online connection state)
        val viewModelFromFactory: MainActivityViewModel by inject()
        binding.apply {
            lifecycleOwner = this@MainActivity
            viewModel = viewModelFromFactory
        }

        if (SharedPreferencesUtils.getForceOffline()) {
            SharedPreferencesUtils.putForceOffline(false)
            loadFragments_offline(
                globalSavedInstanceState,
                false,
                showConnectivityDialog = true,
                exceptionMessage = binding.viewModel?.getApiExceptionMessage()
            )
        } else {
            binding.viewModel?.getOnCreateStateLD()?.observe(this, Observer {
                onCreateState = it
                Timber.tag("ONCS").d(it.toString())
                when (it) {
                    OnCreateState.LOGGED_IN -> {
                        if (Flavor().shouldCheckNetworkConnectionQuality) Flavor().checkNetworkConnectionQuality(this)
                        FirebaseRemoteConfigHelper.fetchAndActivate(
                            doOnSuccess = { checkForUpdate() },
                            doOnFailure = { exception ->
                                Timber.e(exception.stackTraceToString())
                            }
                        )
                        loadFragments(globalSavedInstanceState)
                    }
                    OnCreateState.LOGGED_IN_WITH_LIMITATIONS -> {
                        if (Flavor().shouldCheckNetworkConnectionQuality) Flavor().checkNetworkConnectionQuality(this)
                        FirebaseRemoteConfigHelper.fetchAndActivate(
                            doOnSuccess = { checkForUpdate() },
                            doOnFailure = { exception ->
                                Timber.e(exception.stackTraceToString())
                            }
                        )
                        TvLauncherToast.makeText(
                            this,
                            R.string.app_running_with_limitations,
                            Toast.LENGTH_SHORT
                        )?.show()
                        loadFragments(globalSavedInstanceState) // TODO: we don't know what limitations are yet
                    }
                    OnCreateState.RESTART, null -> {
                        Timber.tag("Restart").i("MainActivity.onCreateState == RESTART")
                        restartActivity()
                    }
                    OnCreateState.OFFLINE_AND_SETUP_WIZARD -> loadFragments_offline(
                        globalSavedInstanceState,
                        showSetupWizardAfter = true,
                        showConnectivityDialog = true,
                        exceptionMessage = null
                    )
                    OnCreateState.OFFLINE_AND_LOGIN_ERROR -> loadFragments_offline(
                        globalSavedInstanceState,
                        showSetupWizardAfter = false,
                        showConnectivityDialog = false,
                        exceptionMessage = null
                    )

                    OnCreateState.OFFLINE_AND_NETWORK_ERROR -> loadFragments_offline(
                        globalSavedInstanceState,
                        false,
                        showConnectivityDialog = true,
                        exceptionMessage = binding.viewModel?.getApiExceptionMessage()
                    )

                    OnCreateState.OFFLINE_AND_TOO_MANY_REQUESTS_ERROR -> loadFragments_offline(
                        globalSavedInstanceState,
                        showSetupWizardAfter = false,
                        showConnectivityDialog = false,
                        showTooManyRequestsDialog = true,
                        exceptionMessage = null
                    )

                    OnCreateState.NOT_ALLOWED -> {
                        remoteKeyManager.setRemoteControlState(RemoteControlState.APP_NOT_ALLOWED)
                    }
                }
            })
        }

        NetworkConnectionState.instance.stateLD.observe(this, Observer {
            // We react depending on launcher internet connection.
            //Toast.makeText(this, it.toString(), Toast.LENGTH_SHORT).show()
            if (onCreateState != OnCreateState.OFFLINE_AND_SETUP_WIZARD
                && onCreateState != OnCreateState.OFFLINE_AND_LOGIN_ERROR
                && onCreateState != OnCreateState.NOT_ALLOWED
            ) {
                checkConnectivityAgainWithDelay()
            }

            if (currentMode != AppState.ONLINE) {
                when (it) {
                    State.OFFLINE -> hideOfflineRow(true)
                    else -> hideOfflineRow(false)
                }
            } else {
                if (it == State.ONLINE && isForegroundPlaybackActive() && ScreenOnOffReceiver.screenWasOff.not()) {
                    // In case of standby, stream retry is handled inside PlayerFragment.
                    playerAction.retryPlay(true)
                    remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
                    binding.offlinePlayerCover.beGone()
                }
            }
        })

        binding.playerContainer.beInVisible()
        binding.playerInBackgroundShadowOverlay.beInVisible()

        initOSEL()

        // uncomment to observe focus changes
//        binding.root.viewTreeObserver.addOnGlobalFocusChangeListener { old, new ->
//            Log.v("FOCUS", "global focus changed old = $old")
//            Log.v("FOCUS", "global focus changed new = $new")
//        }

    }

    @SuppressLint("MissingSuperCall")
    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        when (requestCode) {
            PERMISSION_READ_STATE -> {
                if (grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    // permission granted
                    waitForConnection()
                }
            }
            PERMISSION_STORAGE_READ -> {
                if (grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    setUpPersonalFirebaseDevID()
                }
            }
            PERMISSION_READ_TV_LISTINGS -> {}
            else -> super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        }
    }

    override fun onNewIntent(intent: Intent?) {
        Timber.tag("Restart").i("MainActivity.onNewIntent")
        ::remoteKeyManager.isInitialized.ifTrue {
            if (remoteKeyManager.isInRemoteControlState(RemoteControlState.PLAYER_OFFLINE_MODE)
                || remoteKeyManager.isInRemoteControlState(RemoteControlState.APP_NOT_ALLOWED)) {
                restartActivity()
                return
            }
        }
        super.onNewIntent(intent)
        when {
            intent?.getBooleanExtra(EXTRA_FORCE_RESTART, false) == true -> {
                // restart requested, finish and start new
                Timber.tag("Restart").i("MainActivity.onNewIntent EXTRA_FORCE_RESTART")
                restartActivity()
            }
            INTENT_ACTION_DETAIL == intent?.action -> {
                if (intent.getBooleanExtra(EXTRA_OPEN_USER_PROFILE, false)) {
                    // request to reopen user profile
                    shouldSwitchToUserProfile = true
                }
                val ref = intent.getSerializableExtra("reference") as? SmartMediaReference
                ref?.let { navigate(SmartNavigationTarget.toDetailPage(it)) }
            }
            Intent.ACTION_VIEW == intent?.action -> {
                this.deepLinkingUri = intent.data
                onResumeAction = OnResumeAction.DEEP_LINK
                intent.data = null
                intent.action = null
            }
            INTENT_ACTION_APPS == intent?.action -> {
                //jump to apps after resume
                RestrictionChecker.dismissRestrictionDialog()
                onResumeAction = OnResumeAction.GOTO_APPS
                Timber.tag(TAG_KEY_EVENT_LOG).d(getString(R.string.tab_apps))
            }
            INTENT_ACTION_MAIN == intent?.action && intent.categories.contains(INTENT_CATEGORY_HOME) -> {
                //jump home after resume
                Timber.tag("Restart").i("MainActivity.onNewIntent INTENT_ACTION_MAIN")
                RestrictionChecker.dismissRestrictionDialog()
                onResumeAction = OnResumeAction.GOTO_HOME
                Timber.tag(TAG_KEY_EVENT_LOG).d(getString(R.string.tab_home))
            }
            INTENT_ACTION_EPG == intent?.action -> {
                //jump to epg after resume
                RestrictionChecker.dismissRestrictionDialog()
                onResumeAction = OnResumeAction.GOTO_EPG
            }
        }
    }

    override fun onStop() {
        //These need to be unregistered before onDestroy because of Koin.
        get<AppListRepository>().unregisterAsReceiverOn(this)
        get<ContentResolverRepository>().unregisterNotificationReceiverOn(this)
        super.onStop()
    }

    override fun onRestart() {
        super.onRestart()
        //In case that user switch back to Launcher after MainActivity is stopped.
        get<AppListRepository>().registerAsReceiverOn(this)
        get<ContentResolverRepository>().let {
            it.registerNotificationReceiverOn(this)
            it.updateNotificationLD()
        }
    }

    override fun onDestroy() {
        Timber.d("onDestroy")
        FirebaseCrashlytics.getInstance().log("MainActivity: onDestroy")
        YouboraAnalytics.getInstance(this).endSession()

        // in some cases these receivers are not registered and we want to avoid crashing because of that
        // and if one of the unregistering fails we still want to unregister the others
        try { unregisterReceiver(globalButtonReceiver) } catch (e: IllegalArgumentException) { e.printStackTrace() }
        try { unregisterReceiver(screenOnOffReceiver) } catch (e: IllegalArgumentException) { e.printStackTrace() }
        try { unregisterReceiver(powerOffReceiver) } catch (e: IllegalArgumentException) { e.printStackTrace() }
        //try { unregisterReceiver(hdmiStatusReceiver) } catch (e: IllegalArgumentException) { e.printStackTrace() } //TODO: uncomment or remove upon QA/customer decision - PRJ1210ENT-2122
        try { unregisterReceiver(timeChangeReceiver) } catch (e: IllegalArgumentException) { e.printStackTrace() }
        YouboraAnalytics.removeInstance()
        try {
            NetworkConnectionState.instance.unregister(this)
        } catch (e: Exception){
            e.printStackTrace()
        }
        provideNQC()?.cancel()
        nullableBinding = null
        super.onDestroy()
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        val diff = configuration.diff(newConfig)
        configuration = newConfig
        Timber.d("onConfigurationChanged $diff")
        if (diff and ActivityInfo.CONFIG_KEYBOARD_HIDDEN != 0) Timber.d("onConfigurationChanged CONFIG_KEYBOARD_HIDDEN")
        if (diff and ActivityInfo.CONFIG_LAYOUT_DIRECTION != 0) Timber.d("onConfigurationChanged CONFIG_LAYOUT_DIRECTION")
        if (diff and ActivityInfo.CONFIG_LOCALE != 0) Timber.d("onConfigurationChanged CONFIG_LOCALE")
        super.onConfigurationChanged(newConfig)
        // We're only catching locale changes in this activity

        if (diff and ActivityInfo.CONFIG_LAYOUT_DIRECTION != 0 || diff and ActivityInfo.CONFIG_LOCALE != 0 ) {
            if (lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED)) {
                // If we're in the foreground, restart directly
                restartActivity()
            } else {
                /*
                 * If we're in the background, calling restartActivity will interrupt the user
                 * with a new activity on top.
                 * Instead, make sure we restart when the user returns.
                 */
                onResumeAction = OnResumeAction.RESTART
            }
        }
    }

    private fun loadFragments(savedInstanceState: Bundle?) {
        if (savedInstanceState == null) {
            val topBarFragment = TopbarFragment.newInstance().also { this.topbarFragment = it }

            val tagExt = getTagExtFromPageReference(getPageReference(PageType.MAIN))
            supportFragmentManager.beginTransaction().apply {
                replace(R.id.topbar_container, topBarFragment)
                val rowPageFragment = RowPageFragment.newInstance(getPageReference(PageType.MAIN))
                //yes it can happen - crashlytics fix
                rowPageFragment.isAdded.ifFalse {
                    Timber.tag("fragments").d("add RowPageFragment in online mode")
                    add(R.id.content_frame, rowPageFragment, tagExt)
                }
                playerFragment.isAdded.ifFalse { add(R.id.player_container, playerFragment) }
                Timber.tag("MainActivity.loadFragments").d("commitNow")
                FirebaseCrashlytics.getInstance().log("MainActivity.loadFragments: commitNow")
                commitNow()
            }
            if (TimeChangeReceiver.autoTimeSettings == 0) {
                showTimeIncorrectWithDialog()
            }
        } else {
            topbarFragment = supportFragmentManager.getFragment(
                savedInstanceState,
                SAVED_TOPBAR
            ) as? TopbarFragment
        }
        this.currentMode = AppState.ONLINE

        //uncomment this line if you want to hide splash screen by force
        //binding.viewModel?.splashScreenVisibility?.postValue(View.GONE)

        restartPlayerOnChannel()
    }

    internal fun setOfflineMode() {
        if (currentMode == AppState.ONLINE || currentMode == AppState.UNINITIALIZED) {
            // We restart the activity. The activity will start in offline AppState.
            if (playerAction.isPlayingOrShouldResume() || playerAction.isInErrorState() || isForegroundPlaybackActive()) {
                remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_OFFLINE_MODE)
                binding.offlinePlayerCover.beVisible()
                playerAction.pause(false)
                playerAction.cancelSeeking()
                playerAction.cancelSoftZap()
                playerAction.hideSideMenu()
                playerAction.hideSubscriptions()
            } else {
                val counter = SharedPreferencesUtils.getRestartCounter()
                SharedPreferencesUtils.putForceOffline(counter >= RESTART_COUNTER_LIMIT)
                SharedPreferencesUtils.putRestartCounter(counter + 1)
                this.restartActivity()
            }
        }
    }

    private fun setOnlineMode() {
        val counter = SharedPreferencesUtils.getRestartCounter()
        if (currentMode == AppState.OFFLINE && counter <= RESTART_COUNTER_LIMIT) {
            SharedPreferencesUtils.putRestartCounter(counter + 1)
            SharedPreferencesUtils.putForceOffline(false)
            this.restartActivity()
        }
    }

    private fun checkConnectivityAgainWithDelay() {
        handler.postDelayed(checkConnectivityAgainWithDelay, CONNECTIVITY_CHECK_DELAY)
    }

    private fun setProperConnectionMode() {
        when (currentMode) {
            AppState.ONLINE -> {
                when (NetworkConnectionState.instance.state) {
                    State.OFFLINE -> this.setOfflineMode()
                    State.NO_INTERNET -> TvLauncherToast.makeText(
                        this,
                        R.string.no_internet_error,
                        Toast.LENGTH_SHORT
                    )?.show()
                    State.ONLINE -> {
                    }//code replaced by code immediately after connection available
                }
            }
            AppState.OFFLINE -> {
                when (NetworkConnectionState.instance.state) {
                    State.ONLINE -> {
                        this.setOnlineMode()
                    }
                    State.OFFLINE,
                    State.NO_INTERNET -> {
                    }
                }
            }
            else -> {
            }
        }
    }

    @Suppress("FunctionName")
    private fun loadFragments_offline(
        savedInstanceState: Bundle?,
        showSetupWizardAfter: Boolean = false,
        showConnectivityDialog: Boolean = true,
        showTooManyRequestsDialog: Boolean = false,
        exceptionMessage: Exception?
    ) {
        if (savedInstanceState == null) {
            val topBarFragment = TopbarFragment.newInstance(true).also { this.topbarFragment = it }
            val transaction = supportFragmentManager.beginTransaction()
                .replace(R.id.topbar_container, topBarFragment)
            val tagExt = getTagExtFromPageReference(getPageReference(PageType.MAIN))
            val rowPageFragment = RowPageFragment.newInstance(getPageReference(PageType.APPS))
            //yes it can happen - crashlytics fix
            rowPageFragment.isAdded.ifFalse {
                Timber.tag("fragments").d("add RowPageFragment in offline mode")
                showSetupWizardAfter
                    .ifTrue {
                        transaction.replace(
                            R.id.content_frame,
                            rowPageFragment,
                            tagExt
                        )
                    }
                    .ifElse {
                        transaction.add(
                            R.id.content_frame,
                            rowPageFragment,
                            tagExt
                        )
                    }
            }
            Timber.tag("MainActivity.loadFragments_offline").d("commitNow")
            FirebaseCrashlytics.getInstance().log("MainActivity.loadFragments_offline: commitNow")
            transaction.commitNow()
            showSetupWizardAfter
                .ifTrue { navigate(SmartNavigationTarget.toLogin()) }
                .ifElse {
                    showConnectivityDialog.ifTrue {
                        showConnectivityWithDialog(SmartApiRepository.getError(this)
                            ?: exceptionMessage?.let {
                                ErrorStringMapper.getErrorString(it, this)
                            })
                    }
                }
            showTooManyRequestsDialog
                    .ifTrue { showTooManyRequestsExceptionWithDialog() }
        } else {
            topbarFragment = supportFragmentManager.getFragment(
                savedInstanceState,
                SAVED_TOPBAR
            ) as? TopbarFragment
        }
        remoteKeyManager.setRemoteControlState(RemoteControlState.OFFLINE_MODE)
        this.currentMode = AppState.OFFLINE
    }

    private fun showConnectivityWithDialog(message: String?) {
        val messageDialogModel = MessageDialogModel(
                message ?: getString(R.string.connectivity_issues_dialog_message),
                null,
                getString(R.string.connectivity_issues_settings_button),
                MessageDialogCodes.connectivityIssues
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener =
            MessageDialogFragmentListener.from(this::onNoConnectivityDialogResult)
        dialogFragment.mDismissListener =
            MessageDialogDismissListener.from(this::onNoConnectivityDialogDismiss)
        dialogFragment.show(supportFragmentManager, null)
    }

    private fun showTooManyRequestsExceptionWithDialog() {
        val messageDialogModel = MessageDialogModel(
                resources.getString(R.string.too_many_requests_message),
                null,
                arrayOf(resources.getString(R.string.dialog_error_retry)),
                resources.getString(R.string.cancel),
                MessageDialogCodes.loginFailure
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when(answer.type) {
                    OPTION_A -> { restartActivity() }
                    CANCEL -> { return false } //cancel, do nothing
                    else -> { return false }
                }
                return false
            }
        }
        dialogFragment.isCancelable = true
        navigateDialog(dialogFragment)
    }

    private fun showUpdateAvailableDialog() {
        val messageDialogModel = MessageDialogModel(
                getString(R.string.update_available_message),
                null,
                arrayOf(getString(R.string.update_now)),
                getString(R.string.cancel),
                MessageDialogCodes.updateRequired
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                if (answer.type == OPTION_A) {
                    val intent = Intent(Intent.ACTION_VIEW).apply { addFlags(Intent.FLAG_ACTIVITY_NEW_TASK) }
                    intent.data = Uri.parse(getString(R.string.app_play_url, BuildConfig.APPLICATION_ID))
                    applicationContext.startActivity(intent)
                }
                return false
            }
        }
        dialogFragment.isCancelable = false
        dialogFragment.show(supportFragmentManager, null)
    }

    private fun onNoConnectivityDialogDismiss() {
        this.hideSplashScreen(0)
    }

    private fun onNoConnectivityDialogResult(@Suppress("UNUSED_PARAMETER") answer: MessageDialogAction): Boolean {
        startActivityForResult(
            Intent(Settings.ACTION_SETTINGS),
            REQUEST_CODE_SETTINGS
        )
        return true
    }

    private fun showTimeIncorrectWithDialog() {
        val messageDialogModel = MessageDialogModel(
                getString(R.string.time_settings_issues_dialog_message),
                null,
                getString(R.string.time_settings_issues_settings_button),
                MessageDialogCodes.timeInconsistency
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener =
            MessageDialogFragmentListener.from(this::onTimeIncorrectDialogResult)
        dialogFragment.show(supportFragmentManager, null)
    }

    private fun onTimeIncorrectDialogResult(@Suppress("UNUSED_PARAMETER") answer: MessageDialogAction): Boolean {
        startActivityForResult(
            Intent(Settings.ACTION_SETTINGS),
            REQUEST_CODE_SETTINGS
        )
        return true
    }

    override fun resolveDialogBackOptionPress() {
        if (getDetailFragment()?.isVisible == true){
            closeDetailFragment()
            return
        }
        playerStop(false)
    }

    override fun isInOnlineMode(): Boolean {
        return currentMode == AppState.ONLINE
    }

    override fun provideNQC(): NetworkQualityCheckTimer? {
        return networkQualityCheckTimer
    }

    override fun onResume() {
        super.onResume()
        //hdmiStatusReceiver.appResumed = true //TODO: uncomment or remove upon QA/customer decision - PRJ1210ENT-2122
        if (SharedPreferencesUtils.getForceRestart()) {
            SharedPreferencesUtils.putForceRestart(false)
            onResumeAction = OnResumeAction.RESTART
        }
        if (onResumeAction == OnResumeAction.RESTART) {
            restartActivity()
            return
        }

        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = this, skipWaiting = !ScreenOnOffReceiver.screenWasOff) {
            if (isFinishing) return@waitForConnection
            CoroutineScope(Dispatchers.IO).launch {
                YouboraAnalytics.getInstance(this@MainActivity).resolveResumeState()
            }
            Navigator.getInstance().setContract(this)
            removeErrorDialogs()
            if (currentMode == AppState.ONLINE) {
                //key action is handled here because onNewIntent pause activity
                when (onResumeAction) {
                    OnResumeAction.GOTO_HOME -> {
                        gotoHome()
                    }
                    OnResumeAction.GOTO_APPS -> {
                        closeDetailFragment()
                        playerStop(false) //stop always when going to apps
                        if (binding.viewModel != null) {
                            Navigator.getInstance()
                                .navigate(
                                    SmartNavigationTarget.toBrowsePage(
                                        getPageReference(PageType.APPS)
                                    )
                                )
                        }
                    }
                    OnResumeAction.GOTO_EPG -> {
                        if (Flavor().displayPlayerBehindEPG
                                && playerAction.isPlayingOrShouldResume()
                                && remoteKeyManager.isInRemoteControlState(RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION).not()
                        ) {
                            when (lastPlayerFragmentLocation) {
                                PlayerLocation.FOREGROUND -> {
                                    togglePlayerFrontBack()
                                    closeDetailFragment()
                                    showTopBar()
                                    Navigator.getInstance().navigate(SmartNavigationTarget.to(SmartNavigationAction.LIVE_EPG_FULL))
                                    storedEpgChannelPositionBackupForReturnFromEPG =
                                            storedEpgChannelPosition
                                    storedEpgEventBackupForReturnFromEPG = storedEpgEvent
                                }
                                PlayerLocation.BACKGROUND -> {
                                    togglePlayerFrontBack()
                                    closeDetailFragment()
                                    showTopBar()
                                    storeEventAndChannelPosition(
                                            storedEpgEventBackupForReturnFromEPG,
                                            storedEpgChannelPositionBackupForReturnFromEPG
                                    )
                                }
                                else -> gotoEpg()
                            }
                        } else gotoEpg()
                    }
                    OnResumeAction.DEEP_LINK -> {
                        deepLinkingUri?.let { uri ->
                            try {
                                Flavor().getSmartMediaReferenceFromUri(uri)?.let { reference ->
                                    this.navigateToSmartMediaReference(reference)
                                }
                            } catch (e: Exception) {
                                e.printStackTrace()
                                gotoHome()
                            }
                        }
                    }
                    else -> {
                    }
                }
            }
            onResumeAction = OnResumeAction.NOTHING
            checkMissingNotificationPermission()
            ScreenOnOffReceiver.screenWasOff = false
            if (onCreateState == OnCreateState.LOGGED_IN || onCreateState == OnCreateState.LOGGED_IN_WITH_LIMITATIONS){
                // Check network quality in case that network has changed
                if (Flavor().shouldCheckNetworkConnectionQuality) Flavor().checkNetworkConnectionQuality(this)
            }
        }
        //sometimes the MainActivity is resumed in such weird way that the usual callbacks to hide splash screen are not called at all, so this is fallback
        hideSplashScreen()
    }

    var notificationPermissionChecked = false
    private fun checkMissingNotificationPermission() {
        if (isFinishing) return
        if (notificationPermissionChecked) return
        notificationPermissionChecked = true
        if (!verifyNotificationPermission() && Flavor().displayNotificationAccessWarning) {
            val messageDialogModel = MessageDialogModel(
                    getString(R.string.notification_error_title),
                    getString(R.string.notification_error_message) + " " +getString(R.string.app_name),
                    getString(R.string.notification_error_confirm),
                    MessageDialogCodes.missingNotificationsPermission
            )
            val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
            navigateDialog(dialogFragment)
        }

    }

    private fun gotoEpg() {
        playerStop(false)
        closeDetailFragment()
        showTopBar()
        Navigator.getInstance().navigate(SmartNavigationTarget.to(SmartNavigationAction.LIVE_EPG_FULL))
    }

    private fun closeDetailFragment() {
        //there should be maximally 2 so maybe is safer to limit this while somehow
        try {
            while (supportFragmentManager.findFragmentByTag(DetailFragment::class.java.simpleName) != null) {
                supportFragmentManager.popBackStackImmediate()
            }
        } catch (e:java.lang.IllegalStateException) {
            //sometimes popBackStackImmediate() causes java.lang.IllegalStateException: Fragment already added: RowPageFragment
            Timber.e(e.stackTraceToString())
        }
    }

    private fun gotoHome() {
        playerStop(false)  //stop always when going home
        if (binding.viewModel != null) {
            Navigator.getInstance()
                .navigate(
                    SmartNavigationTarget.toBrowsePage(
                        getPageReference(PageType.MAIN)
                    )
                )
        }
    }

    private fun removeErrorDialogs() {
        for (fragment in supportFragmentManager.fragments) {
            if (fragment is MessageDialogFragment && fragment.isCancelable) {
                /*fix attempt for
                    * java.lang.IllegalStateException: Can not perform this action after onSaveInstanceState
                    * at com.twentyfouri.tvlauncher.ui.MainActivity.removeErrorDialogs(MainActivity.kt)
                    * using commitAllowingStateLoss() */
                // TODO: Remove if no affect is achieved
                Timber.tag("MainActivity.loadFragments_offline")
                    .d("commitAllowingStateLoss $fragment")
                FirebaseCrashlytics.getInstance()
                    .log("MainActivity.loadFragments_offline: commitAllowingStateLoss $fragment")
                supportFragmentManager.beginTransaction().remove(fragment).commitAllowingStateLoss()
            }
        }
    }

    private fun navigateToSmartMediaReference(reference: SmartMediaReference) {
        val smartMediaDetail = Flavor().convertMediaReferenceToDetail(reference)

        when (smartMediaDetail.type) {
            SmartMediaType.LIVE_CHANNEL -> {
                val target = SmartNavigationTarget.toPlayer(reference, null)
                target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                Navigator.getInstance().navigate(target)
            }
            else -> {
                Navigator.getInstance().navigate(SmartNavigationTarget.toDetailPage(reference))
            }
        }
    }

    internal fun getPageReference(pageType: PageType) = binding.viewModel?.getPageReference(pageType)
        ?: throw IllegalStateException("View model not bound")

    override fun onPause() {
        //hdmiStatusReceiver.appResumed = false //TODO: uncomment or remove upon QA/customer decision - PRJ1210ENT-2122
        Navigator.getInstance().setContract(null)
        YouboraAnalytics.getInstance(this@MainActivity).stopDeviceInfoReporting()
        super.onPause()
    }

    @Suppress("ConstantConditionIf")
    override fun dispatchKeyEvent(event: KeyEvent?): Boolean {
        val keyInfo = keyInfoString(event)
        Timber.tag(TAG_KEY_EVENT_LOG).d("RECEIVED $keyInfo EVENT: $event")
        onScreenLogger.keyEvent(event)
        checkKeyForRestart(event)
        if (isPageLoading()) {
            return true
        }
        event?.let {
            return if (remoteKeyManager.handleRemoteKey(event)) {
                Timber.tag(TAG_KEY_EVENT_LOG).d("HANDLED $keyInfo EVENT: $event")
                true
            } else {
                Timber.tag(TAG_KEY_EVENT_LOG).d("PASSED $keyInfo EVENT: $event")
                super.dispatchKeyEvent(event)
            }
        } ?: run {
            return super.dispatchKeyEvent(event)
        }
    }

    private fun keyInfoString(event: KeyEvent?): String {
        return event?.let {
            val action = when (it.action) {
                KeyEvent.ACTION_UP -> "↑"
                KeyEvent.ACTION_DOWN -> "↓"
                else -> ""
            }
            "$action ${KeyEvent.keyCodeToString(it.keyCode)} ${it.keyCode} ${it.scanCode}"
        } ?: "null"
    }

    private var backKeyDownTime = 0L
    private fun checkKeyForRestart(event: KeyEvent?) {
        if(!Flavor().shouldBackLongPressRestartLauncher) return
        if (event?.keyCode == KeyEvent.KEYCODE_BACK) {
            when (event.action) {
                KeyEvent.ACTION_DOWN -> if (backKeyDownTime == 0L) backKeyDownTime = TimeProvider.nowMs()
                KeyEvent.ACTION_UP -> {
                    if (TimeProvider.nowMs() - backKeyDownTime in Flavor().backLongPressTimeInterval) restartActivity()
                    backKeyDownTime = 0L
                }
                else -> backKeyDownTime = 0L
            }
        } else backKeyDownTime = 0L
    }

    private fun initCalligraphy() {
        ViewPump.init(
            ViewPump.builder()
                .addInterceptor(
                    CalligraphyInterceptor(
                        CalligraphyConfig.Builder()
                            .setDefaultFontPath("fonts/OpenSans-Semibold.ttf")
                            .setFontAttrId(R.attr.fontPath)
                            .build()
                    )
                )
                .build()
        )
    }

    fun getTopBar() = topbarFragment

    fun hideTopBar() {
        if (binding.topbarContainer.isVisible()) animateTopBarOut()
    }

    fun showTopBar() {
        //When hiding animation is ongoing we need to cancel it and start show animation.
        //Can happen when on first row user click down (hide top bar) and quickly click up.
        //Opposite situation seems not causing issues so not handled.
        with (binding.topbarContainer) {
            if (isGone()) {
                animateTopBarIn()
            } else if (animation?.hasEnded() == false && isVisible()) {
                animation?.cancel()
                animateTopBarIn()
            }
        }
    }

    private fun animateTopBarIn() {
        animateTopBar(
            R.anim.topbar_anim_in,
            View.VISIBLE,
            View.VISIBLE
        )
    }

    private fun animateTopBarOut() {
        animateTopBar(
            R.anim.topbar_anim_out,
            View.VISIBLE,
            View.GONE
        )
    }

    private fun animateTopBar(animationId: Int, startVisibility: Int, endVisibility: Int) {
        val anim = AnimationUtils.loadAnimation(this, animationId)
        anim.setAnimationListener(object : Animation.AnimationListener {
            override fun onAnimationRepeat(animation: Animation?) {}
            override fun onAnimationEnd(animation: Animation?) {
                binding.topbarContainer.visibility = endVisibility
            }

            override fun onAnimationStart(animation: Animation?) {
                binding.topbarContainer.visibility = startVisibility
            }
        })
        // `Handler().post` is attempt to speed up animation a little -> might be removed in future if nop benefit can be experienced
        //Handler().post {
        binding.topbarContainer.startAnimation(anim)
        anim.fillAfter = true
        //}
    }

    override fun checkAndMaybeRestartActivity() {
        if (NetworkConnectionState.instance.state == State.OFFLINE
            && !isForegroundPlaybackActive()
            && !remoteKeyManager.isInRemoteControlState(RemoteControlState.PLAYER_OFFLINE_MODE)
        ) {
            restartActivity()
        }
    }

    override fun isForegroundPlaybackActive(): Boolean {
        if (this.isFinishing) return false
        return getBindingSafe() != null
                && getBindingSafe()?.playerContainer?.visibility == View.VISIBLE
                && lastPlayerFragmentLocation == PlayerLocation.FOREGROUND
    }

    override fun isLauncher(): Boolean = BuildConfig.IS_APP.not()

    override fun findCurrentFocus(): View? = currentFocus

    override fun stopPlayerInStandby() {
        playerAction.stopPlayerInStandby(isForegroundPlaybackActive())
    }

    private fun checkForUpdate() {
        if (Flavor().shouldCheckIfUpdateAvailable) {
            try {
                val code = firebaseRemoteConfig.getString("google_play_version_code")
                if (code.isNotEmpty() && code.toInt() > BuildConfig.VERSION_CODE) {
                    GlobalScope.launch(Dispatchers.Main) {
                        showUpdateAvailableDialog()
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    private fun checkAndSetUpPersonalFirebaseDevID() {
        val sdcard: File = Environment.getExternalStorageDirectory()
        val file = File(sdcard, FIREBASE_FILE_NAME)
        if (file.exists()) { //if file do not exist then do not bother with read permission and ignore
            if (checkSelfPermission(Manifest.permission.READ_EXTERNAL_STORAGE) == PackageManager.PERMISSION_GRANTED) {
                setUpPersonalFirebaseDevID()
            } else {
                ActivityCompat.requestPermissions(this, arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE), PERMISSION_STORAGE_READ)
            }
        }
        val stringKey = resources.getString(R.string.firebase_dev_id)
        if(!stringKey.isNullOrEmpty()) { //if string is not set then ignore
            FirebaseCrashlytics.getInstance().setCustomKey("S_$stringKey", true)
        }
    }

    private fun setUpPersonalFirebaseDevID() {
        val sdcard: File = Environment.getExternalStorageDirectory()
        val file = File(sdcard, FIREBASE_FILE_NAME)
        try {
            val reader = BufferedReader(FileReader(file))
            val fileKey = reader.readText()
            reader.close()
            FirebaseCrashlytics.getInstance().setCustomKey("F_$fileKey", true)
        } catch (e: Exception) {
            //Ignore any exception
        }
    }

    private fun clearEpgDatabaseIfOverMaximumSize() {
        val youboraAnalytics = YouboraAnalytics.getInstance(this@MainActivity)
        try {
            val epgDatabaseSizeMB = applicationContext.getDatabasePath(DATABASE_NAME).length() / (1024L * 1024L)
            val maxAllowedEpgDatabaseSize = com.twentyfouri.tvlauncher.common.Flavor().epgDatabaseMaximumSizeMB
            if (maxAllowedEpgDatabaseSize in 1 until epgDatabaseSizeMB) {
                Timber.i("EpgDatabase size: ${epgDatabaseSizeMB}MB is over maximum allowed ${maxAllowedEpgDatabaseSize}MB")
                GlobalScope.launch(Dispatchers.Default) {
                    try {
                        com.twentyfouri.tvlauncher.common.Flavor().clearEpgData(this@MainActivity, get<FlowSmartApi>())
                    } catch (e: SQLiteException){
                        Timber.w(e.stackTraceToString())
                        youboraAnalytics.reportEpgDatabaseCleared(success = false)
                    }
                }
                youboraAnalytics.reportEpgDatabaseCleared(success = true)
            }
        } catch (e: Exception) {
            Timber.e("Failed to check EpgDatabase size limitation")
            e.printStackTrace()
            youboraAnalytics.reportEpgDatabaseCleared(success = false)
        }
    }

    //we need to postpone restart activity in case of wake up from standby otherwise launcher could jump into foreground
    //and cover active application
    override fun planRestartActivity() {
        if(lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED)) restartActivity()
        else onResumeAction = OnResumeAction.RESTART
    }

    override fun restartActivity() {
        Timber.tag("Restart").i("MainActivity.restartActivity")
        viewModelStore.clear()
        finish()
        startActivity(intent)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (resultCode == RESULT_CODE_PARENTAL_CONTROL_CHANGED)
            restartActivity()
        when (requestCode) {
            REQUEST_CODE_FAVORITE_APP_SELECTION -> {
                if (resultCode == FavoriteAppsActivity.RESULT_REQUEST_MOVE){
                    // setSelectedState(true) is needed to move app item through app row
                    val currentFocusedView = findCurrentFocus()
                    if (currentFocusedView is RowItemView) {
                        currentFocusedView.setSelectedState(true)
                    }
                }
            }
            REQUEST_CODE_LOGIN_AND_RESTART -> {
                Timber.tag("Restart").i("MainActivity.onActivityResult REQUEST_CODE_LOGIN_AND_RESTART")
                restartActivity()
            }
            REQUEST_CODE_SETTINGS_RESTART -> {
                onResumeAction = OnResumeAction.RESTART
            }
            else -> super.onActivityResult(requestCode, resultCode, data)
        }
    }

    /**
     * Ads BackPressedListener into list of listeners called when onBackPressed() is called.
     */
    fun addBackPressedListener(backPressListener: BackPressListener) =
        backPressListeners.add(backPressListener)

    /**
     * Ads BackPressedListener into list of listeners called when onBackPressed() is called. Listener is placed
     * at first position, which gives it opportunity to react first to onBackPressed()
     */
    @Suppress("unused")
    fun addBackPressedListenerHighPriority(backPressListener: BackPressListener) =
        backPressListeners.add(0, backPressListener)

    fun removeBackPressedListener(backPressListener: BackPressListener) =
        backPressListeners.remove(backPressListener)

    private fun hideSplashScreen(
        delay: Long = resources.getInteger(R.integer.splash_screen_loading_offset).toLong()
    ) {
        handler.postDelayed({
            // This delay is for design purposes. It is a way to ensure that all images are loaded.
            if (!isDestroyed) {
                ConnectionMessagesHelper.checkConnectionAndShowMessage(applicationContext)
                binding.viewModel?.showSplash(false)
            }
        }, delay)
    }

    override fun restartPlayerOnChannel() {
        val channelNumber = SharedPreferencesUtils.getChannelNumber()
        Timber.tag(TAG_PLAYER_LOG).d("stored channel number $channelNumber")

        get<EpgRepository>().getAllChannelsLD().also { ld ->
            ld.observe(this, object : Observer<List<SmartMediaItem>> {
                override fun onChanged(list: List<SmartMediaItem>) {
                    val ref = list.find { it.channelNumber == channelNumber }
                    Timber.tag(TAG_PLAYER_LOG).d("playing reference $ref")
                    if (ref != null) playerPlay(PlayerLocation.FOREGROUND, ref.reference, false)
                    ld.removeObserver(this)
                }
            })
        }
    }

    private fun openLastPlayedChannel() {
        if (!isInOnlineMode()) {
            Timber.tag(TAG_PLAYER_LOG).d("open last played channel blocked because not in ONLINE mode")
            return
        }

        val channelNumber = SharedPreferencesUtils.getLastChannelNumber()
        Timber.tag(TAG_PLAYER_LOG).d("last channel number $channelNumber")

        get<EpgRepository>().getAllChannelsLD().also { ld ->
            ld.observe(this, object : Observer<List<SmartMediaItem>> {
                override fun onChanged(list: List<SmartMediaItem>) {
                    val ref = list.find { it.channelNumber == channelNumber }
                    Timber.tag(TAG_PLAYER_LOG).d("playing reference $ref")
                    if (ref != null) playerPlay(PlayerLocation.FOREGROUND, ref.reference, false)
                    ld.removeObserver(this)
                }
            })
        }
    }

    override fun handleGlobalKeyButton(keyEvent: KeyEvent) {
        remoteKeyManager.handleRemoteKey(keyEvent, isGlobalKey = true)
    }

    override fun startStopRecording() {
        binding.viewModel?.recordingClicked()
    }

    override fun startStopRecordingInPlayer() {
        binding.viewModel?.recordingInPlayerClicked()
    }

    override fun showNumberZap(number: String) {
        //crashlytics fix - binding can be null here
        nullableBinding?.playerUiNumberZap?.setNumber(number)
    }

    override fun playNumberZapChanel(number: String, controlState: RemoteControlState) {
        if (lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED).not()) return
        if (getEpgFragment()?.isVisible == true
            && Flavor().shouldEpgHandleNumberButtons
            && controlState != RemoteControlState.PLAYER_FOREGROUND
            && controlState != RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION
        ) {
            scrollToChannelViaNumberButtons(number.toInt())
        } else {
            val reference = if (Flavor().shouldLookForPlayableChannel) {
                playerFragment.getClosestChannelByNumber(number)
            } else {
                playerFragment.getChannelByNumber(number)
            }

            if (reference != null && lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED)) {
                playerAction.hideSubscriptions()
                playerPlay(PlayerLocation.FOREGROUND, reference)
            }
        }
    }

    internal fun getEpgFragment(): EpgFragment? {
        val pageReference = try {
            getPageReference(PageType.EPG)
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
            null
        }
        return supportFragmentManager.findFragmentByTag(getTagExtFromPageReference(pageReference)) as? EpgFragment
    }

    internal fun getDetailFragment(): DetailFragment? {
        return supportFragmentManager.findFragmentByTag(DetailFragment::class.java.simpleName) as? DetailFragment
    }

    internal fun getListPickerFragment(): ListPickerFragment? {
        return supportFragmentManager.findFragmentByTag(ListPickerFragment::class.java.simpleName) as? ListPickerFragment
    }

    internal fun getPlayerFragment(): PlayerFragment? = playerFragment

    fun removeSidePanel() {
        getListPickerFragment()?.let {
            Timber.tag("MainActivity.removeSidePanel").d("commit $it")
            FirebaseCrashlytics.getInstance().log("MainActivity.removeSidePanel: commit $it")
            supportFragmentManager.beginTransaction().remove(it).commit()
        }
    }

    override fun scrollEPGPageUp() {
        getEpgFragment()?.scrollEPGPageUp()
    }

    override fun scrollEPGPageDown() {
        getEpgFragment()?.scrollEPGPageDown()
    }

    override fun scrollEPGDay(where: Int) {
        if (getEpgFragment()?.isVisible == true) {
            getEpgFragment()?.scrollEPGDay(where)
        }
    }

    override fun scrollToChannelViaNumberButtons(number: Int) {
        getEpgFragment()?.scrollToChannelViaNumberButtons(number)
    }

    override fun updateFocusedEventInfo() {
        getEpgFragment()?.forceFocusStoredEvent()
    }

    override fun openPlayer(startFromBeginning: Boolean):Boolean {
        if (getEpgFragment()?.isVisible == true){
            getEpgFragment()?.openPlayer(startFromBeginning)
            return true
        }
        if (getDetailFragment()?.isVisible == true){
            getDetailFragment()?.openPlayer(startFromBeginning)
            return true
        }
        return false
    }

    private fun hideOfflineRow(hide: Boolean) {
        for (fragment in supportFragmentManager.fragments) {
            (fragment as? RowPageFragment)?.hideOfflineRow(hide)
        }
    }

    private fun isPageLoading(): Boolean =
        binding?.viewModel?.splashScreenVisibility?.value == View.VISIBLE
                || binding?.viewModel?.progressLayoutVisibility?.value == View.VISIBLE

    //region NavigatorContract overrides

    override fun navigateDialog(fragment: DialogFragment) {
        lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).ifTrue {
            try {
                if (fragment is MessageDialogFragment && fragment.tagExt.isNotBlank()) {
                    supportFragmentManager.executePendingTransactions()
                    supportFragmentManager.findFragmentByTag(fragment.tagExt)
                        ?: fragment.show(supportFragmentManager, fragment.tagExt)
                    return
                }
                fragment.show(supportFragmentManager, null)
            } catch (e: Exception) {
                Timber.e("Failed to display dialog ${fragment}.\n${e.stackTraceToString()}")
            }
        }
    }

    override fun navigate(intent: Intent) {
        if(intent.action == Settings.ACTION_SETTINGS) {
            letEpgKnowToStayOnFocusedItem()
        }
        IntentLauncher.launchIntent(this, intent)
    }

    override fun navigateLoader(visible: Boolean, handleSplash: Boolean, delay: Long) {
        if (visible) {
            startProgress(delay)
            if (handleSplash) binding.viewModel?.showSplash(true)
        } else {
            stopProgress()
            if (handleSplash) hideSplashScreen(delay)
        }
    }

    override fun navigate(target: SmartNavigationTarget) {
        if (lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED)) {
            //in case opening "My account" settings let EPG know that it should stay on focused item instead moving focus to live
            when (target.action) {
                SmartNavigationAction.USER_PROFILE,
                SmartNavigationAction.USER_RECORDINGS,
                SmartNavigationAction.SETTINGS -> letEpgKnowToStayOnFocusedItem()
            }
            internalNavigate(target)
        }
    }

    private fun letEpgKnowToStayOnFocusedItem() {
        getEpgFragment()?.let {
            if (it.isVisible) it.stayOnItemAfterResume()
        }
    }

    //this one is from NavigatorContract
    override fun navigatePlayer(location: PlayerLocation, reference: SmartMediaReference, playerPosition: Int) {
        internalNavigatePlayer(
                location = location,
                reference = reference,
                isStartOver = playerPosition == 0,
                isInitialNavigationToPlayer = true,
                playerPosition = playerPosition
        )
    }

    override fun navigateCustom(custom: String) {
        internalNavigateCustom(custom)
    }

    private fun stopProgress() {
        handler.removeCallbacks(showProgressRunnable)
        if (nullableBinding != null) { //don't know how it can happen but once it was crashed when i pressed EPG button very early after home page was displayed
            binding.viewModel?.showProgress(false)
        }
    }

    private fun startProgress(delay: Long) {
        if (delay > 0) {
            handler.postDelayed(showProgressRunnable, delay)
        } else {
            showProgressRunnable.run()
        }
    }

    //endregion
}

//This class intercepts unhandled exceptions to save them and rethrows them after.
class CustomExceptionHandler() : Thread.UncaughtExceptionHandler {
    private val defaultUEH: Thread.UncaughtExceptionHandler? = Thread.getDefaultUncaughtExceptionHandler()
    override fun uncaughtException(t: Thread, e: Throwable) {
        SharedPreferencesUtils.putAppCrashData(e)
        SharedPreferencesUtils.putAppCrashStackTrace(e)
        defaultUEH?.uncaughtException(t, e)
    }
}