package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.util.Log
import android.view.KeyEvent
import android.view.KeyEvent.KEYCODE_DPAD_CENTER
import android.view.View
import android.widget.Toast
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.commit
import androidx.fragment.app.setFragmentResultListener
import androidx.lifecycle.Observer
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ApiMissingException
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogFragment
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogModel
import com.twentyfouri.tvlauncher.common.utils.SharedPreferencesUtils
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.data.RecordingOffset
import com.twentyfouri.tvlauncher.data.RecordingSettingsRepository
import com.twentyfouri.tvlauncher.databinding.FragmentPersonalSettingsBinding
import com.twentyfouri.tvlauncher.viewmodels.PersonalSettingsViewModel
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import org.koin.android.ext.android.get
import org.koin.android.ext.android.getKoin
import org.koin.androidx.viewmodel.ext.android.getViewModel
import org.koin.core.component.getScopeId
import timber.log.Timber

const val USER_RECORDINGS_CODE = 1
const val EXTRA_FRAGMENT_NAVIGATION_CODE = "fragment_navigation_code"

class PersonalSettingsActivity: BaseActivity() {

    lateinit var binding: FragmentPersonalSettingsBinding
    lateinit var listFragment: ListPickerFragment

    companion object {
        const val PERSONAL_SETTINGS_ACTIVITY_SCOPE_ID = "PERSONAL_SETTINGS_ACTIVITY_SCOPE_ID"
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        getKoin().setProperty(PERSONAL_SETTINGS_ACTIVITY_SCOPE_ID, getScopeId())

        try {
            binding = DataBindingUtil.setContentView<FragmentPersonalSettingsBinding>(
                this,
                R.layout.fragment_personal_settings
            )
                .apply {
                    viewModel = getViewModel<PersonalSettingsViewModel>()
                    lifecycleOwner = this@PersonalSettingsActivity
                }
        } catch (t: ApiMissingException) {
            finish()
            // TODO maybe force switch to offline?!
        }

        binding.viewModel?.apply {
            initCurrentProfileOnLifecycle(this@PersonalSettingsActivity)
            observeGdprAcceptedOnLifecycle(this@PersonalSettingsActivity, binding.personalSettingsGdprSwitch::setSwitch)
            observeParentalControlOnLifecycle(this@PersonalSettingsActivity, binding.personalSettingsParentalControlSwitch::setSwitch)
        }

        binding.personalSettingsRecordingsSection.setOnClickListener {
            moveToRecordings()
        }

        binding.personalSettingsGdprSwitch.setOnClickListener {
            val isSwitchOn = binding.viewModel?.gdprAccepted?.value
            binding.viewModel?.toggleGdprAcceptedOnCurrentProfileOnLifecycle(this, isSwitchOn)
        }

        binding.personalSettingsParentalControlSwitch.setOnClickListener {
            val messageDialogModel = PinDialogModel(
                getString(com.twentyfouri.tvlauncher.common.R.string.dialog_parental_pin_message, resources.getInteger(R.integer.pin_length)),
                "")
            val parentalPinDialogFragment = PinDialogFragment.newInstance(messageDialogModel)
            parentalPinDialogFragment.mListener = object :
                PinDialogFragmentListener {
                override fun onResult(answer: Boolean?, keyCode: Int?) {
                    if (answer == true) {
                        val isSwitchOn = binding.viewModel?.parentalControlsActivated?.value
                        binding.viewModel?.toggleParentalControlActivatedOnCurrentProfileOnLifecycle(this@PersonalSettingsActivity, isSwitchOn)
                        SharedPreferencesUtils.putForceRestart(true)
                    }
                }
            }
            supportFragmentManager.also { parentalPinDialogFragment.show(it,"tag_wrong_pin_dialog_fragment") }
        }

        binding.viewModel?.parentalRules?.observe(this@PersonalSettingsActivity, Observer{
            if (it != null) {
                TvLauncherToast.makeText(this@PersonalSettingsActivity, getString(R.string.parental_rules_updated), Toast.LENGTH_SHORT)?.show()
            }
        })

        binding.personalSettingsParentalRulesRefresh.setOnKeyListener { view: View, i: Int, keyEvent: KeyEvent ->
            if (keyEvent.keyCode == KEYCODE_DPAD_CENTER && keyEvent.action == KeyEvent.ACTION_UP) {
                binding.viewModel?.updateParentalRuleOnClick(this@PersonalSettingsActivity)
                true
            } else false
        }

        binding.personalSettingsRecording.setOnClickListener {
            listFragment = ListPickerFragment.newInstance(
                title = getString(R.string.personal_settings_recording_title),
                defaultPosition = ListPickerFragment.DEFAULT_POSITION_TOP,
                data = ArrayList<ListPickerItem>().apply {
                    add(ListPickerItem(
                        id = "0",
                        title = getString(R.string.personal_settings_recording_time_before),
                        subtitle = getString(get<RecordingSettingsRepository>().getStartBeforeStringRes()),
                        label = "",
                        type = SettingsItemType.TITLE_SUBTITLE))
                    add(ListPickerItem(
                        id = "1",
                        title = getString(R.string.personal_settings_recording_time_after),
                        subtitle = getString(get<RecordingSettingsRepository>().getStartAfterStringRes()),
                        label = "",
                        type = SettingsItemType.TITLE_SUBTITLE))
                },
                hasSubData = true,
                subData = ArrayList<ListPickerItem>().apply {
                    add(ListPickerItem(
                        id = RecordingOffset.ONE.minutes.toString(),
                        title = "",
                        subtitle = "",
                        label = getString(RecordingOffset.ONE.stringRes()),
                        type = SettingsItemType.LABEL))
                    add(ListPickerItem(
                        id = RecordingOffset.TWO.minutes.toString(),
                        title = "",
                        subtitle = "",
                        label = getString(RecordingOffset.TWO.stringRes()),
                        type = SettingsItemType.LABEL))
                    add(ListPickerItem(
                        id = RecordingOffset.FIVE.minutes.toString(),
                        title = "",
                        subtitle = "",
                        label = getString(RecordingOffset.FIVE.stringRes()),
                        type = SettingsItemType.LABEL))
                    add(ListPickerItem(
                        id = RecordingOffset.TEN.minutes.toString(),
                        title = "",
                        subtitle = "",
                        label = getString(RecordingOffset.TEN.stringRes()),
                        type = SettingsItemType.LABEL))
                    add(ListPickerItem(
                        id = RecordingOffset.FIFTEEN.minutes.toString(),
                        title = "",
                        subtitle = "",
                        label = getString(RecordingOffset.FIFTEEN.stringRes()),
                        type = SettingsItemType.LABEL))
                }
            )
            Timber.tag("PersonalSettingsActivity.onCreate").d("commit $listFragment")
            FirebaseCrashlytics.getInstance()
                .log("PersonalSettingsActivity.onCreate: commit $listFragment")
            supportFragmentManager.beginTransaction()
                .add(R.id.content_frame, listFragment)
                .addToBackStack(null)
                .commit()
            listFragment.setFragmentResultListener(ListPickerFragment.REQUEST_KEY) { _, bundle ->
                val item = bundle.get(ListPickerFragment.BUNDLE_KEY_RESULT) as? ListPickerItem
                val sectionTitle = bundle.get(ListPickerFragment.BUNDLE_KEY_TITLE) as? String
                if (item != null && sectionTitle != null) {
                    if (sectionTitle == getString(R.string.personal_settings_recording_time_before)) {
                        listFragment.items[0].subtitle = item.label
                        get<RecordingSettingsRepository>().setStartBefore(item.id)
                    } else {
                        listFragment.items[1].subtitle = item.label
                        get<RecordingSettingsRepository>().setStartAfter(item.id)
                    }
                }
            }
        }

        if (intent.getIntExtra(EXTRA_FRAGMENT_NAVIGATION_CODE, 0) == USER_RECORDINGS_CODE) {
            moveToRecordings()
        }
    }

    private fun moveToRecordings() {
        val fragment = RecordingsFragment()
        Timber.tag("PersonalSettingsActivity.moveToRecordings").d("commit $fragment")
        FirebaseCrashlytics.getInstance().log("PersonalSettingsActivity.moveToRecordings: commit $fragment")
        supportFragmentManager.commit {
            add(R.id.content_frame, fragment)
            addToBackStack(null)
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        binding.personalSettingsViewgroup.getChildAt(0).requestFocus()
    }

    override fun onBackPressed() {
        super.onBackPressed()
        overridePendingTransition(R.anim.slide_from_right,R.anim.slide_out_right)
    }
}