package com.twentyfouri.tvlauncher.ui

import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.os.CountDownTimer
import android.os.Handler
import android.os.Looper
import android.view.*
import android.widget.Toast
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.setFragmentResultListener
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.Observer
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.ExoPlaybackException
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.SeekParameters
import com.google.android.exoplayer2.audio.AudioAttributes
import com.google.android.exoplayer2.source.BehindLiveWindowException
import com.google.android.exoplayer2.source.TrackGroupArray
import com.google.android.exoplayer2.trackselection.TrackSelectionArray
import com.google.android.exoplayer2.video.VideoListener
import com.google.firebase.remoteconfig.FirebaseRemoteConfig
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartexoplayer.SmartPlayerFactory
import com.twentyfouri.smartexoplayer.model.DrmConfiguration
import com.twentyfouri.smartexoplayer.model.PlayerConfigurationModel
import com.twentyfouri.smartexoplayer.model.PlayerSourceModel
import com.twentyfouri.smartexoplayer.tracks.TrackInfo
import com.twentyfouri.smartexoplayer.tracks.TrackPreference
import com.twentyfouri.smartexoplayer.tracks.TrackPreferences
import com.twentyfouri.smartexoplayer.tracks.VideoPreference
import com.twentyfouri.smartexoplayer.ui.TimerHelper
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.media.SmartMediaStream
import com.twentyfouri.smartmodel.model.media.SmartPlayerEvent
import com.twentyfouri.smartmodel.model.media.SmartPlayerEventType
import com.twentyfouri.smartmodel.model.media.SmartSeekingRuleType
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.analytics.YouboraAdapter
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.StreamingType
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.ui.SemaphoreState
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.common.utils.*
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_ERROR_LOG
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_LOG
import com.twentyfouri.tvlauncher.databinding.FragmentPlayerBinding
import com.twentyfouri.tvlauncher.receiver.ScreenOnOffReceiver
import com.twentyfouri.tvlauncher.ui.actions.ActivityPlayerAction
import com.twentyfouri.tvlauncher.utils.*
import com.twentyfouri.tvlauncher.viewmodels.PlayerUILiveViewModel
import com.twentyfouri.tvlauncher.viewmodels.PlayerViewModel
import com.twentyfouri.tvlauncher.widgets.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import org.koin.android.ext.android.get
import org.koin.androidx.scope.ScopeFragment
import org.koin.androidx.viewmodel.ext.android.viewModel
import timber.log.Timber
import java.lang.IllegalArgumentException
import java.util.*
import kotlin.concurrent.fixedRateTimer
import kotlin.concurrent.schedule
import kotlin.math.absoluteValue
import kotlin.math.pow

enum class PlayerLocation {
    BACKGROUND,
    FOREGROUND,
    PIP,
    UNCHANGED
}

enum class TrackType {
    AUDIO,
    SUBTITLES
}

enum class StreamRestartReason {
    START_OVER,
    LIVE,
    PAUSE,
    SEEK,
    NONE
}

class NavigationDirection {

    enum class Direction {
        UP,
        DOWN,
        LEFT,
        RIGHT
    }

    var up: Boolean = false
    var down: Boolean = false
    var left: Boolean = false
    var right: Boolean = false

    fun shouldIgnore(direction: Direction): Boolean {
        when (direction) {
            Direction.UP -> {
                if (Flavor().shouldDisableVerticalZapp) return true
                if (up) return true
                if (Flavor().minimalDelayForVerticalRepeat > 0) {
                    up = true
                    Timer("ignoreUp", false).schedule(Flavor().minimalDelayForVerticalRepeat) {
                        up = false
                    }
                }
                return false
            }
            Direction.DOWN -> {
                if (Flavor().shouldDisableVerticalZapp) return true
                if (down) return true
                if (Flavor().minimalDelayForVerticalRepeat > 0) {
                    down = true
                    Timer("ignoreDown", false).schedule(Flavor().minimalDelayForVerticalRepeat) {
                        down = false
                    }
                }
                return false
            }
            Direction.LEFT -> {
                if (Flavor().shouldDisableHorizontalZapp) return true
                if (left) return true
                if (Flavor().minimalDelayForHorizontalRepeat > 0) {
                    left = true
                    Timer("ignoreLeft", false).schedule(Flavor().minimalDelayForVerticalRepeat) {
                        left = false
                    }
                }
                return false
            }
            Direction.RIGHT -> {
                if (Flavor().shouldDisableHorizontalZapp) return true
                if (right) return true
                if (Flavor().minimalDelayForHorizontalRepeat > 0) {
                    right = true
                    Timer("ignoreRight", false).schedule(Flavor().minimalDelayForVerticalRepeat) {
                        right = false
                    }
                }
                return false
            }
        }
    }
}

interface PlayerAction {
    fun stop()
    fun playFromReference(
            reference: SmartMediaReference,
            lastPlayerFragmentLocation: PlayerLocation = PlayerLocation.UNCHANGED,
            isStartOver: Boolean = false,
            isSoftZap: Boolean = false,
            isInitialPlayback: Boolean = false,
            position: Int? = null
    )

    fun isSeeking(): Boolean
    fun pause(byUser: Boolean)
    fun resume()
    fun togglePauseResume()
    fun ff()
    fun rw()
    fun hideControls(): Boolean
    fun showControls(stayShown: Boolean = false)
    fun isUIVisible(): Boolean
    fun getPreviousChannelIdTo(channelReference: SmartMediaReference): String?
    fun getNextChannelIdTo(channelReference: SmartMediaReference): String?
    fun getNextChannelReference(): SmartMediaReference?
    fun getPreviousChannelReference(): SmartMediaReference?
    fun handleKeyRight(action: Int?)
    fun handleKeyLeft(action: Int?)
    fun handleKeyUp(isSoftZap: Boolean = true)
    fun handleKeyDown(isSoftZap: Boolean = true)
    fun handleSecondOK()
    fun handleLiveButtonPress()
    fun getSoftZapMediaReference(): SmartMediaReference?
    fun cancelSoftZap()
    fun softZapping(): Boolean
    fun getChannelByNumber(number: String): SmartMediaReference?
    fun getClosestChannelByNumber(number: String): SmartMediaReference?
    fun showNumberZap(number: String)
    fun switchToLastChannel()
    fun switchToLastEvent()
    fun cancelSeeking()
    fun seekToSelectedPos()
    fun getSeekingRule(): SmartSeekingRuleType
    fun seekToLive()
    fun seekToStartOver()
    fun showUnavailableActionToast()
    fun isTimeShifted(): Boolean
    fun isInErrorState(): Boolean
    fun retryPlay(forced: Boolean = false)
    fun hideSubscriptions()
    fun showSideMenu()
    fun hideSideMenu()
    fun showOrHideSideMenu()
    fun startIntent(intent: Intent)
    fun isPlayingOrShouldResume(): Boolean
    fun resetStreamTimers()
    fun isRewindLongPressHandled(): Boolean
    fun isSideMenuShown(): Boolean
    fun stopPlayerInStandby(isForegroundPlaybackActive: Boolean)
    fun sendBookmark(eventType: SmartPlayerEventType)
}

class PlayerFragment : ScopeFragment(), Player.EventListener, TimerHelper.Listener, PlayerAction, VideoListener {

    companion object {
        private const val FAIL_MAX_RETRIES = 2
        private const val FAIL_MAX_CYCLES = 5
        private const val FAIL_AUTO_RETRY_TIMEOUT = 15000L
        private const val SEEK_UPDATE_INTERVAL = 100 //in millis
        private const val SEEK_STEP_LIMIT = 5 // means x64 ie 2^(x+1)
        private const val STREAM_BEHIND_LIVE_THRESHOLD = 1 * 60 * 1000
        private const val EYE_BLINK = 1000 // used for detect if activity is really paused or just receive intent and went through lifecycle
        private const val ACCEPTABLE_STANDBY_TIME = 5000
        private const val START_BOOKMARK_DELAY = 5000L
    }

    private var player: SmartPlayer? = null
    private val playerViewModel by viewModel<PlayerViewModel>()
    private val pvm = PlayerUIViewsAndModels()
    private lateinit var youboraAnalytics: YouboraAnalytics

    private var playerInfoUpdated: Boolean = false
    private var isInitialPlayback = false
    private var resumePlayer = false
    private var systemPausedInLive = false
    private var playerReady = false
    private var failRetries = 0
    private var failCycles = 0
    private var lastPlayedStream: SmartMediaStream? = null
    private var streamingType = StreamingType.CHANNEL
    private var failAutoRetryTimer: TimerTask? = null
    private var lastPlayedReference: SmartMediaReference? = null
    private var interestedInPlay: Boolean =
            false //prevent start stream after stop action performed (in case user exit player faster then stream is obtained from API)
    private var seekMultiplier = 0
    private var seeking = false
    private var seekIncrement = 0
    private var seekCyclesPassed = 0
    private var seekCyclesPassedMax = 60
    private var isInTrick = false
    private var switchTrickAfterTracksChanged = false
    private var firstFrameRendered = false
    private var inPlayerAction: Boolean = false
    private var isStartOverRequested: Boolean = false
    private var firstOpened: Boolean =
            true //onPlayerPlay is started twice when the player is opened for the first time, using this var to enable restart button
    private var shouldBeSoftZapAfterRetry: Boolean = false
    private var streamRestartReason = StreamRestartReason.NONE
    private var bookmarkHeartbeatTimer: Timer? = null
    private var bookmarkEventStartSent: Boolean = false
    private var rewindBeingHandled: Boolean = false
    private val sideMenu = PlayerSideMenu(this)
    private var pinDialogsShown = 0
    private var playerPausedAtGlobalTime: Long = 0 //used to see how long player is paused
    private val navigationDirection = NavigationDirection()
    private var messageDialogModel: MessageDialogModel? = null
    private var continueWatchingDialogFragment: MessageDialogFragment? = null
    private val firebaseRemoteConfig = FirebaseRemoteConfig.getInstance()
    private var streamTimer: CountDownTimer? = null
    private var streamPausedShouldStayShown: Boolean = false
    private val confirmWatchingTimer = object : CountDownTimer(60/*seconds*/ * 1000/*millis*/, 1000) {
        override fun onTick(millisUntilFinished: Long) {
            if (continueWatchingDialogFragment?.isVisible == true) {
                continueWatchingDialogFragment?.setDialogText(
                        getString(
                                R.string.stream_continue_watching_warning,
                                (millisUntilFinished / 1000).toString()
                        )
                )
            }
        }

        override fun onFinish() {
            player?.stop()
            youboraAnalytics.plugin.allowReportStopPlayback = true
            binding.exoplayerView.subtitleView.setCues(null)
            resetStreamTimers()
            pvm.uiViewModel.errorState = true
            pvm.uiViewModel.streamPausedVisibility.postValue(View.VISIBLE)
            showControls(true)
            if (continueWatchingDialogFragment?.isAdded == true)
                continueWatchingDialogFragment?.dismiss()
            streamPausedShouldStayShown = true
        }
    }
    private val subscriptionScreenHelper = SubscriptionScreenHelper(this)

    fun getPlayer() = player
    fun getPlayerUIView() = pvm.uiView

    var audioSelection: Pair<String?, String?> = Flavor().defaultAudioSelection
    var subtitleSelection: Pair<String?, String?> = Flavor().defaultSubtitleSelection

    private var sideMenuShown = false

    private var isStreamReady = false

    //  region TimerHelper listener override

    override fun onTimer() {
        if (!seeking) {
            val positionMs = player?.currentPosition ?: C.TIME_UNSET
            val durationMs = player?.contentDuration ?: C.TIME_UNSET
            pvm.uiViewModel.onPlayerTimeUpdate(positionMs, durationMs)
        }
    }

    //endregion

    //  region Player.EventListener overrides

    override fun onPlayerStateChanged(playWhenReady: Boolean, playbackState: Int) {
        Timber.tag(TAG_PLAYER_LOG).d( "player state changed: playWhenReady $playWhenReady, playbackState ${translateStateToString(playbackState)}")
        if (playbackState == Player.STATE_ENDED) {
            onPlayerEnded()
        }
        if (playbackState == Player.STATE_READY) {
            val currentPosition: Long = player?.currentPosition ?: 0
            if (playWhenReady) {
                if (currentPosition > 1) onPlayerResume(currentPosition)
                onPlayerPlay()
            } else {
                onPlayerPause(currentPosition)
            }
        }
        if (Flavor().shouldCheckNetworkConnectionQuality) {
            when (playbackState) {
                Player.STATE_READY -> isStreamReady = true
                Player.STATE_ENDED, Player.STATE_IDLE -> isStreamReady = false
                //this should be RE-BUFFER state
                Player.STATE_BUFFERING -> if(isStreamReady) reportStreamingProblem("re-buffer")
            }

            when (playbackState) {
                Player.STATE_BUFFERING,
                Player.STATE_READY -> {
                    (context as? MainActivityAction)?.provideNQC()?.isPlayerPlaying = true
                }
                Player.STATE_IDLE,
                Player.STATE_ENDED -> {
                    (context as? MainActivityAction)?.provideNQC()?.isPlayerPlaying = false
                }
            }
        }
    }

    private fun translateStateToString(state: Int): String {
        return when (state) {
            1 -> "Idle"
            2 -> "Buffering"
            3 -> "Ready"
            4 -> "Ended"
            else -> "Unknown"
        }
    }

    override fun onPositionDiscontinuity(reason: Int) {
        when (reason){
            // User triggered seeking, stream is not ready
            Player.DISCONTINUITY_REASON_SEEK -> isStreamReady = false
            else -> {}
        }
        super.onPositionDiscontinuity(reason)
    }

    private fun reportStreamingProblem(error: String) {
        context?.let {
            val networkInfo = NetworkInfo.getActiveConnectionInfo(it)
            YouboraAnalytics.getInstance(it).reportYouboraNetworkConnection(it, networkInfo, error)
            SharedPreferencesUtils.putConnectionSemaphoreState(
                    if(networkInfo.type == InterfaceType.LAN) SemaphoreState.RED_LAN else SemaphoreState.RED_WIFI
            )
        }
        (context as? MainActivityAction)?.provideNQC()?.let{
            it.cancel()
            it.restart()
        }
        isStreamReady = false
    }

    override fun onPlayerError(error: ExoPlaybackException) {
        Timber.tag(TAG_PLAYER_ERROR_LOG).w(error)
        if (error.cause !is BehindLiveWindowException) failRetries++
        val stream = lastPlayedStream

        if (isStreamReady) reportStreamingProblem(error.message ?: "unknown")

        if (shouldRetryStreamAfterError(stream, error)) {
            Timber.tag(TAG_PLAYER_LOG).d("$failRetries retried")
            (youboraAnalytics.plugin.adapter as? YouboraAdapter)?.onPlayerErrorInternal(error, isFatal = false)
            onDataPrepared(stream!!, PlayerLocation.UNCHANGED)
        } else {
            Timber.tag(TAG_PLAYER_LOG).d("MAX retried")
            (youboraAnalytics.plugin.adapter as? YouboraAdapter)?.onPlayerErrorInternal(error, isFatal = true)
            showPlayFailOverlay()
            if (Flavor().useBookmarksForConcurrency) {
                if (bookmarkHeartbeatTimer != null) {
                    stopBookmarkHeartbeat()
                    sendPlayerEvent(SmartPlayerEventType.ERROR)
                }
            }
            if (Flavor().useBookmarks) {
                sendPlayerEvent(SmartPlayerEventType.ERROR)
            }
        }
    }

    private fun shouldRetryStreamAfterError(stream: SmartMediaStream?, error: ExoPlaybackException): Boolean {
        if (stream == null) return false
        if (error.cause is BehindLiveWindowException) return true
        if (failRetries >= FAIL_MAX_RETRIES) return false
        return true
    }

    override fun onTracksChanged(
            trackGroups: TrackGroupArray,
            trackSelections: TrackSelectionArray
    ) {
        //switch to trick track immediately after startover stream change is not possible because trick track availability is not known yet
        //so we need to wait until update of tracks arrive
        if(switchTrickAfterTracksChanged) {
            switchTrickAfterTracksChanged = false
            switchTrick(true)
        }
        isStreamReady = false // prevents re-buffer reporting when user changes tracks
        val audioTrackPreference = if(isInTrick) TrackPreference.DISABLED else getAudioPreference()
        val trackPreferencesSubtitles = if(isInTrick) TrackPreference.DISABLED else when (subtitleSelection.first?.equals("none") == true) {
            true -> TrackPreference.DISABLED
            false -> TrackPreference.forName(subtitleSelection.first ?: "")
        }
        val videoPreference = if(isInTrick) VideoPreference.TRICK else VideoPreference.DEFAULT

        val trackPreferences = TrackPreferences(1F, videoPreference, audioTrackPreference, trackPreferencesSubtitles)
        player?.trackSelector?.trackPreferences = trackPreferences
        if (!checkBinding("onTracksChanged")) return //Crashlytics fix - the binding can be sometimes null in this method
        binding.exoplayerView.trackSelector = player?.trackSelector
    }

    private fun onPlayerPlay() {
        seekToStartOverIfRequested()
        playerReady = true
        Timber.tag(TAG_PLAYER_LOG).d("storing channel number ${playerViewModel?.channel?.value?.channelNumber}")
        playerViewModel.channel.value?.channelNumber?.let {
            SharedPreferencesUtils.putChannelNumber(it)
        }
    }

    private fun seekToStartOverIfRequested() {
        if (isStartOverRequested) {
            seekToStartOver()
            if (!firstOpened)
                isStartOverRequested = false
        }
        firstOpened = false
    }

    private fun onPlayerResume(position: Long) {
        playerReady = true
        Timber.tag(TAG_PLAYER_LOG).d("resume position $position")
        pvm.uiViewModel.resetPauseTime()
    }

    private fun onPlayerPause(position: Long) {
        playerReady = true
        Timber.tag(TAG_PLAYER_LOG).d("pause position $position")
        pvm.uiViewModel.setPauseTime()
    }

    private fun onPlayerEnded() {
        Timber.tag(TAG_PLAYER_LOG).d("ended")
        (context as? ActivityPlayerAction)?.playerStop()
    }

    //endregion

    //region PlayerAction overrides

    override fun stop() {
        //clear stored channel
        Timber.tag(TAG_PLAYER_LOG).d("clearing stored channel")
        SharedPreferencesUtils.putChannelNumber(-1)
        streamPausedShouldStayShown = false
        interestedInPlay = false
        stayAwake(false)
        if (Flavor().useBookmarksForConcurrency) {
            if (bookmarkHeartbeatTimer != null) {
                stopBookmarkHeartbeat()
                sendPlayerEvent(SmartPlayerEventType.STOP)
            }
        }
        if (Flavor().useBookmarks) {
            sendPlayerEvent(SmartPlayerEventType.STOP)
        }
        resumePlayer = false

        if (isAdded) {
            if (pvm.currentUI != PlayerUIEnum.NONE) {
                pvm.uiViewModel.subscriptionChannel.postValue(null)
                pvm.uiViewModel.resetTimeMillis()
                pvm.uiViewModel.onPlayerActionStop()
            }
            player?.playWhenReady = false
            youboraAnalytics.plugin.allowReportStopPlayback = true
            player?.stop()
            binding.exoplayerView.subtitleView.setCues(null)
            resetStreamTimers()
            player?.release()
            player = null
            youboraAnalytics.detachPlayer()
            clearErrorState()

            // remove all observers
            playerViewModel.playAsMediaType.removeObservers(viewLifecycleOwner)
            playerViewModel.channel.removeObservers(viewLifecycleOwner)
            playerViewModel.detail.removeObservers(viewLifecycleOwner)
            setNoControlsUI()
        }
    }

    /** what kind of reference arrives here?
     * homepage - OnNow row -> detail ... LIVE_EVENT
     * homepage - Channels row ... LIVE_CHANNEL
     * epg - past event -> detail ... LIVE_EVENT
     * epg - current event -> detail ... LIVE_EVENT
     * epg - future event -> detail ... shall never become here
     * Recordings row -> recorded detail ... RECORDING */
    override fun playFromReference(
            reference: SmartMediaReference,
            lastPlayerFragmentLocation: PlayerLocation,
            isStartOver: Boolean,
            isSoftZap: Boolean,
            isInitialPlayback: Boolean,
            position: Int?
    ) {
        if (!isAdded || isDetached) Timber.tag(TAG_PLAYER_ERROR_LOG).e("Calling playFromReference before attach or after detach")
        playerInfoUpdated = false
        interestedInPlay = true
        if (lastPlayedReference != reference) {
            failCycles = 0
            failRetries = 0
        }
        lastPlayedReference = reference
        lastPlayedStream = null
        bookmarkEventStartSent = false
        if (pvm.currentUI != PlayerUIEnum.NONE) {
            pvm.uiViewModel.subscriptionChannel.postValue(null)
        }
        this.isInitialPlayback = isInitialPlayback
        streamRestartReason = StreamRestartReason.NONE
        shouldBeSoftZapAfterRetry = isSoftZap
        clearErrorState()
        playerViewModel.setReference(reference)
        playerViewModel.playAsMediaType.removeObservers(viewLifecycleOwner)
        playerViewModel.channel.removeObservers(viewLifecycleOwner)
        playerViewModel.detail.removeObservers(viewLifecycleOwner)
        playerViewModel.playAsMediaType.observe(viewLifecycleOwner, Observer { type ->
            if (playerViewModel.ready != true) return@Observer
            when (type) {
                null -> return@Observer
                SmartMediaType.LIVE_CHANNEL -> internalPlayChannel(lastPlayerFragmentLocation)
                SmartMediaType.LIVE_EVENT -> internalPlayCatchup(lastPlayerFragmentLocation, isSoftZap, position)
                SmartMediaType.EPISODE,
                SmartMediaType.SERIES,
                SmartMediaType.RECORDING,
                SmartMediaType.MOVIE -> internalPlayVod(lastPlayerFragmentLocation, position)
                else -> displayToast("Media type ${type.name} cannot be played")
            }
            isStartOver.ifTrue { isStartOverRequested = true }
        })
    }

    override fun pause(byUser: Boolean) {
        if (byUser) showControls(true)
        playerPausedAtGlobalTime = TimeProvider.nowMs()
        if (Flavor().useBookmarksForConcurrency) {
            if (bookmarkHeartbeatTimer != null) {
                stopBookmarkHeartbeat()
                sendPlayerEvent(SmartPlayerEventType.PAUSE)
            }
        }
        if (Flavor().useBookmarks) {
            sendPlayerEvent(SmartPlayerEventType.PAUSE)
        }
        if (Flavor().useDifferentUrlForStartOver && !isInStartOverMode() && streamingType == StreamingType.CHANNEL && byUser) {
            systemPausedInLive = false
            streamRestartReason = StreamRestartReason.PAUSE
            lastPlayedStream?.let {
                onDataPrepared(it, PlayerLocation.UNCHANGED)
            }
        } else {
            player?.let {
                //if stream is paused less then minute behind live it is considered as "paused in live"
                systemPausedInLive = !byUser && streamingType == StreamingType.CHANNEL && it.duration - it.currentPosition < STREAM_BEHIND_LIVE_THRESHOLD
            }
            player?.playWhenReady = false
        }
    }

    fun getStreamingType(): StreamingType {
        return streamingType
    }

    override fun sendBookmark(eventType: SmartPlayerEventType) {
        sendPlayerEvent(eventType)
    }

    private fun switchTrick(useTrickTrack: Boolean) {
        if (useTrickTrack && isTrickTrackAvailable()) {
            isInTrick = true
            player?.exoPlayer?.setSeekParameters(SeekParameters.CLOSEST_SYNC)
            player?.trackSelector?.trackPreferences = TrackPreferences(1F, VideoPreference.TRICK, TrackPreference.DISABLED, TrackPreference.DISABLED)
         } else {
            isInTrick = false
            player?.exoPlayer?.setSeekParameters(SeekParameters.DEFAULT)
            player?.trackSelector?.trackPreferences = TrackPreferences(1F, VideoPreference.DEFAULT, TrackPreference.DEFAULT, TrackPreference.DISABLED)
        }
    }

    private fun sendPlayerEvent(eventType: SmartPlayerEventType) {
        lastPlayedStream?.let {
            val event = SmartPlayerEvent(it.reference, eventType)

            when (eventType) {
                SmartPlayerEventType.START -> { //translated in PhoenixAPI to FIRST_PLAY
                    event.position = player?.currentPosition?.toInt() ?: 0
                    pvm.uiViewModel.sendPlayerEvent(event, null)
                    bookmarkEventStartSent = true
                }
                SmartPlayerEventType.RESUME, //translated in Phoenix API to PLAY
                SmartPlayerEventType.HEARTBEAT -> {//translated in PhoenixAPI to HIT
                    event.position = player?.currentPosition?.toInt() ?: 0
                    pvm.uiViewModel.sendPlayerEvent(event, null)
                }
                SmartPlayerEventType.STOP,
                SmartPlayerEventType.ERROR,
                SmartPlayerEventType.PAUSE -> {
                    event.position = player?.currentPosition?.toInt() ?: 0
                    event.duration = player?.duration?.toInt() ?: 0
                    if (event.position != 0 && event.duration != 0 ) {
                        if (Flavor().shouldPostBookmark(event, bookmarkEventStartSent, pvm.uiViewModel.playingEvent.value?.isLive)) {
                            event.apply { position = setBookmarkPosition(event) }
                            pvm.uiViewModel.sendPlayerEvent(event, playerViewModel.buildBookmark(event.position, event.duration))
                        }
                    } else {
                        pvm.uiViewModel.sendPlayerEvent(event, null)
                    }
                }
                else -> {}
            }
        }
    }

    private fun setBookmarkPosition(event: SmartPlayerEvent): Int {
        return if (event.duration - Flavor().bookmarkResetThresholdMillis < event.position) {
            0 //reset bookmark when position exceeds reset threshold
        } else {
            event.position
        }
    }

    private fun startBookmarkHeartbeat(isFirstPlay: Boolean = false) {
        stopBookmarkHeartbeat()
        var shouldSendStart = isFirstPlay
        bookmarkHeartbeatTimer = fixedRateTimer(null, false, Flavor().bookmarkForConcurrencyHeartbeatInitDelay, Flavor().bookmarkForConcurrencyHeartbeatTick) {
            GlobalScope.launch(Dispatchers.Main) {
                if (shouldSendStart) {
                    shouldSendStart = false
                    sendPlayerEvent(SmartPlayerEventType.START)
                } else {
                    sendPlayerEvent(SmartPlayerEventType.HEARTBEAT)
                }
            }
        }
    }

    private fun stopBookmarkHeartbeat() {
        bookmarkHeartbeatTimer?.cancel()
        bookmarkHeartbeatTimer = null
    }

    override fun resume() {
        showControls()
        player?.playWhenReady = true
        val playerPausedFor = TimeProvider.nowMs() - playerPausedAtGlobalTime
        playerPausedAtGlobalTime = 0
        if (systemPausedInLive && playerPausedFor > EYE_BLINK) doSeekToLive() //try to seek into live when paused in live before by system
        if (Flavor().useBookmarksForConcurrency) {
            if (bookmarkHeartbeatTimer == null) {
                sendPlayerEvent(SmartPlayerEventType.RESUME)
                startBookmarkHeartbeat()
            }
        }
    }

    override fun togglePauseResume() {
        if (playerReady) {
            player?.isPlaying?.also {
                if (it) pause(true)
                else resume()
                pvm.uiViewModel.applyManualPause(it)
            }
        }
    }

    override fun ff() {
        if (checkPlayingLiveEvent()) return
        if (pvm.uiViewModel.isFastForwardAllowed()) {
            when (seeking) {
                false -> {
                    player?.playWhenReady = false
                    seekMultiplier = 1
                    if (Flavor().useDifferentUrlForStartOver && !isInStartOverMode() && streamingType == StreamingType.CHANNEL) {
                        streamRestartReason = StreamRestartReason.SEEK
                        lastPlayedStream?.let {
                            onDataPrepared(it, PlayerLocation.UNCHANGED)
                        }
                    } else {
                        startSeeking()
                    }
                }
                true -> {
                    if (seekMultiplier < 0) seekMultiplier = 0
                    seekMultiplier++
                    if (seekMultiplier > SEEK_STEP_LIMIT) seekMultiplier = 1
                }
            }
            setSeekData()
        } else {
            showUnavailableActionToast()
        }
    }

    override fun rw() {
        if (checkPlayingLiveEvent()) return
        when (seeking) {
            false -> {
                player?.playWhenReady = false
                seekMultiplier = -1
                if (Flavor().useDifferentUrlForStartOver && !isInStartOverMode() && streamingType == StreamingType.CHANNEL) {
                    streamRestartReason = StreamRestartReason.SEEK
                    lastPlayedStream?.let {
                        onDataPrepared(it, PlayerLocation.UNCHANGED)
                    }
                } else {
                    startSeeking()
                }
            }
            true -> {
                if (seekMultiplier > 0) seekMultiplier = 0
                seekMultiplier--
                if (seekMultiplier < -SEEK_STEP_LIMIT) seekMultiplier = -1
            }
        }
        setSeekData()
    }

    override fun hideControls(): Boolean {
        isUIVisible().ifFalse { return false }
        pvm.uiView.hideControls() //first back should hide the UI so the next back will be caught by the condition on previous line
        return pvm.uiViewModel.errorState.not() //if there is an error we don't want to wait on another back -> return false -> this will close the player
    }

    override fun showControls(stayShown: Boolean) {
        //do not show controls when on background
        if ((context as? ActivityPlayerAction)?.provideLastPlayerLocation() == PlayerLocation.BACKGROUND) return

        if (stayShown || pinDialogsShown > 0) pvm.uiView.showControls(stayShown) else pvm.uiView.showControls(false)
    }

    override fun getNextChannelReference(): SmartMediaReference? {
        inPlayerAction = true
        return pvm.uiViewModel.getNextChannelReference()
    }

    override fun getPreviousChannelReference(): SmartMediaReference? {
        inPlayerAction = true
        return pvm.uiViewModel.getPreviousChannelReference()
    }

    override fun isSeeking(): Boolean = seeking

    override fun isPlayingOrShouldResume(): Boolean = resumePlayer || player?.isPlaying ?: false

    override fun resetStreamTimers() {
        streamTimer?.cancel()
        confirmWatchingTimer.cancel()
    }

    override fun isRewindLongPressHandled(): Boolean {
        return rewindBeingHandled
    }

    override fun isSideMenuShown() = sideMenuShown

    override fun handleKeyRight(action: Int?) {
        handleKeyCommon().ifTrue { return }
        if (navigationDirection.shouldIgnore(NavigationDirection.Direction.RIGHT)) return
        if (action == KeyEvent.ACTION_UP) {
            if (pvm.uiViewModel.loadingEvents == true) showLoadingMoreEventsToast()
            else pvm.uiViewModel.softZapNextEvent()
        }
    }

    override fun handleKeyLeft(action: Int?) {
        handleKeyCommon().ifTrue { return }
        if (navigationDirection.shouldIgnore(NavigationDirection.Direction.LEFT)) return
        if (action == KeyEvent.ACTION_UP) {
            if (pvm.uiViewModel.loadingEvents) showLoadingMoreEventsToast()
            else pvm.uiViewModel.softZapPreviousEvent()
        }
    }

    private fun showLoadingMoreEventsToast() {
        TvLauncherToast.makeText(
                requireContext(),
                R.string.player_loading_more_events,
                Toast.LENGTH_SHORT
        )?.show()
    }

    override fun handleKeyDown(isSoftZap: Boolean) {
        streamPausedShouldStayShown = false
        if (streamingType == StreamingType.EVENT && Flavor().handleUpDownChannelInCatchup) {
            pvm.uiViewModel.getPreviousChannelReference()?.let {
                playFromReference(it)
            }
        } else {
            handleKeyCommon().ifTrue { isSoftZap.ifTrue { return } }
            if (navigationDirection.shouldIgnore(NavigationDirection.Direction.DOWN)) return
            inPlayerAction = true
            pvm.uiViewModel.softZapPreviousChannel()
        }
    }

    override fun handleKeyUp(isSoftZap: Boolean) {
        streamPausedShouldStayShown = false
        if (streamingType == StreamingType.EVENT && Flavor().handleUpDownChannelInCatchup) {
            pvm.uiViewModel.getNextChannelReference()?.let {
                playFromReference(it)
            }
        } else {
            handleKeyCommon().ifTrue { isSoftZap.ifTrue { return } }
            if (navigationDirection.shouldIgnore(NavigationDirection.Direction.UP)) return
            inPlayerAction = true
            pvm.uiViewModel.softZapNextChannel()
        }
    }

    private fun handleKeyCommon(): Boolean {
        isUIVisible().ifFalse {
            showControls()
            return true
        }
        showControls()
        inPlayerAction = true
        return false
    }

    override fun handleSecondOK() {
        if (isRewindLongPressHandled()) return // already being handled
        if (checkPlayingLiveEvent()) return //check if it is live event, return otherwise
        if (checkSeekingRuleWhetherWeHaveToStayInLive()) return
        rewindBeingHandled = true
        val messageDialogModel = if (streamingType == StreamingType.CHANNEL
                && ((player?.duration ?: 0) - (player?.currentPosition
                        ?: 0)) < STREAM_BEHIND_LIVE_THRESHOLD
        ) {
            val eventTitle = (getPlayerUIView() as? PlayerUILive)?.viewModel?.getPlayingEventTitleSafe() ?: ""
            MessageDialogModel(
                    message = getString(R.string.player_ok_longpress_action_dialog_event, eventTitle),
                    description = null,
                    optionsButtonText = arrayOf("", getString(R.string.button_yes)),
                    cancelButtonText = getString(R.string.button_no),
                    code = MessageDialogCodes.reproduceFromBeginning
            )
        } else {
            MessageDialogModel(
                    message = getString(R.string.player_ok_longpress_action_dialog),
                    description = null,
                    optionsButtonText = arrayOf(getString(R.string.player_ok_longpress_action_go_live), getString(R.string.player_ok_longpress_action_restart)),
                    cancelButtonText = getString(R.string.cancel),
                    code = MessageDialogCodes.reproduceFromBeginning
            )
        }

        val confirmMessageDialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        confirmMessageDialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {
                rewindBeingHandled = false
            }
        }
        confirmMessageDialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                rewindBeingHandled = false
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when (answer.type) {
                    OPTION_A -> {
                        if (streamingType == StreamingType.EVENT || streamingType == StreamingType.CATCHUP_IN_LIVE_PLAYER) {
                            (context as? ActivityPlayerAction)?.restartPlayerOnChannel()
                        } else doSeekToLive()
                    }
                    OPTION_B -> doSeekToStartOver()
                    CANCEL -> {
                    } //cancel, do nothing
                }
                return false
            }
        }

        (view?.context as? FragmentActivity)?.supportFragmentManager?.also {
            confirmMessageDialogFragment.show(
                    it,
                    "tag_dialog_startover"
            )
        }
    }

    private fun checkSeekingRuleWhetherWeHaveToStayInLive(): Boolean {
        val rule = pvm.uiViewModel.getSeekingRule()
        return when (rule) {
            SmartSeekingRuleType.LIVE_ONLY -> true
            else -> false
        }
    }

    override fun handleLiveButtonPress() {
        if (Flavor().shouldPlayerHandleLiveButton) {
            when (streamingType) {
                StreamingType.CHANNEL -> {
                    if (isTimeShifted())
                        (context as? ActivityPlayerAction)?.restartPlayerOnChannel()
                    else
                        showControls()
                }
                StreamingType.EVENT,
                StreamingType.CATCHUP_IN_LIVE_PLAYER -> (context as? ActivityPlayerAction)?.restartPlayerOnChannel()
                else -> doSeekToLive()
            }
        }
    }

    override fun getSoftZapMediaReference() = pvm.uiViewModel.getSoftZapMediaReference()

    override fun cancelSoftZap() {
        pvm.uiViewModel.cancelSoftZap()
    }

    override fun softZapping() = pvm.uiViewModel.softZaping()

    override fun getChannelByNumber(number: String): SmartMediaReference? {
        inPlayerAction = true
        return pvm.uiViewModel.getChannelReferenceByNumber(number)
    }

    override fun getClosestChannelByNumber(number: String): SmartMediaReference? {
        inPlayerAction = true
        val closestChannel = pvm.uiViewModel.getClosestChannelReferenceByNumber(number)
                ?: return null
        val isFlavorSpecificAppChannel = (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(closestChannel.toString()) == true)
        return if (isFlavorSpecificAppChannel) {
            val prevChID = getPreviousChannelReference()?.toString() ?: "-1"
            val nextChID = getNextChannelIdTo(closestChannel) ?: "-1"
            val launchIntent: Intent? = Flavor().getAppChannelsDelegate()?.getAppChannelIntent(closestChannel.toString(), prevChID, nextChID, isNumberZapp = true)
            try {
                launchIntent?.let { startActivity(it) }
            } catch (e: ActivityNotFoundException) {
                Timber.tag(TAG_PLAYER_ERROR_LOG).w(e.stackTraceToString())
            }
            null
        } else {
            closestChannel
        }

    }

    override fun showNumberZap(number: String) {
        binding.playerUiNumberZapp.setNumber(number)
    }

    override fun switchToLastChannel() {
        streamPausedShouldStayShown = false
        val reference = pvm.getExistingUI(PlayerUIEnum.LIVE)?.viewModel?.getLastChannelReference(
            streamingType == StreamingType.EVENT || streamingType == StreamingType.CATCHUP_IN_LIVE_PLAYER
        )
        sendPlayerEvent(SmartPlayerEventType.STOP)
        if (reference != null) {
            player?.stop()
            playFromReference(reference, PlayerLocation.UNCHANGED)
        } else {
            switchToLastEvent()
        }
    }

    override fun switchToLastEvent() {
        pvm.uiViewModel.getLastPlayedEventReference()?.also {
            playFromReference(it, PlayerLocation.UNCHANGED)
        }.ifNull {
            pvm.uiViewModel.playingEvent.removeObservers(viewLifecycleOwner)
            pvm.uiViewModel.subscriptionChannel.postValue(null)
            pinDialogsShown = 0
            onPlayerEnded()
        }
    }

    override fun cancelSeeking() {
        if (seeking) {
            resetSeeking()
            pvm.uiViewModel.cancelSeeking()
            player?.playWhenReady = true
        }
    }

    override fun seekToSelectedPos() {
        if (seeking) {
            resetSeeking()
            pvm.uiViewModel.seekToSelectedPos(player, false)
            player?.playWhenReady = true
        }
    }

    override fun getSeekingRule() = pvm.uiViewModel.getSeekingRule()

    override fun seekToLive() {
        when (streamingType) {
            StreamingType.CHANNEL -> {
                getPlayerUIView().viewModel.playingEvent.value.ifNull { return }
                cancelSoftZap()
                showControls()
                when (pvm.uiViewModel.getSeekingRuleTypeInternal()) {
                    SmartSeekingRuleType.LIVE_ONLY -> showUnavailableActionToast()
                    SmartSeekingRuleType.TIMESHIFT -> doSeekToLive()
                    SmartSeekingRuleType.STARTOVER -> isTimeShifted().ifTrue { confirmSeek(true) }.ifElse { showUnavailableActionToast() }
                    null -> showUnavailableActionToast()
                    else -> {}
                }
            }
            StreamingType.EVENT -> {
                when (pvm.uiViewModel.getSeekingRuleTypeInternal()) {
                    SmartSeekingRuleType.LIVE_ONLY,
                    SmartSeekingRuleType.STARTOVER -> showUnavailableActionToast()
                    SmartSeekingRuleType.TIMESHIFT -> Unit
                    null -> showUnavailableActionToast()
                    else -> {}
                }
            }
            StreamingType.CATCHUP_IN_LIVE_PLAYER,
            StreamingType.VOD -> {
                handleLiveButtonPress()
            }
        }
    }

    override fun seekToStartOver() {
        if (isStateSaved) return
        when (streamingType) {
            StreamingType.CHANNEL -> {
                getPlayerUIView().viewModel.playingEvent.value.ifNull { return }
                cancelSoftZap()
                showControls()
                when (pvm.uiViewModel.getSeekingRuleTypeInternal()) {
                    SmartSeekingRuleType.LIVE_ONLY -> showUnavailableActionToast()
                    SmartSeekingRuleType.TIMESHIFT -> doSeekToStartOver()
                    SmartSeekingRuleType.STARTOVER -> {
                        when (isTimeShifted()) {
                            true -> confirmSeek(false)
                            false -> doSeekToStartOver()
                        }
                    }
                    null -> showUnavailableActionToast()
                    else -> {}
                }
            }
            StreamingType.CATCHUP_IN_LIVE_PLAYER,
            StreamingType.EVENT,
            StreamingType.VOD -> {
                showControls()
                when (pvm.uiViewModel.getSeekingRuleTypeInternal()) {
                    SmartSeekingRuleType.LIVE_ONLY -> confirmSeek(false)
                    SmartSeekingRuleType.TIMESHIFT -> doSeekToStartOver()
                    SmartSeekingRuleType.STARTOVER -> confirmSeek(false)
                    null -> showUnavailableActionToast()
                    else -> {}
                }
            }
        }
    }

    override fun showUnavailableActionToast() {
        val messageRes = when (streamingType) {
            StreamingType.VOD,
            StreamingType.CATCHUP_IN_LIVE_PLAYER,
            StreamingType.EVENT -> R.string.player_action_not_available
            StreamingType.CHANNEL -> R.string.player_action_not_available_channel
        }
        showControls()
        displayToast(messageRes)
    }

    override fun isTimeShifted() = pvm.uiViewModel.isTimeShifted()

    override fun isInErrorState(): Boolean {
        return try {
            pvm.uiViewModel.errorState
        } catch (e: IllegalArgumentException){
            false
        }
    }

    override fun retryPlay(forced: Boolean) {
        if (forced) failCycles = 0
        lastPlayedReference?.let {
            streamPausedShouldStayShown = false
            playFromReference(it, isSoftZap = shouldBeSoftZapAfterRetry)
        }
    }

    override fun hideSubscriptions() {
        subscriptionScreenHelper.hideSubscriptionScreen()
        pvm.uiViewModel.subscriptionChannel.postValue(null)
    }

    //endregion

    //region Fragment overrides

    private var nullableBinding: FragmentPlayerBinding? = null

    val binding: FragmentPlayerBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        nullableBinding = FragmentPlayerBinding.inflate(inflater, container, false)
        return binding.root
//        return inflater.inflate(R.layout.fragment_player, container, false)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        nullableBinding = null
    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        pvm.setCurrentUIView(context, PlayerUIEnum.NO_CONTROLS)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.exoplayerView.style = PlayerViewStyle()
        binding.exoplayerView.captionStyle = (Flavor().getSubtitlesStyle())
        binding.exoplayerView.controls = pvm.uiView

        setupRecordingObserver()
        messageDialogModel = MessageDialogModel(
                message = getString(R.string.stream_continue_watching_warning, "59"),
                description = null,
                singleButton = getString(R.string.stream_confirm_button),
                code = MessageDialogCodes.playerWillBePaused
        )
        audioSelection = SharedPreferencesUtils.getAudioSelection(
                name = Flavor().defaultAudioSelection.first,
                language = Flavor().defaultAudioSelection.second
        )
        subtitleSelection = SharedPreferencesUtils.getSubtitleSelection(
                name = Flavor().defaultSubtitleSelection.first,
                language = Flavor().defaultSubtitleSelection.second
        )
        youboraAnalytics = YouboraAnalytics.getInstance(requireContext())
    }

     private fun setupSubscriptionObserver() {
         if (pvm.uiViewModel.subscriptionChannel.hasActiveObservers().not()) {
             pvm.uiViewModel.subscriptionChannel.observe(viewLifecycleOwner) { channelItem ->
                 channelItem?.let {
                     if (playerViewModel.channel.value == channelItem && binding.exoplayerView.isShown) {
                         subscriptionScreenHelper.showSubscriptionScreen(it)
                     }
                 } ?: subscriptionScreenHelper.hideSubscriptionScreen()
             }
         }
     }

    override fun onResume() {
        super.onResume()
        Timber.tag(TAG_PLAYER_LOG).d("Player fragment resumed")
        NetworkConnectionState.instance.waitForConnection(lifecycleOwner = viewLifecycleOwner, skipWaiting = !ScreenOnOffReceiver.screenWasOff) {

            if (resumePlayer) {
                resumePlayer = false

                val playerPausedFor = TimeProvider.nowMs() - playerPausedAtGlobalTime
                if (playerPausedFor > ACCEPTABLE_STANDBY_TIME) {
                    playerPausedAtGlobalTime = 0
                    (context as? ActivityPlayerAction)?.restartPlayerOnChannel()
                } else {
                    resume()
                }
            }
        }
    }

    override fun onPause() {
        Timber.tag(TAG_PLAYER_LOG).d("Player fragment paused")
        cancelSeeking()
        if (player?.isPlaying == true) {
            pause(false)
            resumePlayer = true
        }
        hideSubscriptions() //QUESTIONABLE, need test
        hideSideMenu()      //QUESTIONABLE, need test
        stopBookmarkHeartbeat()
        resetStreamTimers()
        super.onPause()
    }

    override fun stopPlayerInStandby(isForegroundPlaybackActive: Boolean) {
        if (isForegroundPlaybackActive && !resumePlayer) {
            resumePlayer = isForegroundPlaybackActive
        }
        player?.stop() // player must be stopped to stop all background network traffic
    }

    override fun onSaveInstanceState(outState: Bundle) {
        /*
        fix attempt for
        java.lang.IllegalStateException: Can not perform this action after onSaveInstanceState
        at com.twentyfouri.tvlauncher.ui.PlayerFragment$initStreamTimer$1.onFinish(PlayerFragment.kt:1545)
        This crash means, that streamTimer cannot be finished after onSaveInstanceState.
        So we try to finish it before, every time this method is called.
        */
        //TODO: Remove this method, if no affect is achieved
        streamTimer?.cancel()
        confirmWatchingTimer.cancel()
        super.onSaveInstanceState(outState)
    }

    override fun showSideMenu() {
        sideMenu.showSideMenu()
        sideMenu.listFragment?.setFragmentResultListener(ListPickerFragment.REQUEST_KEY) { _, bundle ->
            sideMenu.onActivityResult(bundle)
            sideMenuShown = false
        }
        sideMenuShown = true
    }

    override fun hideSideMenu() {
        sideMenu.hideSideMenu()
        sideMenuShown = false
    }

    override fun showOrHideSideMenu() {
        if (sideMenuShown) hideSideMenu() else showSideMenu()
    }

    //endregion

    //region private methods

    fun resetSeeking() {
        showControls(false)
        if(isInTrick) switchTrick(false)
        seeking = false
        seekMultiplier = 0
        setSeekData()
    }

    private fun setNoControlsUI() {
        if (pvm.uiView is PlayerUINoControls) return
        pvm.uiViewModel.stopTimer()

        pvm.setCurrentUIView(requireContext(), PlayerUIEnum.NO_CONTROLS)
        pvm.uiViewModel.startTimer()
        binding.exoplayerView.controls = pvm.uiView
    }

    private fun showPlayFailOverlay() {
        if(!checkBinding("showPlayFailOverlay")) return
        youboraAnalytics.plugin.allowReportStopPlayback = true
        player?.stop()
        binding.exoplayerView.subtitleView.setCues(null)
        resetStreamTimers()
        player?.release()
        player = null

        playerReady = false
        stayAwake(false)
        failRetries = 0
        failCycles++
        showControls(true)

        if (failCycles >= FAIL_MAX_CYCLES) {
            Timber.tag(TAG_PLAYER_LOG).d("MAX fail cycles")
            failCycles = 0
        } else {
            failAutoRetryTimer?.cancel()
            failAutoRetryTimer = null
            failAutoRetryTimer = Timer("failAutoRetry", false).schedule(FAIL_AUTO_RETRY_TIMEOUT) {
                CoroutineScope(Dispatchers.Main).launch {
                    //check to prevent from calling during or after destroy or in case subscription screen is visible
                    if (view != null && pvm.uiViewModel.subscriptionChannel.value == null && pinDialogsShown == 0) retryPlay()
                }
            }
        }
        pvm.uiViewModel.errorState = true
        pvm.uiViewModel.playFailedVisibility.postValue(View.VISIBLE)
    }

    private fun stayAwake(stayAwake: Boolean) {
        Timber.tag(TAG_PLAYER_LOG).d("Stay awake $stayAwake")
        if (stayAwake) activity?.window?.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
        else activity?.window?.clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

    private fun onDataPrepared(stream: SmartMediaStream, location: PlayerLocation, position: Int? = null, startedByUser: Boolean = false) {
        if(!checkBinding("onDataPrepared")) return //Crashlytics fix - the binding can be sometimes null in this method
        isStreamReady = false
        switchTrickAfterTracksChanged = false
        if (!interestedInPlay) return //prevent start stream after stop action performed (in case user exit player faster then stream is obtained from API)
        stayAwake(true)
        val firstPlay = lastPlayedStream == null
        lastPlayedStream = stream
        if (Flavor().useBookmarksForConcurrency) {
            if (firstPlay) {
                startBookmarkHeartbeat(true)
            } else {
                if (bookmarkHeartbeatTimer == null && !isInStartOverMode()) {
                    sendPlayerEvent(SmartPlayerEventType.RESUME)
                    startBookmarkHeartbeat()
                }
            }
        }
        youboraAnalytics.plugin.allowReportStopPlayback = startedByUser //Do not allow to report playback STOP when we reset the player internally
        player?.stop()
        binding.exoplayerView.subtitleView.setCues(null)
        resetStreamTimers()
        player?.release()
        player?.removeListener(this)
        player?.exoPlayer?.removeVideoListener(this)
        player = null
        playerReady = false
        isInTrick = false

        val audioAttributes = AudioAttributes.Builder()
                .setUsage(C.USAGE_MEDIA)
                .setContentType(C.CONTENT_TYPE_MOVIE)
                .build()

        //original stream
        Timber.tag(TAG_PLAYER_LOG).d("videoUrl O: ${stream.primaryUrl}")
        //"http://186.10.58.75/wp/test.entel.cl/test2232/vxfmt=dp/manifest.mpd?device_profile=CLEAR_DASH_TV_PR_FULL"
        //val videoUrl = "http://livesim.dashif.org/livesim/mup_300/tsbd_3600/testpic_2s/Manifest.mpd"
        val videoUrl = stream.primaryUrl.let {
            when (streamingType) {
                StreamingType.VOD -> it
                StreamingType.CHANNEL -> {
                    if (Flavor().useDifferentUrlForStartOver && isInStartOverMode()) {
                        Flavor().getStartOverStreamUrl(stream, pvm.uiViewModel.getPlayingEventStart())
                    } else {
                        Flavor().getLiveStreamUrl(stream)
                    }
                }
                StreamingType.CATCHUP_IN_LIVE_PLAYER,
                StreamingType.EVENT -> Flavor().getCatchupStreamUrl(stream)
            }
        }
        Timber.tag(TAG_PLAYER_LOG).d("videoUrl M: $videoUrl")
        val playerConfigurationModel = PlayerConfigurationModel.Builder()
        playerConfigurationModel.useTunneling = Flavor().isTunnelingEnabled()
        playerConfigurationModel.bufferForPlaybackMs = Flavor().getBufferForPlaybackMs
        playerConfigurationModel.bufferForPlaybackAfterRebufferMs = Flavor().getReBufferForPlaybackMs
        playerConfigurationModel.livePresentationDelayMs = Flavor().getLivePresentationDelayMs
        playerConfigurationModel.livePresentationDelayOverrideManifest = Flavor().getLivePresentationDelayOverrideManifest
        playerConfigurationModel.bandwidthMeter = Flavor().getCustomBandwidthMeter(requireContext())
        playerConfigurationModel.httpDataSourceFactory = LoggingDefaultHttpDataSource.httpDataSourceFactoryProvider

        Timber.tag(TAG_PLAYER_LOG).d("config useTunneling: ${playerConfigurationModel.useTunneling}," +
                " buffer: ${playerConfigurationModel.bufferForPlaybackMs}," +
                " re-buffer: ${playerConfigurationModel.bufferForPlaybackAfterRebufferMs}," +
                " liveDelay: ${playerConfigurationModel.livePresentationDelayMs}," +
                " liveDelayOverride: ${playerConfigurationModel.livePresentationDelayOverrideManifest}")

        stream.streamLicense?.let {
            val drmConfiguration = DrmConfiguration.Builder()
            drmConfiguration.drmLicenseUrl = it.licenseUrl
            drmConfiguration.drmSchemeUuid = Flavor().getDrmScheme()
            drmConfiguration.drmProperties = HashMap(Flavor().getDrmProperties(it))
            drmConfiguration.drmCallback = Flavor().getDrmCallback(
                    drmConfiguration.drmCallback,
                    stream.extras["drmToken"].getString() ?: "",
                    stream.primaryUrl
            )
            playerConfigurationModel.drm = drmConfiguration
        }

        //build player source model
        val source = PlayerSourceModel.Builder()
        source.uri = videoUrl

        //initialize player
        player = SmartPlayerFactory(requireContext()).build(playerConfigurationModel.build())

        //Youbora guys are bitching about that the Youbora is attached too late, so here it is right
        //after the player is created
        player?.also {
            youboraAnalytics.attachPlayer(
                    player = it,
                    streamingType = streamingType,
                    isInStartoverMode = isStartOverRequested,
                    stream = stream,
                    contentChannel = playerViewModel.channel.value?.title,
                    activity = this.activity,
                    reference = lastPlayedReference
            )
            Timber.tag(TAG_PLAYER_LOG).d("Youbora attached")
        }
        youboraAnalytics.plugin.disableIfTitleNullOrEmpty()
        player?.addListener(this)
        player?.prepare(source.build())
        player?.exoPlayer?.addVideoListener(this)
        binding.exoplayerView.player = player
        player?.audioComponent?.setAudioAttributes(audioAttributes, true)
        playerInfoUpdated = true
        when {
            location == PlayerLocation.BACKGROUND -> hideControls()
            streamRestartReason == StreamRestartReason.PAUSE -> {
            } //let controls as they are (probably already set to stayShown)
            else -> showControls()
        }

        if (pinDialogsShown == 0
            && streamPausedShouldStayShown.not()
            && lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED)) {
            player?.playWhenReady = true
            if (position != null) {
                player?.seekTo((position * 1000).toLong())
            }
            if (Flavor().shouldStartStreamTimeout)
                setupAndStartTimer()
        }

        when (streamRestartReason) {
            StreamRestartReason.START_OVER -> {
                val seekBackFromLive = TimeProvider.nowMs() - (pvm.uiViewModel.getPlayingEventStart())
                if (seekBackFromLive > (Flavor().startOverOffsetSize * 1000)) player?.seekTo(0)
                else player?.seekTo((Flavor().startOverOffsetSize * 1000) - seekBackFromLive)
            }
            StreamRestartReason.PAUSE -> {
                player?.playWhenReady = false
                showControls(true)
            }
            StreamRestartReason.SEEK -> {
                player?.playWhenReady = false
                switchTrickAfterTracksChanged = true //reminder to switch to trick track (if available) once tracks info is available
                startSeeking()
            }
            StreamRestartReason.NONE,
            StreamRestartReason.LIVE -> {
            }
        }
    }

    override fun onRenderedFirstFrame() {
        super.onRenderedFirstFrame()
        firstFrameRendered = true
    }

    private fun setupYouboraInfoObservers() {
        if (pvm.uiViewModel.getPlayingEventTitleLD()?.hasActiveObservers() == false) {
            pvm.uiViewModel.getPlayingEventTitleLD()?.observe(viewLifecycleOwner,
                    Observer { eventTitle ->
                        if (eventTitle != resources.getString(R.string.player_loading_channel)) {
                            youboraAnalytics.plugin.reenableIfNecessary()
                            youboraAnalytics.plugin.options.contentTitle = eventTitle
                        }
                    }
            )
        }
    }

    private fun getAllTracks() = getPlayer()?.let { it.trackSelector.getTrackInfos(it) }

    private fun getAudioPreference(): TrackPreference {
        getAllTracks()?.filter { it.type == TrackInfo.TYPE_AUDIO }?.let {
            val trackInfo = it.find { trackInfo ->
                trackInfo.name == audioSelection.first
            }
            if (trackInfo != null) return TrackPreference.forTrackInfo(trackInfo)
        }
        return TrackPreference.DEFAULT
    }

    private fun isTrickTrackAvailable(): Boolean {
       return getAllTracks()?.filter { it.type == TrackInfo.TYPE_VIDEO }?.find { it.format.roleFlags and C.ROLE_FLAG_TRICK_PLAY != 0 } != null
    }

    private fun isInStartOverMode(): Boolean {
        return streamRestartReason != StreamRestartReason.NONE && streamRestartReason != StreamRestartReason.LIVE
    }

    fun storeTrackSelection(type: TrackType, name: String?, language: String?) {
        when (type) {
            TrackType.AUDIO -> {
                SharedPreferencesUtils.putAudioSelection(name, language)
                audioSelection = Pair(name, language)
            }
            TrackType.SUBTITLES -> {
                SharedPreferencesUtils.putSubtitleSelection(name, language)
                subtitleSelection = Pair(name, language)
            }
        }
    }

    private fun clearErrorState() {
        confirmWatchingTimer.cancel()
        streamTimer?.cancel()
        failAutoRetryTimer?.cancel()
        failAutoRetryTimer = null
        if (pvm.currentUI != PlayerUIEnum.NONE) { // prevent IllegalArgumentException, which is thrown when currenUI is NONE
            pvm.uiViewModel.errorState = false
            pvm.uiViewModel.playFailedVisibility.postValue(View.GONE)
            if (streamPausedShouldStayShown.not()) pvm.uiViewModel.streamPausedVisibility.postValue(View.GONE)
        }
    }

    private fun startSeeking() {
        showControls(true)
        val streamDuration = player?.duration //expecting that during seeking stream window duration will not change
        pvm.uiViewModel.startSeekingPrepare()
        switchTrick(true)
        seeking = true
        fixedRateTimer(null, false, 200L, SEEK_UPDATE_INTERVAL.toLong()) {
            if (!seeking) {
                cancel()
                return@fixedRateTimer
            }
            seekCyclesPassed++
            Handler(Looper.getMainLooper()).post {
                if(isInTrick && (firstFrameRendered || seekCyclesPassed > seekCyclesPassedMax)) {
                    firstFrameRendered = false
                    pvm.uiViewModel.seekToSelectedPos(player, true)
                    seekCyclesPassed = 0
                }
                seekMultiplier = pvm.uiViewModel.seekTick(streamDuration, seekIncrement, seekMultiplier)
                if (seekMultiplier == 0 && seeking) {
                    seekToSelectedPos()
                    ((view?.context as? FragmentActivity) as MainActivity).setRemoteControlStateAfterSeekToLive()
                }
            }
        }
    }

    private fun setSeekData() {
        val multiplierText = if (seekMultiplier == 0) "" else "x${2f.pow(1 + seekMultiplier.absoluteValue).toInt()}"
        pvm.uiViewModel.applySeekMultiplier(seekMultiplier)
        pvm.uiViewModel.seekText.postValue(multiplierText)
        seekIncrement = getIncrement(seekMultiplier, SEEK_UPDATE_INTERVAL)
    }

    @Suppress("SameParameterValue")
    private fun getIncrement(multiplier: Int, interval: Int): Int {
        if (multiplier == 0) return 0
        val seekTime = 2f.pow(1 + seekMultiplier.absoluteValue) * 1000
        val ticksPerSecond = 1000 / interval
        return (seekTime / ticksPerSecond).toInt() * (if (multiplier > 0) 1 else -1)
    }

    override fun isUIVisible() = pvm.uiView.isUIVisible()

    private fun doSeekToLive() {
        if (Flavor().useDifferentUrlForStartOver) {
            lastPlayedStream?.let {
                streamRestartReason = StreamRestartReason.LIVE
                onDataPrepared(it, PlayerLocation.UNCHANGED)
            }
        } else {
            player?.let { it.seekTo(it.duration) }
        }
    }

    private fun doSeekToStartOver() {
        when (streamingType) {
            StreamingType.CHANNEL -> {
                if (Flavor().useDifferentUrlForStartOver && !isInStartOverMode()) {
                    lastPlayedStream?.let {
                        streamRestartReason = StreamRestartReason.START_OVER
                        onDataPrepared(it, PlayerLocation.UNCHANGED)
                    }
                } else {
                    val seekBackFromLive = TimeProvider.nowMs() - (pvm.uiViewModel.getPlayingEventStart())
                    player?.duration?.also {
                        if (seekBackFromLive > it) player?.seekTo(0)
                        else player?.seekTo(it - seekBackFromLive) //seek one minute before event start
                    }
                }
            }
            StreamingType.CATCHUP_IN_LIVE_PLAYER,
            StreamingType.EVENT,
            StreamingType.VOD -> player?.seekTo(0)
        }
    }

    private fun confirmSeek(toLive: Boolean) {
        pause(true)
        val messageDialogModel = MessageDialogModel(
                when (toLive) {
                    true -> resources.getString(R.string.player_seek_confirmation_to_live)
                    false -> resources.getString(R.string.player_seek_confirmation_to_startover)
                },
                null,
                arrayOf(resources.getString(R.string.player_seek_confirmation_yes)),
                resources.getString(R.string.player_seek_confirmation_no),
                MessageDialogCodes.confirmSeek
        )
        val confirmMessageDialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        confirmMessageDialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                if (answer.type == OPTION_A) {
                    when (toLive) {
                        true -> doSeekToLive()
                        false -> doSeekToStartOver()
                    }
                }
                resume()
                return false
            }
        }
        (view?.context as? FragmentActivity)?.supportFragmentManager?.also {
            confirmMessageDialogFragment.show(
                    it,
                    "tag_play_failed_dialog_fragment"
            )
        }
    }

    private fun displayToast(message: Int) {
        if (context == null) return
        val offset = pvm.uiView.getToastBottomOffset()
        TvLauncherToast.makeText(context as Context, message, Toast.LENGTH_SHORT)?.apply {
            offset.also { setGravity(Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL, 0, it) }
            show()
        }
    }

    private fun displayToast(message: String) {
        if (context == null) return
        val offset = pvm.uiView.getToastBottomOffset()
        TvLauncherToast.makeText(context as Context, message, Toast.LENGTH_SHORT)?.apply {
            offset.also { setGravity(Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL, 0, it) }
            show()
        }
    }

    private fun checkBinding(method: String): Boolean {
        if(nullableBinding == null) {
            Timber.tag(TAG_PLAYER_ERROR_LOG).e("Binding is null in PlayerFragment in $method method")
            return false
        }
        return true
    }

    private fun internalPlayChannel(location: PlayerLocation) {
        if(!checkBinding("internalPlayChannel")) return
        streamingType = StreamingType.CHANNEL
        binding.exoplayerView.controls = getPlayerUI()
        if (!pvm.uiViewModel.playingEventParentalLD.hasActiveObservers()) {
            setupActiveEventObserverForParentalContent()
        }
        ConnectionMessagesHelper.checkConnectionAndShowMessage(context)
        showControls(true)
        playerViewModel.detail.value?.also {
            val channel = playerViewModel.channel.value
            channel.ifNull { if (Flavor().allowChannelCacheReset) resetCachedChannels() }
            pvm.uiViewModel.observeMediaStreamLD(
                    detail = it,
                    channel = channel,
                    doBlock = { stream -> onDataPrepared(stream, location, startedByUser = true) },
                    catchBlock = {
                        if (isStateSaved) return@observeMediaStreamLD true
                        launch(Dispatchers.Main) {
                            failRetries++
                            if (failRetries >= FAIL_MAX_RETRIES) {
                                Timber.tag(TAG_PLAYER_LOG).d( "LIVE MAX retried")
                                showPlayFailOverlay()
                            } else {
                                Timber.tag(TAG_PLAYER_LOG).d("LIVE $failRetries retried")
                                internalPlayChannel(location)
                            }
                        }
                        return@observeMediaStreamLD false
                    },
                    lifecycleOwner = viewLifecycleOwner
            )
        }
    }

    private fun resetCachedChannels() {
        GlobalScope.launch(Dispatchers.IO) {
            com.twentyfouri.tvlauncher.common.Flavor().clearEpgData(context, get())
        }
    }

    private fun setupRecordingObserver() {
        (activity as? MainActivity)?.getBindingSafe()?.viewModel?.recordingInPlayerClicked?.observe(
            viewLifecycleOwner
        ) {
            (pvm.uiViewModel as? PlayerUILiveViewModel)?.startStopRecording(context as LifecycleOwner)
            showControls()
        }
    }

    private fun setupActiveEventObserverForParentalContent() {
        pvm.uiViewModel.playingEventParentalLD.observe(viewLifecycleOwner, Observer {
            RestrictionChecker.dismissRestrictionDialog()
            val doOnSuccess: () -> Unit = {
                inPlayerAction = false
                pinDialogsShown = 0
                binding.exoplayerView.disablePlayerOverLay()
                pvm.uiViewModel.playRestrictedVisibility.postValue(View.GONE)
                RestrictionChecker.dismissRestrictionDialog()
                if (isInErrorState().not() && player?.isPlaying == false && playerInfoUpdated) {
                    resume()
                }
                if (isInErrorState()
                    && view != null
                    && pvm.uiViewModel.subscriptionChannel.value == null
                    && streamPausedShouldStayShown.not()) {
                        failAutoRetryTimer?.cancel()
                        retryPlay(true)
                }
            }
            val doOnRestrictionBlock = {
                if (player?.isPlaying == true) pause(false)
            }
            val goToNextChannel: () -> Unit = {
                streamPausedShouldStayShown = false
                player?.stop()
                binding.exoplayerView.subtitleView.setCues(null)
                resetStreamTimers()
                RestrictionChecker.dismissRestrictionDialog()
                pinDialogsShown = 0
                binding.exoplayerView.disablePlayerOverLay()
                checkRemoteStateAfterReturnFromSubscriptions()
                getNextChannelReference()?.also {
                    playFromReference(it, PlayerLocation.UNCHANGED)
                }
            }
            val goToPreviousChannel: () -> Unit = {
                streamPausedShouldStayShown = false
                player?.stop()
                binding.exoplayerView.subtitleView.setCues(null)
                resetStreamTimers()
                RestrictionChecker.dismissRestrictionDialog()
                pinDialogsShown = 0
                binding.exoplayerView.disablePlayerOverLay()
                checkRemoteStateAfterReturnFromSubscriptions()
                getPreviousChannelReference()?.also {
                    playFromReference(it, PlayerLocation.UNCHANGED)
                }
            }
            val goToLastPlayedChannel = {
                checkRemoteStateAfterReturnFromSubscriptions()
                binding.exoplayerView.disablePlayerOverLay()
                switchToLastChannel()
            }

            //should this be same as inPlayerAction ?
            pinDialogsShown++
            binding.exoplayerView.enablePlayerOverLay()
            playerViewModel.channel.value?.channelNumber?.let {
                SharedPreferencesUtils.putChannelNumber(it)
            }
            RestrictionChecker.checkRestrictions(
                    context = requireContext(),
                    mediaItem = it,
                    isInPlayerMode = true,
                    doOnSuccess = doOnSuccess,
                    doOnRestrictionBlock = doOnRestrictionBlock,
                    goToNextChannel = goToNextChannel,
                    goToPreviousChannel = goToPreviousChannel,
                    goToLastPlayedChannel = goToLastPlayedChannel,
                    useButtonDialog = isInitialPlayback
            )
        })
    }

    //this method is needed only for envision
//    private fun checkPinAndPlay(stream: SmartMediaStream, location: PlayerLocation, mediaItem: SmartMediaItem?) {
//        val doOnSuccess: () -> Unit = {
//            inPlayerAction = false
//            onDataPrepared(stream, location)
//        }
//        val goToNextChannel: () -> Unit =
//                { getNextChannelReference()?.also { playFromReference(it, PlayerLocation.UNCHANGED) } }
//        val goToPreviousChannel: () -> Unit =
//                { getPreviousChannelReference()?.also { playFromReference(it, PlayerLocation.UNCHANGED) } }
//        val goToLastPlayedChannel = { playLastPlayedChannel() }
//
//        if (inPlayerAction) {
//            RestrictionChecker.checkRestrictions(
//                    context = requireContext(),
//                    mediaItem = mediaItem,
//                    doOnSuccess = doOnSuccess,
//                    isInPlayerMode = true,
//                    goToNextChannel = goToNextChannel,
//                    goToPreviousChannel = goToPreviousChannel,
//                    goToLastPlayedChannel = goToLastPlayedChannel
//            )
//        } else {
//            playerViewModel.channel.removeObservers(viewLifecycleOwner)
//            doOnSuccess.invoke()
//        }
//    }

    private fun internalPlayCatchup(location: PlayerLocation, isSoftZap: Boolean, position: Int? = null) {
        if(!checkBinding("internalPlayCatchup")) return
        if (isSoftZap && pvm.uiViewModel.isSoftZappedPlayableCatchup().not()) {
            showUnavailableActionToast()
            return
        }
        if (pvm.uiViewModel.isCatchupSubscribed.hasActiveObservers().not()) {
            setupCatchupSubscriptionObserver()
        }
        streamingType = if (isSoftZap) StreamingType.CATCHUP_IN_LIVE_PLAYER else StreamingType.EVENT
        binding.exoplayerView.controls = getPlayerUI()

        playerViewModel.detail.value?.also { detail ->
            val doOnSuccess = {
                pvm.uiViewModel.observeMediaStreamLD(
                        detail = detail,
                        channel = playerViewModel.channel.value,
                        doBlock = { stream ->
                            val doOnSuccess: () -> Unit = {
                                onDataPrepared(stream, location, position, startedByUser = true)
                            }
                            RestrictionChecker.checkRestrictions(
                                    context = requireContext(),
                                    mediaItem = detail,
                                    isInPlayerMode = true,
                                    doOnSuccess = doOnSuccess,
                                    goToLastPlayedChannel = { playLastPlayedChannel() }
                            )
                        },
                        catchBlock = {
                            launch(Dispatchers.Main) {
                                failRetries++
                                if (failRetries >= FAIL_MAX_RETRIES) {
                                    Timber.tag(TAG_PLAYER_LOG).d("CATCHUP MAX retried")
                                    if (Flavor().handleInPlayerMessageCatchupNotAvailableYet) {
                                        val now = TimeProvider.nowMs()
                                        val end = detail.endDate
                                        val ongoing = end?.isAfter(now - Flavor().maxExpectedCatchupEncodingTimeMs) == true && end.isBefore(TimeProvider.nowMs())
                                        val catchupInLive = streamingType == StreamingType.CATCHUP_IN_LIVE_PLAYER

                                        //if it seems that catchup is not available yet because it is still encoding on server side display message with "...YET"
                                        //otherwise display common fail error message
                                        val messageId = when {
                                            ongoing && catchupInLive -> R.string.play_failed_catchup_not_available_yet //currently same for both
                                            !ongoing && catchupInLive -> R.string.play_failed_channel
                                            ongoing && !catchupInLive -> R.string.play_failed_catchup_not_available_yet //currently same for both
                                            !ongoing && !catchupInLive -> R.string.play_failed_catchup
                                            else -> R.string.play_failed_catchup //default
                                        }
                                        pvm.uiViewModel.playFailedText.postValue(getString(messageId))
                                    }
                                    showPlayFailOverlay()
                                } else {
                                    Timber.tag(TAG_PLAYER_LOG).d( "CATCHUP $failRetries retried")
                                    internalPlayCatchup(location, isSoftZap)
                                }
                            }
                            return@observeMediaStreamLD false
                        },
                        lifecycleOwner = viewLifecycleOwner,
                        isPlayingCatchup = isSoftZap
                )
            }
            RestrictionChecker.checkRestrictions(
                    context = requireContext(),
                    mediaItem = detail,
                    isInPlayerMode = true,
                    doOnSuccess = { doOnSuccess.invoke() },
                    goToLastPlayedChannel = { playLastPlayedChannel() }
            )
        }
    }

    private fun setupCatchupSubscriptionObserver() {
        pvm.uiViewModel.isCatchupSubscribed.observe(viewLifecycleOwner, { status ->
            if(status == false) {
                pvm.uiViewModel.isCatchupSubscribed.value = true
                displayToast(R.string.play_failed_subscription_toast)
            }
        })
    }

    private fun checkRemoteStateAfterReturnFromSubscriptions() {
        if (pvm.uiViewModel.subscriptionChannel.value != null) {
            (context as ActivityPlayerAction).checkRemoteStateAfterReturnFromSubscriptions()
        }
    }

    private fun internalPlayVod(location: PlayerLocation, position: Int? = null) {
        if(!checkBinding("internalPlayVod")) return
        streamingType = StreamingType.VOD
        binding.exoplayerView.controls = getPlayerUI()

        playerViewModel.detail.value?.also {
            pvm.uiViewModel.observeMediaStreamLD(
                    detail = it,
                    channel = null,
                    doBlock = { stream -> onDataPrepared(stream, location, position, startedByUser = true) },
                    catchBlock = {
                        launch(Dispatchers.Main) {
                            failRetries++
                            if (failRetries >= FAIL_MAX_RETRIES) {
                                Timber.tag(TAG_PLAYER_LOG).d("CATCHUP MAX retried")
                                showPlayFailOverlay()
                            } else {
                                Timber.tag(TAG_PLAYER_LOG).d("CATCHUP $failRetries retried")
                                internalPlayVod(location)
                            }
                        }
                        return@observeMediaStreamLD false
                    },
                    lifecycleOwner = viewLifecycleOwner,
                    isPlayingCatchup = false
            )
        }
    }

    private fun getPlayerUI(): PlayerUIBase {
        pvm.uiViewModel.stopTimer()
        pvm.uiView.timerHelper.removeListener(this)
        when (streamingType) {
            StreamingType.CATCHUP_IN_LIVE_PLAYER,
            StreamingType.CHANNEL -> pvm.setCurrentUIView(requireContext(), PlayerUIEnum.LIVE)
            StreamingType.VOD -> pvm.setCurrentUIView(requireContext(), PlayerUIEnum.VOD)
            StreamingType.EVENT -> pvm.setCurrentUIView(requireContext(), PlayerUIEnum.CATCHUP)
        }
        //TODO assign timeHelper listener after new player is constructed
        //currently it is probably? causing displaying progress of old stream in new controls
        //also try to clean values upon removeListener
        pvm.uiView.timerHelper.addListener(this)
        pvm.uiViewModel.startTimer()
        playerViewModel.detail.observe(viewLifecycleOwner) {
            pvm.uiViewModel.setDetail(it)
        }
        setupSubscriptionObserver()
        setupYouboraInfoObservers()
        return pvm.uiView
    }

    //method used only when back button on pin dialog pressed
    private fun playLastPlayedChannel() {
        pvm.uiViewModel.getLastChannelReference(
            streamingType == StreamingType.EVENT || streamingType == StreamingType.CATCHUP_IN_LIVE_PLAYER
        )?.also { channelReference ->
            if (pinDialogsShown > 1) {
                onPlayerEnded()
                pinDialogsShown = 0
            } else {
                inPlayerAction = true
                playFromReference(channelReference, PlayerLocation.UNCHANGED)
            }
        }.ifNull {
            onPlayerEnded()
            pinDialogsShown = 0
        }
    }

    override fun startIntent(intent: Intent) {
        //this for some users throw Fatal Exception: android.content.ActivityNotFoundException in case of Netflix.
        //issue is not reproducible. Try/catch added to avoid crash
        try {
            this.startActivity(intent)
        } catch (e: ActivityNotFoundException) {
            Timber.tag(TAG_PLAYER_ERROR_LOG).w(e.stackTraceToString())
        }
    }

    override fun getPreviousChannelIdTo(channelReference: SmartMediaReference): String? {
        return pvm.uiViewModel.getPreviousChannelIdTo(channelReference)
    }

    override fun getNextChannelIdTo(channelReference: SmartMediaReference): String? {
        return pvm.uiViewModel.getNextChannelIdTo(channelReference)
    }

    private fun setupAndStartTimer() {
        FirebaseRemoteConfigHelper.fetchAndActivate(
            doOnSuccess = {
                val remoteTimePeriod =
                    firebaseRemoteConfig.getLong("continue_watching_dialog_period")
                val timePeriod = if (remoteTimePeriod > 0) remoteTimePeriod else 150
                initStreamTimer(timePeriod)
                streamTimer?.start()
            },
            defaultFallback = {
                initStreamTimer(150)
                streamTimer?.start()
            }
        )
    }

    private fun initStreamTimer(timePeriod: Long) {
        streamTimer = object :
                CountDownTimer(timePeriod/*minutes*/ * 60/*seconds*/ * 1000/*millis*/, 1000) {
            override fun onTick(millisUntilFinished: Long) {
//                Log.d("bandwidth", "PlayerFragment ${player?.bandwidthMeter?.bitrateEstimate}")
            }

            override fun onFinish() {
                continueWatchingDialogFragment?.dismiss()
                continueWatchingDialogFragment = messageDialogModel?.let { MessageDialogFragment.newInstance(it) }
                continueWatchingDialogFragment?.mDismissListener =
                        MessageDialogDismissListener.from {
                            confirmWatchingTimer.cancel()
                            if (!pvm.uiViewModel.errorState && isPlayingOrShouldResume()) {
                                this.start()
                            }
                            continueWatchingDialogFragment = null
                        }
                if (isAdded) {
                    if (!isStateSaved) {
                        continueWatchingDialogFragment?.show(
                                parentFragmentManager,
                                "continue_watching"
                        )
                    }
                    confirmWatchingTimer.start()
                }
            }
        }
    }

    private fun checkPlayingLiveEvent(): Boolean {
        return getPlayerUIView().viewModel is PlayerUILiveViewModel
                && getPlayerUIView().viewModel.playingEvent.value == null
    }
    //endregion
}
