package com.twentyfouri.tvlauncher.ui

import android.content.Intent
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.tvlauncher.adapters.ColumnItemAdapter
import com.twentyfouri.tvlauncher.data.ColumnItem
import com.twentyfouri.tvlauncher.databinding.FragmentRecordingsEpisodesBinding
import com.twentyfouri.tvlauncher.viewmodels.RecordingsEpisodesViewModel
import com.twentyfouri.tvlauncher.widgets.RecordingItemView
import org.koin.androidx.viewmodel.ext.android.getViewModel

class RecordingsEpisodesFragment : BaseFragment() {

    private var binding: FragmentRecordingsEpisodesBinding? = null
    private var parentColumnItem: ColumnItem? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentRecordingsEpisodesBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@RecordingsEpisodesFragment.viewLifecycleOwner
        binding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val viewModelFromFactory = getViewModel<RecordingsEpisodesViewModel>()

        parentColumnItem?.let { viewModelFromFactory.setParentColumnItem(it) }

        binding?.viewModel = viewModelFromFactory

        val columnItemAdapter = ColumnItemAdapter()
        binding?.recordingsEpisodesRecycler?.adapter = columnItemAdapter
        setObserverOnColumnItems()

        binding?.recordingsEpisodesRecycler?.requestFocus()
    }

    override fun onDestroyView() {
        binding = null
        super.onDestroyView()
    }

    private fun setChildColumnItemsObserver() {
        binding?.viewModel?.childColumnItems?.observe(viewLifecycleOwner, Observer { columnItemList ->
            if (columnItemList.isEmpty().not()) {
                (binding?.recordingsEpisodesRecycler?.adapter as? ColumnItemAdapter)?.apply {
                    submitList(columnItemList)
                    notifyDataSetChanged()
                }
            }
        })
    }

    private fun setObserverOnColumnItems() {
        binding?.recordingsEpisodesRecycler?.addOnChildAttachStateChangeListener(object : RecyclerView.OnChildAttachStateChangeListener {
            override fun onChildViewAttachedToWindow(view: View) {
                val recordingItemViewModel = (view as? RecordingItemView)?.binding?.viewModel ?: return
                recordingItemViewModel.onItemClicked.observe(viewLifecycleOwner, Observer {
                    it?.let { onItemClicked(it) }
                })
            }

            override fun onChildViewDetachedFromWindow(view: View) {
                val recordingItemViewModel = (view as? RecordingItemView)?.binding?.viewModel ?: return
                recordingItemViewModel.onItemClicked.removeObservers(viewLifecycleOwner)
            }
        })
    }

    private fun onItemClicked(item: ColumnItem) {
        val intent = Intent(this.requireContext(), MainActivity::class.java)
        intent.action = MainActivity.INTENT_ACTION_DETAIL
        intent.putExtra(MainActivity.EXTRA_OPEN_USER_PROFILE, true)
        intent.putExtra("reference",item.recording?.reference)
        startActivity(intent)
    }

    fun bind(item: ColumnItem) {
        parentColumnItem = item
        binding?.viewModel?.setParentColumnItem(item)
    }

    override fun onPause() {
        binding?.viewModel?.childColumnItems?.removeObservers(viewLifecycleOwner)
        super.onPause()
    }

    override fun onResume() {
        super.onResume()
        binding?.viewModel?.refreshRecordings(viewLifecycleOwner)
        setChildColumnItemsObserver()
    }

}