package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.constraintlayout.widget.ConstraintSet
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.model.dashboard.SmartPageReference
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.databinding.FragmentTopbarBinding
import com.twentyfouri.tvlauncher.extensions.computeLeftEdgeForVerticalAlignementWith
import com.twentyfouri.tvlauncher.utils.StringsMapper
import com.twentyfouri.tvlauncher.viewmodels.TopbarViewModel
import com.twentyfouri.tvlauncher.widgets.FocusHandleFrameLayout
import org.koin.androidx.scope.ScopeFragment
import org.koin.androidx.viewmodel.ext.android.getViewModel

open class TopbarFragment : ScopeFragment() {

    open lateinit var binding : FragmentTopbarBinding
    private var pageReferenceToSelect: SmartPageReference? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentTopbarBinding.inflate(
        inflater,
        container,
        false)
    .apply {
        lifecycleOwner = this@TopbarFragment.viewLifecycleOwner
        binding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        try {
            super.onViewCreated(view, savedInstanceState)
        } catch (ise: IllegalStateException) {
            // https://github.com/InsertKoinIO/koin/issues/1433
        }

        val viewModelFromFactory = getViewModel<TopbarViewModel>()
        viewModelFromFactory.setIsOffline(requireArguments().getBoolean(ARG_OFFLINE))

        binding.viewModel = viewModelFromFactory

        setupUnSelectingOfMenuItems(viewModelFromFactory)
        setupSubtextOfMenuItems(viewModelFromFactory)

        binding.topbarFocusHandleLayout.disablePostCallingFocus = true

        binding.topbarFocusHandleLayout.focusRememberListener = object : FocusHandleFrameLayout.FocusRememberListener {
            override fun provideChildToBeFocused(child: View?, focused: View?, lastFocusedView: View?): View? {
                val menuLeft = binding.viewModel?.menuLeft?.value
                if (menuLeft != null) {
                    for (menuItem in menuLeft) {
                        if (menuItem.model.isSelected.value == true) {
                            return menuItem.view
                        }
                    }
                }
                return lastFocusedView
            }
        }
    }

    //TODO do this via the LiveData observing each other
    private fun setupSubtextOfMenuItems(_viewModel: TopbarViewModel) {
        _viewModel.menuRight.observe(viewLifecycleOwner, Observer {
            it.forEach { viewHolder ->
                viewHolder.model.isFocused.observe(viewLifecycleOwner, Observer {isFocused ->
                    binding.subtext.visibility = if (isFocused) View.VISIBLE else View.INVISIBLE
                    if (isFocused) {
                        binding.subtext.text = StringsMapper.translate(viewHolder.model.title)

                        val constraintSet = ConstraintSet()
                        constraintSet.clone(binding.topbarViewgroup)
                        constraintSet.connect(
                            binding.subtext.id,
                            ConstraintSet.LEFT,
                            ConstraintSet.PARENT_ID,
                            ConstraintSet.LEFT,
                            binding.menuRight.left + binding.subtext.computeLeftEdgeForVerticalAlignementWith(viewHolder.view)
                        )
                        constraintSet.applyTo(binding.topbarViewgroup)
                    }
                })
            }
        })
    }

    //TODO do this via the LiveData observing each other
    private fun setupUnSelectingOfMenuItems(_viewModel: TopbarViewModel) {
        _viewModel.menuLeft.observe(viewLifecycleOwner, Observer { menuItemViewHolders ->
            selectByPageReference(pageReferenceToSelect)
            menuItemViewHolders.forEach { menuItemViewHolder ->
                menuItemViewHolder.model.isFocused.observe(viewLifecycleOwner, Observer { isFocused ->
                    if (isFocused)
                        _viewModel.menuLeft.value
                            ?.filterNot { it == menuItemViewHolder }
                            ?.forEach { it.model.setIsFocused(false) }
                })
            }
        })
    }

    fun selectByPageReference(pageReference: SmartPageReference?, forceFocus: Boolean = false) {
        pageReferenceToSelect = pageReference
        if (::binding.isInitialized) {
            binding.viewModel?.menuLeft?.value?.let { menuLeft ->
                //it is necessary to unselect all first, otherwise it blinks
                for (menuItem in menuLeft) {
                    menuItem.model.setIsSelected(false)
                }
                for (menuItem in menuLeft) {
                    menuItem.model.target?.pageReference.let {
                        menuItem.model.setIsSelected(it == pageReference)
                        if (it == pageReference && forceFocus) menuItem.view?.requestFocus()
                    }
                }
            }
        }
    }

    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        if (hidden) {
            binding.topbarFocusHandleLayout.descendantFocusability = ViewGroup.FOCUS_BLOCK_DESCENDANTS
        } else {
            binding.topbarFocusHandleLayout.descendantFocusability = ViewGroup.FOCUS_AFTER_DESCENDANTS
        }
    }

    fun selectByPageType(pageType: PageType, forceFocus: Boolean = false) {
        val pageReference = Flavor().getPageReference(pageType, binding.viewModel?.menus?.value)
        selectByPageReference(pageReference, forceFocus)
    }

    companion object {
        const val ARG_OFFLINE = "ARG_OFFLINE"

        fun newInstance(offline: Boolean = false): TopbarFragment {
            val args = Bundle(1)
            args.putBoolean(ARG_OFFLINE, offline)
            val fragment = TopbarFragment()
            fragment.arguments = args
            return fragment
        }
    }
}