package com.twentyfouri.tvlauncher.utils

import android.os.Bundle
import android.util.Log
import androidx.fragment.app.commit
import com.google.android.exoplayer2.Format
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.androidcore.utils.ResourceImageSpecification
import com.twentyfouri.smartexoplayer.tracks.TrackInfo
import com.twentyfouri.smartexoplayer.tracks.TrackPreference
import com.twentyfouri.smartexoplayer.tracks.TrackPreferences
import com.twentyfouri.smartexoplayer.tracks.VideoPreference
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.ui.ListPickerFragment
import com.twentyfouri.tvlauncher.ui.MainActivity
import com.twentyfouri.tvlauncher.ui.PlayerFragment
import com.twentyfouri.tvlauncher.ui.TrackType
import com.twentyfouri.tvlauncher.widgets.PlayerUICatchup
import com.twentyfouri.tvlauncher.widgets.PlayerUILive
import com.twentyfouri.tvlauncher.widgets.PlayerUIVod
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import timber.log.Timber

class PlayerSideMenu(private val playerFragment: PlayerFragment) {

    private enum class PlayerSideMenuItemID {
        AUDIO,
        SUBTITLES,
    }

    internal var listFragment: ListPickerFragment? = null

    fun showSideMenu() {
        val data = ArrayList<ListPickerItem>()

        data.add(
            ListPickerItem(
                id = "",
                title = "",
                subtitle = "",
                label = playerFragment.getString(R.string.player_ok_longpress_action_audio_option),
                icon = null,
                type = SettingsItemType.SECTION_LABEL
            )
        )

        getAllTracks()?.filter { it.type == TrackInfo.TYPE_AUDIO }?.let {
            it.forEachIndexed { index, audioTrackInfo ->
                data.add(
                    ListPickerItem(
                        id = PlayerSideMenuItemID.AUDIO.toString(),
                        title = audioTrackInfo.name.capitalize(),
                        subtitle = "",
                        label = "",
                        icon = if (audioTrackInfo.selected || it.size == 1) ResourceImageSpecification(R.drawable.player_side_menu_selected)
                        else ResourceImageSpecification(R.drawable.player_side_menu_not_selected),
                        type = SettingsItemType.TITLE_WITH_ICON,
                        customValue = index
                    )
                )
            }
        }

        data.add(
            ListPickerItem(
                id = "",
                title = "",
                subtitle = "",
                label = playerFragment.getString(R.string.player_ok_longpress_action_subtitles_option),
                icon = null,
                type = SettingsItemType.SECTION_LABEL
            )
        )

        getAllTracks()?.filter { it.type == TrackInfo.TYPE_SUBTITLES }?.let {subtitleTracks->
            val subtitleTracksAll = subtitleTracks.toMutableList()
            subtitleTracksAll.add(TrackInfo(
                TrackInfo.TYPE_SUBTITLES,
                "none",
                Format.createSampleFormat("",""),
                playerFragment.subtitleSelection.first?.equals("none") == true))

            subtitleTracksAll.forEachIndexed { index, subtitleTrackInfo ->
                data.add(
                    ListPickerItem(
                        id = PlayerSideMenuItemID.SUBTITLES.toString(),
                        title = if(subtitleTrackInfo.name == "none") playerFragment.getString(R.string.player_ok_longpress_action_no_subtitles) else subtitleTrackInfo.name.capitalize(),
                        subtitle = "",
                        label = "",
                        icon = if (subtitleTrackInfo.selected || subtitleTracksAll.size == 1) ResourceImageSpecification(R.drawable.player_side_menu_selected)
                        else ResourceImageSpecification(R.drawable.player_side_menu_not_selected),
                        type = SettingsItemType.TITLE_WITH_ICON,
                        customValue = index
                    )
                )
            }
        }


        val title = when (playerFragment.getPlayerUIView()) {
            is PlayerUILive -> (playerFragment.getPlayerUIView() as PlayerUILive).viewModel.getPlayingEventTitleSafe()
            is PlayerUIVod -> (playerFragment.getPlayerUIView() as PlayerUIVod).binding.catchupTitle.text.toString()
            is PlayerUICatchup -> (playerFragment.getPlayerUIView() as PlayerUICatchup).binding.catchupTitle.text.toString()
            else -> ""
        }

        listFragment = ListPickerFragment.newInstance(
            title = title,
            defaultPosition = ListPickerFragment.DEFAULT_POSITION_TOP,
            data = data
        )

        listFragment?.let {
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Player side menu show: $title")
            Timber.tag("PlayerSideMenu.showSideMenu").d("commit $it")
            FirebaseCrashlytics.getInstance().log("PlayerSideMenu.showSideMenu: commit $it")
            playerFragment.parentFragmentManager.commit {
                add(R.id.player_frame_overlay, it)
                addToBackStack(null)
            }
        }
        (playerFragment.activity as MainActivity).remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_BEHIND_SIDE_MENU)
    }

    fun hideSideMenu() {
        listFragment?.let {
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Player side menu hide")
            Timber.tag("PlayerSideMenu.hideSideMenu").d("commit $it")
            FirebaseCrashlytics.getInstance().log("PlayerSideMenu.hideSideMenu: commit $it")
            playerFragment.parentFragmentManager.commit { remove(it) }
        }
    }

    fun onActivityResult(data: Bundle) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Player side menu hide")
        (playerFragment.activity as MainActivity).remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
        val listPickerItem = data.get(ListPickerFragment.BUNDLE_KEY_RESULT) as? ListPickerItem
        if (listPickerItem != null) {
            when (listPickerItem.id) {
                PlayerSideMenuItemID.AUDIO.toString() -> {
                    val index = listPickerItem.customValue
                    getAllTracks()?.filter { it.type == TrackInfo.TYPE_AUDIO }?.let {
                        if (index < it.size) {
                            val trackPreferences = TrackPreferences(
                                1F,
                                VideoPreference.DEFAULT,
                                TrackPreference.forTrackInfo(it[index]),
                                playerFragment.getPlayer()?.trackSelector?.trackPreferences?.subtitles ?: TrackPreference.DEFAULT
                            )
                            playerFragment.getPlayer()?.trackSelector?.trackPreferences = trackPreferences
                            playerFragment.storeTrackSelection(TrackType.AUDIO, it[index].name, it[index].language)
                        }
                    }
                }
                PlayerSideMenuItemID.SUBTITLES.toString() -> {
                    val index = listPickerItem.customValue
                    getAllTracks()?.filter { it.type == TrackInfo.TYPE_SUBTITLES }?.let { subtitleTracks->
                        val subtitleTracksAll = subtitleTracks.toMutableList()
                        subtitleTracksAll.add(TrackInfo(TrackInfo.TYPE_SUBTITLES,"none",Format.createSampleFormat("",""),playerFragment.subtitleSelection.first?.equals("none") == true))
                        if (index < subtitleTracksAll.size) {
                            val trackPreferences = TrackPreferences(
                                1F,
                                VideoPreference.DEFAULT,
                                playerFragment.getPlayer()?.trackSelector?.trackPreferences?.audio ?: TrackPreference.DEFAULT,
                                if (subtitleTracksAll[index].name == "none") TrackPreference.DISABLED else TrackPreference.forTrackInfo(subtitleTracksAll[index])
                            )
                            playerFragment.getPlayer()?.trackSelector?.trackPreferences = trackPreferences
                            playerFragment.storeTrackSelection(TrackType.SUBTITLES, subtitleTracksAll[index].name, subtitleTracksAll[index].language)
                        }
                    }
                }
            }
        }
    }

    private fun getAllTracks() = playerFragment.getPlayer()?.let { it.trackSelector.getTrackInfos(it) }
}
