package com.twentyfouri.tvlauncher.utils

import android.app.Activity
import android.content.Intent
import android.os.Handler
import android.os.Looper
import android.view.KeyEvent
import android.view.KeyEvent.*
import com.twentyfouri.smartmodel.epg.EpgDatabase
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.media.SmartPlayerEventType
import com.twentyfouri.smartmodel.model.media.SmartSeekingRuleType
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.utils.SharedPreferencesUtils
import com.twentyfouri.tvlauncher.ui.EpgFragment
import com.twentyfouri.tvlauncher.ui.MainActivity
import com.twentyfouri.tvlauncher.ui.PlayerAction
import com.twentyfouri.tvlauncher.ui.PlayerLocation
import com.twentyfouri.tvlauncher.ui.actions.ActivityEpgAction
import com.twentyfouri.tvlauncher.ui.actions.ActivityPlayerAction
import com.twentyfouri.tvlauncher.widgets.SettingsItemView
import kotlinx.coroutines.*
import org.koin.core.component.KoinComponent
import java.util.*
import kotlin.concurrent.timer

enum class RemoteControlState {
    NORMAL,
    PLAYER_FOREGROUND,
    PLAYER_BACKGROUND,
    PLAYER_NUMBER_ZAPP,
    PLAYER_SEEKING,
    PLAYER_BEHIND_SUBSCRIPTION,
    PLAYER_BEHIND_SIDE_MENU,
    APP_NOT_ALLOWED,
    PLAYER_OFFLINE_MODE,
    OFFLINE_MODE
}

enum class KeyEventAction {
    BACK,
    STOP,
    PLAY_PAUSE,
    RW,
    FF,
    CHANNEL_DOWN,
    CHANNEL_UP,
    DPAD_LEFT,
    DPAD_RIGHT,
    DPAD_DOWN,
    DPAD_UP,
    DPAD_CENTER,
    NUMBER,
    LAST,
    STARTOVER,
    INFO,
    RECORD,
    LIVE,
    DIAGNOSIS,
    APPS,
    AUDIO_SUBTITLES,
    NO_ACTION
}

enum class RemoteKeysFilter {
    WHITELIST,
    BLACKLIST
}

data class RemoteControlKeys(val remoteKeysFilter: RemoteKeysFilter, val keyCodes: MutableList<Int>)

class RemoteKeyManager(
    private val playerAction: PlayerAction,
    private val activityPlayerAction: ActivityPlayerAction,
    private val epgActivityAction: ActivityEpgAction,
    private val activityAction: MainActivityAction
): KoinComponent {

    companion object {
        private const val HARD_ZAPPING_DELAY: Long = 400
        private const val DEBUG_MODE = false //enable if you want to send custom keys via adb
        private const val KEY_REPEAT_DELAY: Long = 400
    }

    private var remoteControlState = RemoteControlState.NORMAL
    private var remoteControlKeys = getNormalRemoteControlKeys()
    private var pressedNumbers = ""
    private var numberZappTimer = Timer()
    private var downKeyCode: Int = -1 //used for prevent incomplete DOWN/UP key action
    private var downKeyCodeRepeat: Int = -1
    private var downKeyCodeRepeatTimestamp: Long = -1
    private var previousState: RemoteControlState = RemoteControlState.NORMAL
    private var channelSwitchJob: Job? = null

    fun setRemoteControlState(state: RemoteControlState) {
        if (remoteControlState == state) return
        downKeyCode = -1
        remoteControlState = state
        remoteControlKeys = when (remoteControlState) {
            RemoteControlState.NORMAL -> getNormalRemoteControlKeys()
            RemoteControlState.PLAYER_BACKGROUND -> getPlayerBackgroundRemoteControlKeys()
            RemoteControlState.PLAYER_FOREGROUND -> getPlayerForegroundRemoteControlKeys()
            RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION -> {
                playerAction.showControls()
                getPlayerBackgroundRemoteControlKeys()
            }
            RemoteControlState.PLAYER_NUMBER_ZAPP -> getPlayerNumberZappRemoteControlKeys()
            RemoteControlState.PLAYER_SEEKING -> getPlayerSeekingRemoteControlKeys()
            RemoteControlState.PLAYER_BEHIND_SIDE_MENU -> getPlayerBehindSideMenuRemoteControlKeys()
            RemoteControlState.APP_NOT_ALLOWED -> getAppNotAllowedRemoteControlKeys()
            RemoteControlState.PLAYER_OFFLINE_MODE -> getNormalRemoteControlKeys() // any key will cause app restart
            RemoteControlState.OFFLINE_MODE -> getOfflineModeRemoteControlKeys()
        }
    }

    fun isInRemoteControlState(remoteControlState: RemoteControlState): Boolean {
        return this.remoteControlState == remoteControlState
    }

    fun handleRemoteKey(event: KeyEvent, isGlobalKey: Boolean = false): Boolean {
        @Suppress("ConstantConditionIf")
        if (DEBUG_MODE && event.action == ACTION_DOWN) {
            //these key codes are just random picked to test reactions using e.g. "adb shell input keyevent 131", replace at will
            if (event.keyCode == KEYCODE_F1) {
                event.actionBack()
                return true
            }
            if (event.keyCode == KEYCODE_F3) {
                event.actionLast()
                return true
            }
            if (event.keyCode == KEYCODE_F5) {
                runBlocking { EpgDatabase.getInstance(activityAction as Activity).epgDao.deleteAllEpgEvents() }
                return true
            }
            if (event.keyCode == KEYCODE_F6) {
                event.handleKeyEpg()
                return true
            }
            if (event.keyCode == KEYCODE_F7) {
                (activityAction as? MainActivity)?.also { act ->
                    val intent = Intent(act, MainActivity::class.java).apply {
                        action = MainActivity.INTENT_ACTION_MAIN
                        addCategory(MainActivity.INTENT_CATEGORY_HOME)
                    }
                    act.startActivity(intent)
                }
            }
            if (event.keyCode == KEYCODE_F8) {
                return event.actionApps()
            }
        }

        //debug player front/background
        if(event.action == ACTION_UP && event.keyCode == KEYCODE_GRAVE) {
            activityPlayerAction.togglePlayerFrontBack()
            return false
        }

        if(remoteControlState == RemoteControlState.PLAYER_OFFLINE_MODE && event.action == ACTION_UP) {
            SharedPreferencesUtils.putRestartCounter(SharedPreferencesUtils.getRestartCounter() + 1)
            SharedPreferencesUtils.putForceOffline(false)
            activityAction.restartActivity()
            return true
        }

        // always handle APPS, SEARCH, EPG and UNKNOWN key
        if (event.keyCode == KEYCODE_UNKNOWN) return event.handleKeyUnknown()
        if (Flavor().shouldHandleGuideButton() && (event.keyCode == KEYCODE_GUIDE)) return event.handleKeyEpg()
        if (event.keyCode == KEYCODE_ALL_APPS) return event.actionApps()
        if (event.keyCode == KEYCODE_SEARCH) return event.handleKeySearch() //TODO check what happen when player is currently playing

        return if (remoteControlKeys.remoteKeysFilter == RemoteKeysFilter.WHITELIST) {
            if (remoteControlKeys.keyCodes.any { it == event.keyCode })
                handleKey(event, isGlobalKey) //key is whitelisted so distribute it and return if somebody consumed it
            else
                true //key is NOT whitelisted so return like it is consumed
        } else {
            if (!remoteControlKeys.keyCodes.any { it == event.keyCode })
                handleKey(event, isGlobalKey) //key is not on blacklist so distribute and return if somebody consumed it
            else
                true //key is on blacklist so return like it is consumed
        }
    }

    private fun preventKeyRepeatOnLongPress(event: KeyEvent): Boolean {
        return if(event.action == ACTION_DOWN) {
            if(downKeyCodeRepeat == event.keyCode && TimeProvider.nowMs() - downKeyCodeRepeatTimestamp < KEY_REPEAT_DELAY) {
                true
            } else {
                downKeyCodeRepeat = event.keyCode
                downKeyCodeRepeatTimestamp = TimeProvider.nowMs()
                false
            }
        } else {
            downKeyCodeRepeat = -1
            downKeyCodeRepeatTimestamp = -1
            false
        }
    }

    //here we can specify in which situations and which keys should prevent repeating
    //consider RemoteControlState and KeyEventAction
    private fun shouldPreventKeyRepeatOnLongPress(event: KeyEvent): Boolean {
        if(!Flavor().shouldPreventKeyRepeatOnLongPress) return false
        return when (Flavor().getActionForKeyEvent(event)) {
            KeyEventAction.DPAD_LEFT,
            KeyEventAction.DPAD_RIGHT,
            KeyEventAction.DPAD_DOWN,
            KeyEventAction.DPAD_UP -> true
            else -> false
        }
    }

    private fun handleKey(event: KeyEvent, isGlobalKey: Boolean): Boolean {
        //check for repeating by long press
        if(shouldPreventKeyRepeatOnLongPress(event) && preventKeyRepeatOnLongPress(event)) return true

        // do not catch keys when state is NORMAL
        if (remoteControlState == RemoteControlState.NORMAL && event.isHandledInNormalMode().not())
            return false

        with(event) {
            if (action == ACTION_DOWN || isGlobalKey) {
                cancelScheduledChannelSwitch() //any key should do that
                downKeyCode = keyCode
            }
            if (action == ACTION_UP && downKeyCode != keyCode) return false //ignore incomplete DOWN/UP which can occur when remoteControlState changed

            when (Flavor().getActionForKeyEvent(event)) {
                KeyEventAction.BACK -> return actionBack()
                KeyEventAction.STOP -> return actionStop()
                KeyEventAction.PLAY_PAUSE -> return actionPlayPause()
                KeyEventAction.RW -> if (actionRewind()) return true else {}
                KeyEventAction.FF -> if (actionFastForward()) return true else {}
                KeyEventAction.CHANNEL_DOWN ->  return actionChannelDown()
                KeyEventAction.CHANNEL_UP -> return actionChannelUp()
                KeyEventAction.DPAD_LEFT -> if (actionDpadLeft()) return true else {}
                KeyEventAction.DPAD_RIGHT -> if (actionDpadRight()) return true else {}
                KeyEventAction.DPAD_DOWN ->  if (actionKeyDpadDown()) return true else {}
                KeyEventAction.DPAD_UP -> if (actionDpadUp()) return true else {}
                KeyEventAction.DPAD_CENTER -> return actionDpadCenter()
                KeyEventAction.NUMBER -> if (actionNumber()) return true else {}
                KeyEventAction.LAST -> if (actionLast()) return true else {}
                KeyEventAction.STARTOVER -> if (actionStartOver()) return true else {}
                KeyEventAction.INFO -> if (actionInfo()) return true else {}
                KeyEventAction.RECORD -> actionRecord()
                KeyEventAction.LIVE -> actionLive()
                KeyEventAction.DIAGNOSIS -> return actionShowDiagnosis()
                KeyEventAction.APPS -> return actionApps()
                KeyEventAction.AUDIO_SUBTITLES -> return actionShowAudioSubtitleSelection()
                KeyEventAction.NO_ACTION -> {}
            }
        }

        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            playerAction.cancelSeeking()
            setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
        }

        return false
    }

    private fun KeyEvent.actionShowAudioSubtitleSelection(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND || remoteControlState == RemoteControlState.PLAYER_BEHIND_SIDE_MENU) {
            if(action == ACTION_UP)
                playerAction.showOrHideSideMenu()
            }
        return true
    }

    private fun KeyEvent.actionApps(): Boolean {
        if (action == ACTION_UP) {
            Navigator.getInstance().navigateCustom(Navigator.NAVIGATE_CUSTOM_APPS_PAGE)
        }
        return true
    }

    private fun KeyEvent.actionShowDiagnosis(): Boolean {
        if (action == ACTION_UP) {
            Navigator.getInstance().navigate(SmartNavigationTarget(SmartNavigationAction.DIAGNOSTICS))
        }
        return true
    }

    private fun cancelScheduledChannelSwitch() {
        channelSwitchJob?.cancel()
        channelSwitchJob = null
    }

    private fun scheduleChannelSwitch() {
        //no need to cancel previous job as that was cancelled at the beginning of handleKey()
        channelSwitchJob = CoroutineScope(Dispatchers.Default).launch {
            delay(HARD_ZAPPING_DELAY)
            withContext(Dispatchers.Main) {
                playerAction.getSoftZapMediaReference()?.also {
                    switchPlayerToSoftZapRef(it)
                }
            }
        }
    }

    private fun onPressChannelUp() {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND ||
            remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION ) {

            if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION){
                playerAction.hideSubscriptions()
            }
            playerAction.handleKeyUp(isSoftZap = false)
            scheduleChannelSwitch()
        } else {
            epgActivityAction.scrollEPGPageUp()
        }
    }

    private fun onPressChannelDown() {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND ||
            remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION ) {

            if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION){
                playerAction.hideSubscriptions()
            }
            playerAction.handleKeyDown(isSoftZap = false)
            scheduleChannelSwitch()
        } else {
            epgActivityAction.scrollEPGPageDown()
        }
    }

    private fun KeyEvent.handleKeyUnknown(): Boolean {
        // TODO this is a temporary solution key events shouldn't be managed checking scan codes. They should use only key codes.
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_DOWN) {
                if (scanCode == 90) { // This scan code belongs to next channel button.
                    // TODO remove at the moment it is only managed by: KeyEvent.handleKeyChannelUp()
                    onPressChannelUp()
                    return true
                } else if (scanCode == 91) { // This scan code belongs to previous channel button.
                    // TODO remove at the moment it is only managed by: KeyEvent.handleKeyChannelDown()
                    onPressChannelDown()
                    return true
                }
            }
        }
        return false
    }

    private fun KeyEvent.handleKeyEpg(): Boolean {
        if (action == ACTION_UP) {
            val i = Intent()
            i.action = MainActivity.INTENT_ACTION_EPG
            activityAction.handleEpgButtonPress(i, remoteControlState.name)
        }
        return true
    }

    private fun KeyEvent.handleKeySearch(): Boolean {
        if (action == ACTION_UP) {
            Navigator.getInstance().navigate(SmartNavigationTarget.toSearch(""))
        }
        return true
    }

    private fun KeyEvent.isHandledInNormalMode(): Boolean {
        if (keyCode == KEYCODE_MEDIA_RECORD ||
                (keyCode in KEYCODE_NUMPAD_0..KEYCODE_NUMPAD_9) ||
                (keyCode in KEYCODE_0..KEYCODE_9) ||
                keyCode == KEYCODE_CHANNEL_DOWN ||
                keyCode == KEYCODE_CHANNEL_UP ||
                keyCode == KEYCODE_MEDIA_PLAY_PAUSE ||
                keyCode == KEYCODE_MEDIA_FAST_FORWARD ||
                keyCode == KEYCODE_MEDIA_REWIND ||
                keyCode == KEYCODE_TV ||
                keyCode == KEYCODE_F4 ||
                keyCode == KEYCODE_PROG_RED ||
                keyCode == KEYCODE_PROG_BLUE) // KEYCODE_F4 is current Entel keyCode for startover
            return true
        return false
    }

    private fun KeyEvent.actionRecord(): Boolean {
        if (action == ACTION_UP && Flavor().isRecordingAllowed) {
            if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
                activityAction.startStopRecordingInPlayer()
            } else {
                activityAction.startStopRecording()
            }
        }
        return true
    }

    private fun KeyEvent.actionInfo(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_UP)
                playerAction.showControls()
            return true
        }
        return false
    }

    private fun KeyEvent.actionStartOver(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_UP) {
                playerAction.seekToStartOver()
            }
            return true
        }
        if (action == ACTION_UP) {
            if (remoteControlState == RemoteControlState.NORMAL || remoteControlState == RemoteControlState.PLAYER_BACKGROUND) {
                return epgActivityAction.openPlayer(startFromBeginning = true)
            }
        }
        return false
    }

    private fun KeyEvent.actionLast(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND || remoteControlState == RemoteControlState.PLAYER_BACKGROUND) {
            if (action == ACTION_UP)
                playerAction.switchToLastChannel()
            return true
        }
        return false
    }

    private fun KeyEvent.actionNumber(): Boolean {
        if(!activityAction.isInOnlineMode()) return false
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND
            || remoteControlState == RemoteControlState.PLAYER_BACKGROUND
            || remoteControlState == RemoteControlState.PLAYER_SEEKING
            || remoteControlState == RemoteControlState.NORMAL
            || remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION
        ) {
            if (action == ACTION_UP) {
                previousState = remoteControlState
                playerAction.cancelSeeking()
                setRemoteControlState(RemoteControlState.PLAYER_NUMBER_ZAPP)
                zappNumberPressed(keyCodeToNumberString(keyCode))
            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed(keyCodeToNumberString(keyCode))
            return true
        }
        return false
    }

    private fun handleKeyDpadCenterWhenPlayerForeground() {
        playerAction.getSoftZapMediaReference()
            ?.also { switchPlayerToSoftZapRef(it) }
            .ifNull {
                playerAction.isInErrorState()
                    .ifTrue { playerAction.retryPlay() }
                    .ifElse {
                        (playerAction.isUIVisible() && Flavor().shouldHandleOkOnPlayerControls())
                            .ifTrue { playerAction.handleSecondOK() }
                            .ifElse { playerAction.showControls(false) }
                    }
            }
        if (playerAction.isSeeking()) playerAction.seekToSelectedPos()
    }

    private fun switchPlayerToSoftZapRef(softZapRef: SmartMediaReference) {
        val channelId: String = softZapRef.toString()
        val channelIsApp = Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(channelId)
        if (Flavor().bookmarkForConcurrencyHeartbeatInitDelay <= 0) {
            playerAction.sendBookmark(SmartPlayerEventType.STOP)
        }
        channelIsApp.ifTrue {
            val intent = Flavor().getAppChannelsDelegate()?.getSoftZappingAppChannelIntent(
                softZapRef, playerAction.getPreviousChannelReference().toString(),
                playerAction.getNextChannelReference().toString()
            )

            intent?.let { playerAction.startIntent(it) }
        }.ifElse {
            activityPlayerAction.playerPlay(PlayerLocation.UNCHANGED, softZapRef, true)
        }
    }

    private fun KeyEvent.actionDpadCenter(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (Flavor().shouldHandleLongPress() && isLongPress) {
                playerAction.showSideMenu()
            }
            if (action == ACTION_UP) {
                handleKeyDpadCenterWhenPlayerForeground()
            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("E")
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (Flavor().shouldHandleLongPress() && isLongPress) {
                playerAction.showSideMenu()
            }
            if (action == ACTION_UP) {
                playerAction.seekToSelectedPos()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SIDE_MENU){
            //Prevent other fragments consume ok button when side menu is opened
            activityAction.findCurrentFocus()?.let { return it !is SettingsItemView }
        }
        return false
    }

    private fun KeyEvent.actionDpadRight(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            playerAction.handleKeyRight(action)
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("C")
            return true
        }
        return false
    }

    private fun KeyEvent.actionDpadLeft(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            playerAction.handleKeyLeft(action)
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("C")
            return true
        }
        return false
    }

    private fun KeyEvent.actionDpadUp(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_DOWN)
                playerAction.handleKeyUp()
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("C")
            return true
        }
        return false
    }

    private fun KeyEvent.actionKeyDpadDown(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_DOWN)
                playerAction.handleKeyDown()
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("C")
            return true
        }
        return false
    }

    private fun KeyEvent.actionChannelUp(): Boolean {
        if (action == ACTION_DOWN) {
            onPressChannelUp()
        }
        return true
    }

    private fun KeyEvent.actionChannelDown(): Boolean {
        if (action == ACTION_DOWN) {
            onPressChannelDown()
        }
        return true
    }

    private fun KeyEvent.actionFastForward(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (action == ACTION_UP) {
                when (playerAction.getSeekingRule()) {
                    SmartSeekingRuleType.LIVE_TIMESHIFT,
                    SmartSeekingRuleType.CATCHUP_TIMESHIFT,
                    SmartSeekingRuleType.TIMESHIFT -> {
                        playerAction.ff()
                        setRemoteControlState(RemoteControlState.PLAYER_SEEKING)
                    }
                    else -> playerAction.seekToLive()
                }
            }
            return true
        } else {
            if (remoteControlState != RemoteControlState.PLAYER_SEEKING && action == ACTION_UP)
                epgActivityAction.scrollEPGDay(EpgFragment.SCROLL_DAY_FUTURE)
        }
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (action == ACTION_UP)
                playerAction.ff()
            return true
        }
        return false
    }

    private fun KeyEvent.actionRewind(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND) {
            if (Flavor().shouldHandleLongPress() && isLongPress) {
                if (playerAction.isUIVisible() && Flavor().shouldHandleOkOnPlayerControls()) {
                    playerAction.handleSecondOK()
                }
            }
            else if (action == ACTION_UP && playerAction.isRewindLongPressHandled().not()) {
                when (playerAction.getSeekingRule()) {
                    SmartSeekingRuleType.CATCHUP_TIMESHIFT,
                    SmartSeekingRuleType.LIVE_TIMESHIFT,
                    SmartSeekingRuleType.TIMESHIFT -> {
                        playerAction.rw()
                        setRemoteControlState(RemoteControlState.PLAYER_SEEKING)
                    }
                    SmartSeekingRuleType.LIVE_ONLY -> playerAction.showUnavailableActionToast()
                    SmartSeekingRuleType.CATCHUP_STARTOVER,
                    SmartSeekingRuleType.LIVE_STARTOVER,
                    SmartSeekingRuleType.STARTOVER -> playerAction.seekToStartOver()
                }
            }
            return true
        } else {
            if (remoteControlState != RemoteControlState.PLAYER_SEEKING && action == ACTION_UP)
                epgActivityAction.scrollEPGDay(EpgFragment.SCROLL_DAY_PAST)
        }
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (action == ACTION_UP)
                playerAction.rw()
            return true
        }
        return false
    }

    private fun KeyEvent.actionPlayPause(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (action == ACTION_UP) {
                playerAction.seekToSelectedPos()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            }
            return true
        }
        if (action == ACTION_UP) {
            if (remoteControlState == RemoteControlState.NORMAL || remoteControlState == RemoteControlState.PLAYER_BACKGROUND) {
                return epgActivityAction.openPlayer()
            } else {
                when (playerAction.getSeekingRule()) {
                    SmartSeekingRuleType.LIVE_ONLY -> playerAction.showUnavailableActionToast()
                    else -> playerAction.togglePauseResume()
                }
            }
        }
        return true
    }

    private fun KeyEvent.actionStop(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (action == ACTION_UP) {
                playerAction.cancelSeeking()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            }
            return true
        }
        if (action == ACTION_UP)
            playerAction.seekToLive()
        return true
    }

    private fun KeyEvent.actionBack(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_NUMBER_ZAPP) {
            if (action == ACTION_UP)
                zappNumberPressed("C")
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_SEEKING) {
            if (action == ACTION_UP) {
                playerAction.cancelSeeking()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION) {
            if (action == ACTION_UP) {
                activityPlayerAction.requestFocusAfterReturningFromSubscriptions()
                playerAction.hideSubscriptions()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
                playerAction.switchToLastChannel()

            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SIDE_MENU) {
            if (action == ACTION_UP) {
                playerAction.hideSideMenu()
                activityPlayerAction.requestLastFocusedView()
                setRemoteControlState(RemoteControlState.PLAYER_FOREGROUND)
            }
            return true
        }
        if (remoteControlState == RemoteControlState.PLAYER_BACKGROUND) {
            return false
        }

        if (remoteControlState == RemoteControlState.OFFLINE_MODE){
            return false
        }

        if (action == ACTION_UP) {
            if (playerAction.softZapping())
                playerAction.cancelSoftZap()
            else if (playerAction.hideControls().not()) { //controls will hide if currently shown and return result
                if(Flavor().shouldHandleBackAsLast()) {
                    actionLast()
                } else {
                    activityPlayerAction.playerStop(true)
                    epgActivityAction.updateFocusedEventInfo()
                }
            }
        }
        return true
    }

    private fun KeyEvent.actionLive(): Boolean {
        if (remoteControlState == RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION) {
            return false
        }
        if (action == ACTION_UP) {
            if (remoteControlState == RemoteControlState.PLAYER_FOREGROUND)
                playerAction.handleLiveButtonPress()
            else
                activityAction.handleLiveButtonPress()
        }
        return true
    }


    private fun getNormalRemoteControlKeys(): RemoteControlKeys {
        //return empty blacklist so all keys are allowed
        return RemoteControlKeys(RemoteKeysFilter.BLACKLIST, arrayListOf())
    }

    private fun getOfflineModeRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(RemoteKeysFilter.BLACKLIST, arrayListOf(
                KEYCODE_MEDIA_PLAY_PAUSE,
                KEYCODE_MEDIA_RECORD,
                KEYCODE_GUIDE,
                KEYCODE_CHANNEL_DOWN,
                KEYCODE_CHANNEL_UP,
                KEYCODE_TV, //Live button
                KEYCODE_F4 //Startover button on Entel RCU
        ))
    }

    private fun getPlayerForegroundRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(
            RemoteKeysFilter.WHITELIST, arrayListOf(
                KEYCODE_MEDIA_REWIND,
                KEYCODE_MEDIA_FAST_FORWARD,
                KEYCODE_MEDIA_PLAY_PAUSE,
                KEYCODE_MEDIA_STOP,
                KEYCODE_MEDIA_RECORD,
                KEYCODE_BACK,
                KEYCODE_CHANNEL_DOWN,
                KEYCODE_CHANNEL_UP,
                KEYCODE_DPAD_LEFT,
                KEYCODE_DPAD_RIGHT,
                KEYCODE_DPAD_CENTER,
                KEYCODE_NUMPAD_0,
                KEYCODE_NUMPAD_1,
                KEYCODE_NUMPAD_2,
                KEYCODE_NUMPAD_3,
                KEYCODE_NUMPAD_4,
                KEYCODE_NUMPAD_5,
                KEYCODE_NUMPAD_6,
                KEYCODE_NUMPAD_7,
                KEYCODE_NUMPAD_8,
                KEYCODE_NUMPAD_9,
                KEYCODE_0,
                KEYCODE_1,
                KEYCODE_2,
                KEYCODE_3,
                KEYCODE_4,
                KEYCODE_5,
                KEYCODE_6,
                KEYCODE_7,
                KEYCODE_8,
                KEYCODE_9,
                KEYCODE_LAST_CHANNEL,
                KEYCODE_PROG_GREEN,
                KEYCODE_PROG_BLUE,
                KEYCODE_PROG_RED,
                KEYCODE_INFO,
                KEYCODE_DPAD_UP,
                KEYCODE_DPAD_DOWN,
                KEYCODE_BUTTON_B,
                KEYCODE_F4,
                KEYCODE_TV,
                KEYCODE_CAPTIONS
            )
        )
    }

    private fun getPlayerNumberZappRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(
            RemoteKeysFilter.WHITELIST, arrayListOf(
                KEYCODE_BACK,
                KEYCODE_DPAD_CENTER,
                KEYCODE_DPAD_UP,
                KEYCODE_DPAD_DOWN,
                KEYCODE_DPAD_LEFT,
                KEYCODE_DPAD_RIGHT,
                KEYCODE_NUMPAD_0,
                KEYCODE_NUMPAD_1,
                KEYCODE_NUMPAD_2,
                KEYCODE_NUMPAD_3,
                KEYCODE_NUMPAD_4,
                KEYCODE_NUMPAD_5,
                KEYCODE_NUMPAD_6,
                KEYCODE_NUMPAD_7,
                KEYCODE_NUMPAD_8,
                KEYCODE_NUMPAD_9,
                KEYCODE_0,
                KEYCODE_1,
                KEYCODE_2,
                KEYCODE_3,
                KEYCODE_4,
                KEYCODE_5,
                KEYCODE_6,
                KEYCODE_7,
                KEYCODE_8,
                KEYCODE_9
            )
        )
    }

    private fun getPlayerSeekingRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(
            RemoteKeysFilter.WHITELIST, arrayListOf(
                KEYCODE_BACK,
                KEYCODE_DPAD_CENTER,
                KEYCODE_MEDIA_STOP,
                KEYCODE_MEDIA_FAST_FORWARD,
                KEYCODE_MEDIA_REWIND,
                KEYCODE_MEDIA_PLAY_PAUSE,
                KEYCODE_CHANNEL_UP,
                KEYCODE_CHANNEL_DOWN,
                KEYCODE_NUMPAD_0,
                KEYCODE_NUMPAD_1,
                KEYCODE_NUMPAD_2,
                KEYCODE_NUMPAD_3,
                KEYCODE_NUMPAD_4,
                KEYCODE_NUMPAD_5,
                KEYCODE_NUMPAD_6,
                KEYCODE_NUMPAD_7,
                KEYCODE_NUMPAD_8,
                KEYCODE_NUMPAD_9,
                KEYCODE_0,
                KEYCODE_1,
                KEYCODE_2,
                KEYCODE_3,
                KEYCODE_4,
                KEYCODE_5,
                KEYCODE_6,
                KEYCODE_7,
                KEYCODE_8,
                KEYCODE_9
            )
        )
    }

    private fun getPlayerBackgroundRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(RemoteKeysFilter.BLACKLIST, arrayListOf())
    }

    private fun getPlayerBehindSideMenuRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(
            RemoteKeysFilter.WHITELIST, arrayListOf(
                KEYCODE_DPAD_UP,
                KEYCODE_DPAD_DOWN,
                KEYCODE_DPAD_CENTER,
                KEYCODE_BACK,
                KEYCODE_CAPTIONS,
            )
        )
    }

    private fun getAppNotAllowedRemoteControlKeys(): RemoteControlKeys {
        return RemoteControlKeys(
            RemoteKeysFilter.WHITELIST, arrayListOf(
                KEYCODE_DPAD_CENTER
            )
        )
    }

    private fun zappNumberPressed(number: String) {
        if (number == "C") { //cancel
            pressedNumbers = ""
            numberZappTimer.cancel()
            activityPlayerAction.showNumberZap("")
            setRemoteControlState(previousState)
            return
        }

        if (number == "E") { //enter
            switchToZapNumberChannel(pressedNumbers)
            return
        }

        pressedNumbers += number
        numberZappTimer.cancel()

        if (pressedNumbers.length >= 4) {
            pressedNumbers = number
        }
        activityPlayerAction.showNumberZap(pressedNumbers)
        numberZappTimer = timer(null, false, 3000L, 3000L) {
            Handler(Looper.getMainLooper()).post {
                switchToZapNumberChannel(pressedNumbers)
            }
        }
    }

    private fun keyCodeToNumberString(keyCode: Int): String {
        return when (keyCode) {
            KEYCODE_0,
            KEYCODE_NUMPAD_0 -> "0"
            KEYCODE_1,
            KEYCODE_NUMPAD_1 -> "1"
            KEYCODE_2,
            KEYCODE_NUMPAD_2 -> "2"
            KEYCODE_3,
            KEYCODE_NUMPAD_3 -> "3"
            KEYCODE_4,
            KEYCODE_NUMPAD_4 -> "4"
            KEYCODE_5,
            KEYCODE_NUMPAD_5 -> "5"
            KEYCODE_6,
            KEYCODE_NUMPAD_6 -> "6"
            KEYCODE_7,
            KEYCODE_NUMPAD_7 -> "7"
            KEYCODE_8,
            KEYCODE_NUMPAD_8 -> "8"
            KEYCODE_9,
            KEYCODE_NUMPAD_9 -> "9"
            else -> ""
        }
    }

    private fun switchToZapNumberChannel(number: String) {
        setRemoteControlState(previousState)
        activityPlayerAction.showNumberZap("")
        if (number.isNotEmpty()) activityPlayerAction.playNumberZapChanel(number, remoteControlState)
        pressedNumbers = ""
        numberZappTimer.cancel()
    }
}