package com.twentyfouri.tvlauncher.utils

import android.content.Context
import android.content.res.Resources
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.MenuType
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R

/**
 * Mapping from backend strings to the app's translations.
 *
 * This object uses static initialization for fetching string resources.
 * It's not ideal, but this object is used in a few places where a context is not
 * available.
 */
object StringsMapper {
    val PLAYER_NO_DATA = "player_no_data"
    val EPG_NO_DATA = "epg_no_data"

    private var resources:Resources? = null

    fun initialize(context:Context) {
        resources = context.applicationContext.resources
    }

    fun translate(id:String):String {
        return translate(id, id)
    }

    fun translate(id:String, fallback:String):String {
        val resourceId = getResourceId(id)
        return when (resourceId) {
            null -> fallback
            else -> resources!!.getString(resourceId)
        }
    }

    private fun getResourceId(id: String):Int? = when(id) {
        PLAYER_NO_DATA -> R.string.player_no_data
        EPG_NO_DATA -> R.string.epg_no_data
        else -> null
    } ?: when (Flavor().getMenuType(id)) {
        MenuType.HOME -> R.string.tab_home
        MenuType.EPG -> R.string.tab_epg
        MenuType.GRID -> R.string.tab_grid
        MenuType.APPS -> R.string.tab_apps
        MenuType.NOTIFICATIONS -> R.string.top_icon_notifications
        MenuType.VOICE -> R.string.top_icon_voice
        MenuType.SEARCH -> R.string.top_icon_search
        MenuType.PERSONAL -> R.string.top_icon_personal
        MenuType.DIAGNOSTICS -> R.string.top_icon_diagnostics
        MenuType.SETTINGS -> R.string.top_icon_settings
        else -> null
    } ?: when (Flavor().getPlaylistTypeForLocalization(id)) {
        PlaylistType.FAVORITE_CHANNELS -> R.string.row_favorite_channels
        PlaylistType.RECOMMENDED_APPS -> R.string.row_recommended_apps
        PlaylistType.FAVORITE_APPS -> R.string.row_all_apps
        PlaylistType.ALL_CHANNELS,
        PlaylistType.ALL_CHANNELS_PAGED -> R.string.row_all_channels
        PlaylistType.ALL_APPS -> R.string.row_all_apps
        PlaylistType.ALL_GAMES -> R.string.row_all_games
        PlaylistType.PLAY_STORE -> R.string.row_play_store
        PlaylistType.ON_NOW -> R.string.row_filtered_channels
        PlaylistType.DO_NOT_MISS_CHANNELS -> R.string.row_do_not_miss_channels
        PlaylistType.LAST_WATCHED -> R.string.row_last_watched
        PlaylistType.RECENT_RECORDINGS -> R.string.personal_settings_recordings
        PlaylistType.CATCHUP_PROGRAMS -> R.string.row_catchup_events
        PlaylistType.MOST_WATCHED -> null //TODO check
        else -> null
    }
}