package com.twentyfouri.tvlauncher.viewmodels

import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistFilter
import com.twentyfouri.smartmodel.model.dashboard.SmartPlaylistFilterReference
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.extensions.isSameDay
import com.twentyfouri.tvlauncher.extensions.isToday
import com.twentyfouri.tvlauncher.widgets.SettingsItemType
import org.joda.time.DateTime
import org.joda.time.format.DateTimeFormat
import java.util.*

class FilterDateViewModel(
    type: Type,
    filters: List<SmartPlaylistFilter>,
    selectedFiltersReferences: Collection<SmartPlaylistFilterReference>,
    resourceRepository: ResourceRepository,
) : FilterViewModel(type, filters, selectedFiltersReferences, resourceRepository) {

    override fun getTitle(): String {
        val selectedFilterReference = selectedFiltersReferences.find { Flavor().getFilterType(it) == type }
        val selectedFilter = filters.find { it.reference == selectedFilterReference }
        val titleFormatted = selectedFilter?.reference?.title?.let { convertDate(it, true) }
        return titleFormatted ?: type.getAllTitle()
    }

    override fun getPickerItem(filter: SmartPlaylistFilter): ListPickerItem {
        return ListPickerItem(
            id = filter.reference.hashCode().toString(),
            label = convertDate(filter.reference.title, false),
            title = "",
            subtitle = "",
            type = SettingsItemType.LABEL
        )
    }

    private fun convertDate(date: String, short: Boolean): String {
        val dateTime = DateTime.parse(date)
        val now = TimeProvider.now()
        return when {
            dateTime.isToday -> resourceRepository.getString(R.string.epg_date_today)
            dateTime.isSameDay(now.minusDays(1)) -> resourceRepository.getString(R.string.epg_date_yesterday)
            dateTime.isSameDay(now.plusDays(1)) -> resourceRepository.getString(R.string.epg_date_tomorrow)
            dateTime.isSameDay(now.minusDays(Flavor().getEpgConfigValues.EPG_DAYS_INTO_PAST + 1)) ->
                resourceRepository.getString(R.string.epg_date_before_epg_window, Flavor().getEpgConfigValues.EPG_DAYS_INTO_PAST)
            else -> {
                val formatter = DateTimeFormat.forPattern(
                    if (short) {
                        "d MMM"
                    } else {
                        "EEEE, d MMMM"
                    }
                ).withLocale(Locale.getDefault())
                formatter.print(dateTime)
            }
        }
    }
}
