package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import androidx.lifecycle.*
import com.twentyfouri.smartmodel.model.user.SmartUserProfile
import com.twentyfouri.smartmodel.model.user.SmartUserProfileChange
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.data.PersonalSettingsRepository
import com.twentyfouri.tvlauncher.data.RecordingsRepository
import com.twentyfouri.tvlauncher.R

class PersonalSettingsViewModel(
    private val personalSettingsRepository: PersonalSettingsRepository,
    recordingsRepository: RecordingsRepository
): ViewModel() {

    val recordingIconId = R.drawable.ic_record_r

    private val currentProfile = MutableLiveData<SmartUserProfile>()
    private val _parentalRules = MutableLiveData<List<String>?>().apply { value = null }

    @Suppress("SpellCheckingInspection")
    val gdprAccepted: LiveData<Boolean>
    val parentalControlsActivated: LiveData<Boolean>
    val recordingsVisibility: LiveData<Int>
    val userName = personalSettingsRepository.getUserNameLD()
    val parentalRules = _parentalRules
    val parentalRulesRefreshVisibility: LiveData<Int>

    init {
        gdprAccepted = Transformations.map(currentProfile) { Flavor().isGdprAcceptedProfile(it.extras) }
        parentalRulesRefreshVisibility = Transformations.map(currentProfile) { if (Flavor().parentalRulesRefreshActivated ) View.VISIBLE else View.GONE }
        parentalControlsActivated = Transformations.map(currentProfile) { Flavor().isParentalControlActivatedProfile(it.extras) }
        recordingsVisibility = Transformations.map(recordingsRepository.getRecordingsEnabledLD()) { if (it) View.VISIBLE else View.GONE }
    }

    fun initCurrentProfileOnLifecycle(lifecycleOwner: LifecycleOwner) {
        personalSettingsRepository.getCurrentProfileLD().observe(lifecycleOwner, Observer { currentProfile.postValue(it) })
    }

    @Suppress("SpellCheckingInspection")
    fun observeGdprAcceptedOnLifecycle(lifecycleOwner: LifecycleOwner, updateFun: (checked: Boolean) -> Unit) {
        gdprAccepted.observe(lifecycleOwner, Observer { updateFun(it) })
    }

    fun observeParentalControlOnLifecycle(lifecycleOwner: LifecycleOwner, updateFun: (checked: Boolean) -> Unit) {
        parentalControlsActivated.observe(lifecycleOwner, Observer{ updateFun (it) })
    }

    @Suppress("SpellCheckingInspection")
    fun toggleGdprAcceptedOnCurrentProfileOnLifecycle(lifecycleOwner: LifecycleOwner, switchIsChecked: Boolean?) {
        checkNotNull(currentProfile.value) { "There is no profile to update" }
        val change = SmartUserProfileChange(currentProfile.value!!.reference).apply { extras = Flavor().gdprProfileChangeObject(switchIsChecked?.not()) }
        personalSettingsRepository.updateCurrentProfileLD(change).observe(lifecycleOwner, Observer { currentProfile.postValue(it) })
    }

    fun toggleParentalControlActivatedOnCurrentProfileOnLifecycle(lifecycleOwner: LifecycleOwner, switchIsChecked: Boolean?) {
        checkNotNull(currentProfile.value) { "There is no profile to update" }
        val change = SmartUserProfileChange(currentProfile.value!!.reference).apply { extras = Flavor().parentalControlProfileChangeObject(switchIsChecked?.not()) }
        personalSettingsRepository.updateCurrentProfileLD(change).observe(lifecycleOwner, Observer { currentProfile.postValue(it) })
    }

    fun updateParentalRuleOnClick(lifecycleOwner: LifecycleOwner) {
        personalSettingsRepository.updateParentalRule().observe(lifecycleOwner, Observer {
            _parentalRules.postValue(it)
        })
    }
}