package com.twentyfouri.tvlauncher.viewmodels

import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.media.*
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import kotlinx.coroutines.CoroutineScope

class PlayerUICatchupViewModel(
        epgRepository: EpgRepository,
        private val dateTimeRepository: DateTimeRepository,
        metadataRepository: MetadataRepository,
        resourceRepository: ResourceRepository,
        youboraAnalytics: YouboraAnalytics

) : PlayerUIViewModel(metadataRepository, dateTimeRepository, resourceRepository, youboraAnalytics) {

    //observed from View
    val barMaxTimeString: LiveData<String>
    val barCurrentTimeString: LiveData<String>
    val catchupTitle: LiveData<String>
    val catchupTime: LiveData<String>

    private var lastPlayedEventReference: SmartMediaReference? = null
    private var currentlyPlayedEventReference: SmartMediaReference? = null

    //other produced LiveData
    val ageRatings: LiveData<MutableList<SmartAgeRating>>
    private val seekingRuleType = MutableLiveData<SmartSeekingRuleType>()
    @Suppress("JoinDeclarationAndAssignment")
    private val barMaxTime: LiveData<Long>
    private val barCurrentTime: LiveData<Long>
    private var internalFastForwardAllowed: Boolean = true

    init {
        channels = epgRepository.getAllChannelsLD()
        barMaxTime = Transformations.map(playerDuration) { if (it>0) it else 0 }
        barCurrentTime = CombinationTransformations.combineNonNullable(playerDuration, playerPosition) { aDuration, aPosition -> if (aDuration > 0) aPosition else 0 }
        barMaxTimeString = Transformations.map(barMaxTime) { dateTimeRepository.formatTime(it) }
        barCurrentTimeString = Transformations.map(barCurrentTime) { dateTimeRepository.formatTime(it) }
        // this approach didn't work because of unknown reasons, the inside of this map is never called:
        //seekingRuleType = Transformations.map(detail) { Flavor().getSeekingTypeOfMedia(it.reference, true) }
        // so I had to fill seekingRuleType inside of different map method:
        catchupTitle = Transformations.map(detail) {
            internalFastForwardAllowed = true
            it.seekingRuleType?.also { srt -> seekingRuleType.value = srt }
            it.title
        }
        catchupTime = Transformations.map(detail) { dateTimeRepository.formatCatchupDate(it.startDate) }
        ageRatings = Transformations.map(detail) { it.ageRatings }
    }

    //region PlayerUIViewModel overrides

    override fun seekTick(streamDuration: Long?, seekIncrement: Int, seekMultiplier: Int): Int {
        val shouldAllowFiveMinuteFastForward = detail.value?.restrictions?.find { it.type == SmartRestrictionType.FAST_FORWARD } != null
        var newSeekMultiplier = seekMultiplier
        barProgress.value?.let { primaryProgress ->
            var currentPos = primaryProgress
            currentPos += seekIncrement
            if (currentPos < 0) {
                currentPos = 0
                if (seekMultiplier < 0) newSeekMultiplier = 0
            }
            streamDuration?.let {
                if (currentPos > it) {
                    currentPos = it.toInt()
                    if (seekMultiplier > 0) newSeekMultiplier = 0
                }
            }
            if (shouldAllowFiveMinuteFastForward && seekIncrement > 0 && currentPos > FF_MINUTES_ALLOWED_FOR_RECORDING) { //5 minutes ff is allowed for recorded items with FAST_FORWARD restriction
                newSeekMultiplier = 0
                internalFastForwardAllowed = false // blocking FF after 5 minutes
            }
            else
                playerPosition.postValue(currentPos.toLong())

            if (currentPos <= FF_MINUTES_ALLOWED_FOR_RECORDING)
                internalFastForwardAllowed = true // allowing FF again for the first 5 minutes
        }
        return newSeekMultiplier
    }

    override fun seekToSelectedPos(player: SmartPlayer?, isInTrick: Boolean) {
        barProgress.value?.also { player?.seekTo(it.toLong()) }
    }

    override fun getSeekingRuleTypeInternal(): SmartSeekingRuleType? {
        return if (detail.value?.restrictions?.find { it.type == SmartRestrictionType.FAST_FORWARD } != null) {
            SmartSeekingRuleType.TIMESHIFT
        } else {
            seekingRuleType.value
        }
    }

    override fun getNextChannelReference(): SmartMediaReference? {
        return channels.value?.getOrNull(getNextChannelIndex())?.reference
    }

    override fun getPreviousChannelReference(): SmartMediaReference? {
        return channels.value?.getOrNull(getPreviousChannelIndex())?.reference
    }

    private fun getChannelIndex(): Int? {
        return channels.value?.indexOf(channels.value?.find { it.reference == detail.value?.channelReference })
    }

    override fun getChannelReferenceByNumber(number: String): SmartMediaReference? {
        return try {
            getChannelByNumber(number.toInt())?.reference
        } catch (e: NumberFormatException) {
            null
        }
    }

    override fun getClosestChannelReferenceByNumber(number: String): SmartMediaReference? {
        return try {
            getClosestChannelByNumber(number.toInt(), channels.value)?.reference
        } catch (e: NumberFormatException) {
            null
        }
    }

    private fun getNextChannelIndex(): Int {
        getChannelIndex()?.let { playingIndex ->
            channels.value?.let {
                return if (playingIndex >= it.size - 1) 0
                else playingIndex + 1
            }
        }
        return 0
    }

    private fun getPreviousChannelIndex(): Int {
        getChannelIndex()?.let { playingIndex ->
            channels.value?.let {
                return if (playingIndex == 0) it.size - 1
                else playingIndex - 1
            }
        }
        return 0
    }

    override fun getLastPlayedEventReference(): SmartMediaReference? {
        return lastPlayedEventReference
    }

    override fun observeMediaStreamLD(
        detail: SmartMediaDetail,
        channel: SmartMediaItem?,
        doBlock: (stream: SmartMediaStream) -> Unit,
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean,
        lifecycleOwner: LifecycleOwner,
        isPlayingCatchup: Boolean
    ) {
        if(currentlyPlayedEventReference != detail.reference){
            lastPlayedEventReference = currentlyPlayedEventReference
            currentlyPlayedEventReference = detail.reference
        }
        channel?.also { setPlayingChannel(it) }
        super.observeMediaStreamLD(
            detail,
            channel,
            doBlock,
            catchBlock,
            lifecycleOwner,
            isPlayingCatchup
        )
    }

    override fun isFastForwardAllowed(): Boolean = internalFastForwardAllowed

    override fun getPlayingEventTitleLD(): LiveData<String> = catchupTitle

    //endregion
}