package com.twentyfouri.tvlauncher.viewmodels

import androidx.lifecycle.*
import androidx.lifecycle.Transformations
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.watchlist.SmartContinueWatchingItem
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.extensions.ifNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations

class PlayerViewModel(
    epgRepository: EpgRepository,
    metadataRepository: MetadataRepository,
    dateTimeRepository: DateTimeRepository
): ViewModel() {
    var ready: Boolean = false

    private val reference = MutableLiveData<SmartMediaReference>()

    val channels = epgRepository.getAllChannelsLD()
    val detail: LiveData<SmartMediaDetail>
//    val ageRatings: LiveData<MutableList<SmartAgeRating>>
    val channel: LiveData<SmartMediaItem?>
    val playAsMediaType: LiveData<SmartMediaType?>

    init {
        detail = Transformations.switchMap(reference) {
            ready = false
            metadataRepository.getMediaDetailLD(it, null, true) //TODO Is the bookmark needed here?
        }

//        ageRatings = Transformations.map(detail) { it.ageRatings }

        channel = Transformations.switchMap(detail) {
            CombinationTransformations.map (
                {
                    detail.value?.findMyChannel(channels.value)
                },
                listOf(detail, channels)
            )
        }

        playAsMediaType = Transformations.map(channel) {
            ready = true
            detail.value.ifNull { throw IllegalStateException("The detail should have been loaded way before now") }
            (detail.value!!.isOnAirNow(dateTimeRepository)).ifTrue { return@map SmartMediaType.LIVE_CHANNEL }
            detail.value!!.type //note that this returns LIVE_CHANNEL when the "detail" is a channel
        }
    }

    private fun SmartMediaDetail.isOnAirNow(dateTimeRepository: DateTimeRepository): Boolean {
        return type == SmartMediaType.LIVE_EVENT && dateTimeRepository.isLive(startDate, endDate)
    }

    private fun SmartMediaDetail.findMyChannel(aChannels: List<SmartMediaItem>?): SmartMediaItem? {
        val aChannelReference = if (type == SmartMediaType.LIVE_CHANNEL) reference else channelReference
        return aChannels?.find { it.reference == aChannelReference }
    }

    fun getChannelByReference(channelReference: SmartMediaReference) = channels.value?.find { it.reference == channelReference }

    fun setReference(mediaReference: SmartMediaReference) { reference.value = mediaReference}

    fun buildBookmark(position: Int, duration: Int): SmartContinueWatchingItem? {
        val bookmarkReference = Flavor().getBookmarkReference(detail.value ?: return null)
                ?: return null
        val bookmark = SmartContinueWatchingItem(bookmarkReference)
        bookmark.position = position
        bookmark.duration = duration
        return bookmark
    }
}