package com.twentyfouri.tvlauncher.viewmodels

import android.animation.ValueAnimator
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.provider.Settings
import android.view.KeyEvent
import android.view.View
import android.widget.ImageView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.*
import androidx.lifecycle.Transformations.map
import com.twentyfouri.androidcore.utils.ColorImageSpecification
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.smartmodel.serialization.SmartDataPrimitive
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.AppListItem
import com.twentyfouri.tvlauncher.data.AppUsageRepository
import com.twentyfouri.tvlauncher.data.channel.ExternalRowItem
import com.twentyfouri.tvlauncher.extensions.isToday
import com.twentyfouri.tvlauncher.extensions.seekingRuleAllowsStartover
import com.twentyfouri.tvlauncher.ui.EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE
import com.twentyfouri.tvlauncher.ui.PlayerLocation
import com.twentyfouri.tvlauncher.utils.*
import kotlinx.coroutines.flow.*
import org.joda.time.format.DateTimeFormatterBuilder
import timber.log.Timber

class RowItemViewModel(
    private val navigator: Navigator,
    private val resourceRepository: ResourceRepository,
    private val appUsageRepository: AppUsageRepository
) : ViewModel() {

    private val _mediaItem = MutableLiveData<SmartMediaItem>()
    private val _appListItem = MutableLiveData<AppListItem>()
    private val gridItem = MutableLiveData<SmartMediaItem>()
    private val isSelected = MutableLiveData<Boolean>().apply { value = false }
    private val appListItemGooglePlay = MutableLiveData<AppListItem>()
    private val _appChannelProgram = MutableLiveData<ExternalRowItem>()
    private val _isFocused = MutableLiveData<Boolean?>().apply { value = null }
    private val _labelFirstTileShift = MutableLiveData<Float>().apply { value = 0.0f }

    val mediaItem: LiveData<SmartMediaItem>
    val appListItem: LiveData<AppListItem> = _appListItem
    val appChannelProgram: LiveData<ExternalRowItem> = _appChannelProgram
    val isFocused: LiveData<Boolean?> = _isFocused
    val title: LiveData<String?>
    val imageSpecification: LiveData<ImageSpecification>
    val iconSpecification: LiveData<ImageSpecification>
    val onClickListener: LiveData<View.OnClickListener>
    val onLongClickListener: LiveData<View.OnLongClickListener>
    val onKeyEventListener: LiveData<View.OnKeyListener>
    val cardWidth: LiveData<Int>
    val cardHeight: LiveData<Int>
    val iconWidth: LiveData<Int>
    val iconHeight: LiveData<Int>
    val imageHeight: LiveData<Int>
    val cardLayoutPadding: LiveData<Int>
    val imageScaleType: LiveData<ImageView.ScaleType>
    val labelTopMargin = MutableLiveData<Int>().apply { value = 0 }
    val labelTopMarginSmall = resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_top_margin)
    val labelTopMarginMiddle = resourceRepository.getDimensionPixelSize(R.dimen.middle_card_text_top_margin)
    var labelTopMarginValueAnimator: ValueAnimator? = null
    val labelStartMargin: LiveData<Int>
    val labelEndMargin: LiveData<Int>
    val labelFirstTileShift: LiveData<Float> = _labelFirstTileShift
    val cardRadius: LiveData<Int>
    var useSmallScaleAnimation: Boolean = true
    var isFirst: Boolean = false
    var position: Int = -1

    //TODO during some items update "isFirst" property is not updated and title is not properly drawn

    init {
        mediaItem = CombinationTransformations.map(
            _mediaItem,
            {
                it
            },
            gridItem,
            {
                it
            }
        )
        title = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> it.title
                    SmartMediaType.LIVE_EVENT -> it.title
                    else -> null
                }
            },
            gridItem,
            {
                buildString {
                    append(it.title)
                    it.startDate?.let { startDate ->
                        val formatter = DateTimeFormatterBuilder()
                            .apply {
                                startDate.isToday.ifFalse {
                                    appendDayOfMonth(1)
                                    appendLiteral(' ')
                                    appendMonthOfYearShortText()
                                    appendLiteral(' ')
                                }
                            }
                            .appendHourOfDay(1)
                            .appendLiteral(':')
                            .appendMinuteOfHour(2)
                            .toFormatter()
                        append(" - ")
                        append(Flavor().getFormattedDateForGrid(formatter.print(startDate)))
                    }
                }
            },
            _appListItem,
            { it.getLabelAsSafeString() },
            appListItemGooglePlay,
            { it.getLabelAsSafeString() },
            _appChannelProgram,
            { it.title }
        )
        cardWidth = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_width)
                    else -> {
                        useSmallScaleAnimation = false
                        resourceRepository.getDimensionPixelSize(R.dimen.big_card_width)
                    }
                }
            },
            gridItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.middle_card_width) },
            _appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_width) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_width) },
            _appChannelProgram,
            {
                useSmallScaleAnimation = false
                resourceRepository.getDimensionPixelSize(R.dimen.big_card_width)
            }
        )
        cardHeight = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_height)
                    else -> resourceRepository.getDimensionPixelSize(R.dimen.big_card_height)
                }
            },
            gridItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.middle_card_height) },
            _appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_height) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_width) },
            _appChannelProgram,
            { resourceRepository.getDimensionPixelSize(R.dimen.big_card_height) }
        )
        iconWidth = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_width)
                    else -> {
                        useSmallScaleAnimation = false
                        resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_width)
                    }
                }
            },
            gridItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_width) },
            appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_width) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_width) },
            _appChannelProgram,
            {
                useSmallScaleAnimation = false
                resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_width)
            }
        )
        iconHeight = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_height)
                    else -> resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_height)
                }
            },
            gridItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_height) },
            appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_height) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_logo_height) },
            _appChannelProgram,
            { resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_height) }
        )
        cardRadius = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_radius)
                    else -> resourceRepository.getDimensionPixelSize(R.dimen.big_card_radius)
                }
            },
            gridItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_radius) },
            _appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_radius) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_radius) },
            _appChannelProgram,
            { resourceRepository.getDimensionPixelSize(R.dimen.big_card_radius) }
        )
        imageHeight = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_center_logo_height)
                    else -> ConstraintLayout.LayoutParams.MATCH_PARENT
                }
            },
            gridItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> resourceRepository.getDimensionPixelSize(R.dimen.small_card_center_logo_height)
                    else -> ConstraintLayout.LayoutParams.MATCH_PARENT
                }
            },
            _appListItem,
            { ConstraintLayout.LayoutParams.MATCH_PARENT },
            _appChannelProgram,
            { ConstraintLayout.LayoutParams.MATCH_PARENT }
        )

        isFocused.asFlow().distinctUntilChanged().map { focused ->
            if(focused == true) {
                if(gridItem.value != null) labelTopMarginMiddle
                else labelTopMarginSmall
            } else 0
        }.onEach {
            labelTopMarginValueAnimator?.cancel()
            val goal = it
            val start = labelTopMargin.value ?: 0
            if(goal == start) return@onEach
            labelTopMarginValueAnimator = ValueAnimator.ofInt(start, goal).apply {
                duration = 150L
                addUpdateListener { animation -> labelTopMargin.value = animation.animatedValue as Int }
                start()
            }
        }.launchIn(viewModelScope)

        labelEndMargin = CombinationTransformations.map(
            _mediaItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            gridItem,
            {
                if(isFirst) 2 * resourceRepository.getDimensionPixelSize(R.dimen.middle_card_text_end_margin)
                else resourceRepository.getDimensionPixelSize(R.dimen.middle_card_text_end_margin)
            },
            appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            _appChannelProgram,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) }
        )
        labelStartMargin = CombinationTransformations.map(
            _mediaItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            gridItem,
            {
                if(isFirst) 2 * resourceRepository.getDimensionPixelSize(R.dimen.middle_card_text_start_margin)
                else resourceRepository.getDimensionPixelSize(R.dimen.middle_card_text_start_margin)
            },
            appListItem,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            appListItemGooglePlay,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) },
            _appChannelProgram,
            { resourceRepository.getDimensionPixelSize(R.dimen.small_card_text_side_margins) }
        )
        cardLayoutPadding = map(
            isSelected
        ) {
            if (it) {
                resourceRepository.getDimensionPixelSize(R.dimen.card_padding_selected)
            } else {
                resourceRepository.getDimensionPixelSize(R.dimen.card_padding)
            }
        }
        imageSpecification = CombinationTransformations.map(
            _mediaItem,
            { getImageSpecification(it) },
            gridItem,
            { getImageSpecification(it) },
            _appListItem,
            { getImageSpecification(it) },
            appListItemGooglePlay,
            { getImageSpecificationGooglePlay() },
            _appChannelProgram,
            { getImageSpecificationAppChannelProgram(it) },
            listOf(cardWidth, cardHeight, imageHeight)
        )
        iconSpecification = CombinationTransformations.map(
            _mediaItem,
            { getIconSpecification(it) },
            gridItem,
            { getIconSpecification(it) },
            appListItemGooglePlay,
            { getIconSpecificationGooglePlay(it) }
        )
        onClickListener = CombinationTransformations.map(
            _mediaItem,
            { getOnClickListener(it) },
            gridItem,
            { getOnClickListener(it) },
            _appListItem,
            { getOnClickListener(it) },
            appListItemGooglePlay,
            { getOnClickListener(it) },
            _appChannelProgram,
            { getOnClickListenerAppChannelProgram(it) }
        )
        onLongClickListener = map(_appListItem) {
            getOnLongClickListener(it)
        }
        onKeyEventListener = CombinationTransformations.map(
            _mediaItem,
            { getOnKeyEventListener(it) },
            gridItem,
            { getOnKeyEventListener(it) }
        )
        imageScaleType = CombinationTransformations.map(
            _mediaItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> ImageView.ScaleType.FIT_CENTER
                    else -> ImageView.ScaleType.CENTER_CROP
                }
            },
            gridItem,
            {
                when (it.type) {
                    SmartMediaType.LIVE_CHANNEL -> ImageView.ScaleType.FIT_CENTER
                    else -> ImageView.ScaleType.CENTER_CROP
                }
            },
            _appListItem,
            { ImageView.ScaleType.CENTER_CROP },
            _appChannelProgram,
            { ImageView.ScaleType.CENTER_CROP }
        )
    }

    private fun getOnClickListener(item: AppListItem) = View.OnClickListener { view ->
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Row item $position selected: ${title.value}")
        if (view.isSelected) {
            // User confirms new position of app item
            view.setSelectedState(false)
            return@OnClickListener
        }
        try {
            navigator.navigate(item.intent)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        appUsageRepository.saveTimeOfPackage(item.intent.component?.packageName)
    }

    private fun getOnClickListenerAppChannelProgram(item: ExternalRowItem) = View.OnClickListener {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Row item $position selected: ${title.value}")
        try {
            IntentLauncher.buildMediaIntentFromUriByPackage(item.packageName, item.actionUri)?.also { intent -> navigator.navigate(intent) }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun getOnLongClickListener(item: AppListItem) = View.OnLongClickListener {
        try {
            item.isFavouriteItem.ifTrue {
                val target = SmartNavigationTarget(
                    action = SmartNavigationAction.FAVORITE_APPS_SCREEN,
                    extras = SmartDataValue.from(item.intent.component?.packageName
                        ?: item.intent.`package`))
                navigator.navigate(target)
                return@OnLongClickListener true
            }
            item.intent.hasExtra(EXTRA_FAVORITE_APPS_FRAGMENT_NAVIGATION_CODE).ifTrue {
                //do nothing
                return@OnLongClickListener true
            }
            val intent = Intent()
            val uri = Uri.fromParts("package", item.intent.component?.packageName, null)
            intent.action = Settings.ACTION_APPLICATION_DETAILS_SETTINGS
            intent.data = uri
            navigator.navigate(intent)
            true

        } catch (e: Exception) {
            e.printStackTrace()
            false
        }
    }

    private fun getOnKeyEventListener(item: SmartMediaItem) = View.OnKeyListener { v, keyCode, event ->
        when (event.keyCode) {
            KeyEvent.KEYCODE_DPAD_CENTER -> {
                item.isObsoleteAndNotRecording().ifTrue {
                   return@OnKeyListener true
                } //Old item, detail disabled
            }
            KeyEvent.KEYCODE_INFO -> {
                if (item.type != SmartMediaType.LIVE_CHANNEL && event.action == KeyEvent.ACTION_UP) {
                    navigator.navigate(SmartNavigationTarget.toDetailPage(item.reference))
                    true
                } else false
            }
            KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE -> {
                item.isObsoleteAndNotRecording().ifTrue {
                    return@OnKeyListener true
                } //Old item, playback disabled
                if (item.type == SmartMediaType.LIVE_CHANNEL || !item.isInFuture()) {
                    val target = SmartNavigationTarget.toPlayer(item.reference, null)
                    target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                    if (item.positionInSeconds > 0) target.playerPosition = item.positionInSeconds
                    navigator.navigate(target)
                    true
                } else {
                    //do nothing, item is not playable
                    false
                }
            }
            KeyEvent.KEYCODE_F4 -> { //KEYCODE_F4 is current startover keycode on new Entel RCU
                item.isObsoleteAndNotRecording().ifTrue {
                    return@OnKeyListener true
                } //Old item, playback disabled
                if (event.action == KeyEvent.ACTION_UP && item.seekingRuleAllowsStartover(isCatchup = null)
                        && !item.isInFuture() && item.type != SmartMediaType.LIVE_CHANNEL) {
                    showStartOverConfirmationDialog(item, v)
                    true
                } else false
            }
            else -> false
        }
    }

    private fun SmartMediaItem.isInFuture() = startDate?.isAfterNow == true

    private fun SmartMediaItem.isFakeLiveEvent(): Boolean {
        (type == SmartMediaType.LIVE_EVENT).ifFalse { return false }
        (extras["no-data"] as? SmartDataPrimitive)?.getBoolean(false).ifTrue { return true }
        (startDate == null && endDate == null).ifTrue { return true }
        return false
    }

    // This is very old event that has passed away from EPG already
    private fun SmartMediaItem.isObsoleteAndNotRecording(): Boolean {
        (this == gridItem.value).ifTrue { return false }
        endDate?.isBefore(TimeProvider.now().minusDays(Flavor().getEpgConfigValues.EPG_DAYS_INTO_PAST)).ifFalse { return false }
        (type == SmartMediaType.RECORDING).ifTrue { return false }
        (type == SmartMediaType.LIVE_CHANNEL).ifTrue { return false }
        return true
    }

    private fun getOnClickListener(item: SmartMediaItem) = View.OnClickListener {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Row item $position selected: ${title.value} ${Flavor().getLogInfoFromSmartMediaItem(item)}")
        val immediateReferenceToPlay = when {
            item.isObsoleteAndNotRecording() -> null
            item.isInFuture() -> null
            item.type == SmartMediaType.LIVE_EVENT && Flavor().navigationHomeLiveSkipDetail -> item.reference
            item.type == SmartMediaType.LIVE_CHANNEL -> item.reference
            item.isFakeLiveEvent() -> item.channelReference
            else -> null
        }

        if (immediateReferenceToPlay != null) {
                val target = SmartNavigationTarget.toPlayer(immediateReferenceToPlay, null)
                target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                navigator.navigate(target)
            // we show detail also for live events
//        } else if (item.type == SmartMediaType.LIVE_EVENT) {
//            val doOnSuccess: () -> Unit = {
//                ParentalPinChecker.isAdultVerified = false
//                val target = SmartNavigationTarget.toPlayer(item.channelReference!!, null)
//                target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
//                navigator.navigate(target)
//            }
//            if (!ParentalPinChecker.checkIsAdultContentLocked(
//                    context = it.context,
//                    mediaItem = item,
//                    doOnSuccess = doOnSuccess
//                )
//            ) {
//                doOnSuccess.invoke()
//            }
        } else {
            navigator.navigate(SmartNavigationTarget.toDetailPage(item.reference))
        }

    }

    private fun getIconSpecificationGooglePlay(it: AppListItem) = object : ImageSpecification() {
        override fun getImmediateDrawable(context: Context) = it.icon

        //NOT USED right now. If you need to use this function, current implementation works as get(), not copy()
        override fun copy(): ImageSpecification = this
    }

    private fun getIconSpecification(it: SmartMediaItem): ImageSpecification {
        val width = resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_width)
        val height = resourceRepository.getDimensionPixelSize(R.dimen.big_card_logo_height)
        
        val url = Flavor().pickBasedOnFlavor(Flavor().getImageOfType(it, ImageType.OVERLAY), width, height, SmartImages.UNRESTRICTED)

        return if (url.isNullOrEmpty()) EmptyImageSpecification()
        else ExpirableGlideImageSpecification(url)
    }

    private fun getImageSpecificationGooglePlay() = ColorImageSpecification(resourceRepository.getColor(R.color.purple_1))

    private fun getImageSpecification(it: AppListItem) = object : ImageSpecification() {
        override fun getImmediateDrawable(context: Context) = it.banner

        //NOT USED right now. If you need to use this function, current implementation works as get(), not copy()
        override fun copy(): ImageSpecification = this
    }

    private fun getImageSpecificationAppChannelProgram(it: ExternalRowItem): ImageSpecification {
        return when {
            it.logoUri.isNotEmpty() -> { ExpirableGlideImageSpecification(it.logoUri) }
            it.thumbnailUri.isNotEmpty() -> ExpirableGlideImageSpecification(it.thumbnailUri)
            it.previewImageUri.isNotEmpty() -> ExpirableGlideImageSpecification(it.previewImageUri)
            else -> EmptyImageSpecification()
        }
    }

    private fun getImageSpecification(it: SmartMediaItem): ImageSpecification {
        val cardWidth = cardWidth.value ?: 0
        val cardHeight = cardHeight.value ?: 0
        val imageHeight = imageHeight.value ?: 0
        val height = if (imageHeight > 0) imageHeight else cardHeight
        val width = cardWidth - (cardHeight - height)
        val foundUrl = when (_mediaItem.value?.type ?: it.type) { //not sure if the "?: it.type" should be here
            SmartMediaType.LIVE_EVENT -> Flavor().pickBasedOnFlavor(it.images,
                width,
                height,
                SmartImages.UNRESTRICTED
            )
            SmartMediaType.LIVE_CHANNEL -> Flavor().pickBasedOnFlavor(Flavor().getImageOfType(it, ImageType.LIGHT),
                width,
                height,
                SmartImages.UNRESTRICTED
            )
            else -> Flavor().pickBasedOnFlavor(it.images,
                width,
                height,
                SmartImages.UNRESTRICTED
            )
        }
        return ExpirableGlideImageSpecification(foundUrl ?: "", R.drawable.default_image_small).apply {
            //Log.e("GlideImageSpecification", "url = ${toString()}")
        }
    }

    fun setItem(item: SmartMediaItem) { _mediaItem.value = item }

    fun setItem(item: AppListItem) { _appListItem.value = item }

    fun setGridItem(item: SmartMediaItem) { gridItem.value = item }

    fun setItemGooglePlay(item: AppListItem) { appListItemGooglePlay.value = item }

    fun setAppChannelProgram(item: ExternalRowItem) { _appChannelProgram.value = item }

    fun setIsFocused(boolean: Boolean?) {
        if(boolean == true) {
            val additionalInfo = when {
                _mediaItem.value != null -> {
                    Flavor().getLogInfoFromSmartMediaItem(_mediaItem.value)
                }
                _appChannelProgram.value != null -> {
                    if(_appChannelProgram.value?.title.isNullOrEmpty()) _appChannelProgram.value?.packageName
                    else ""
                }
                else -> ""
            }
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Row item $position focused: ${title.value} $additionalInfo")
        }
        _isFocused.value = boolean
    }

    fun setIsSelected(boolean: Boolean?) { isSelected.value = boolean ?: false}

    fun setLabelFirstTileShift(shift: Float) { _labelFirstTileShift.value = shift }

    fun getIsItemObsolete(): Boolean = mediaItem.value != gridItem.value && mediaItem.value?.isObsoleteAndNotRecording() ?: false

    private fun showStartOverConfirmationDialog(item: SmartMediaItem, view: View) {
        val messageDialogModel = MessageDialogModel(
                message = resourceRepository.getString(R.string.player_ok_longpress_action_dialog_event, item.title ?: ""),
                description = null,
                optionsButtonText = arrayOf(resourceRepository.getString(R.string.button_yes)),
                cancelButtonText = resourceRepository.getString(R.string.cancel),
                code = MessageDialogCodes.reproduceFromBeginning
        )
        val confirmMessageDialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        confirmMessageDialogFragment.mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {} //do nothing
        }
        confirmMessageDialogFragment.mListener = object : MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                if (answer !is MessageDialogAction.Result) {
                    return false
                }
                when(answer.type) {
                    OPTION_A -> {
                        val target = SmartNavigationTarget.toPlayer(item.reference, null,0)
                        target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                        navigator.navigate(target)
                    }
                    CANCEL -> {} //cancel, do nothing
                    else -> {}
                }
                return false
            }
        }
        (view.context as? FragmentActivity)?.supportFragmentManager?.also {
            confirmMessageDialogFragment.show(
                it,
                "tag_dialog_startover"
            )
        }
    }
}