package com.twentyfouri.tvlauncher.viewmodels

import android.util.SparseIntArray
import androidx.lifecycle.*
import com.twentyfouri.smartmodel.model.dashboard.SmartPageReference
import com.twentyfouri.smartmodel.model.dashboard.SmartPageSection
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.extensions.ifNotEqualTo
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.RecordingsRepository
import com.twentyfouri.tvlauncher.data.RowPageRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations


class RowPageViewModel(
    private val rowPageRepository: RowPageRepository,
    private val setupDataRepository: SetupDataRepository,
    private val recordingsRepository: RecordingsRepository?,
    private val resourceRepository: ResourceRepository,
    private val epgRepository: EpgRepository
) : ViewModel() {

    private val pageReference = MutableLiveData<SmartPageReference>()
    private val pageSections = MutableLiveData<MutableList<SmartPageSection>>()
    private val asynchronousRowViewModels = MutableLiveData<MutableList<RowViewModel>>()

    private val unfilteredPageSections: LiveData<List<SmartPageSection>>
    private val filteredPageSections: LiveData<List<SmartPageSection>>
    private val favoritesEnabled: LiveData<Boolean?>
    private val recordingsEnabled: LiveData<Boolean?>

    val contentOffset: LiveData<Int>
    val sectionViewModels: LiveData<List<SectionViewModel>>
    val rowViewModels: LiveData<MutableList<RowViewModel>>

    private lateinit var rowDaemon: RowDaemon
    private var pageSectionUpdatesStopped: Boolean = false

    init {
        unfilteredPageSections = Transformations.switchMap(pageReference) { getUnfilteredPageDataSectionsLD(it) }
        favoritesEnabled = CombinationTransformations.switchMap(unfilteredPageSections) { getFavoritesEnabledLD() }
        recordingsEnabled = CombinationTransformations.switchMap(unfilteredPageSections) { getRecordingsEnabledLD() }
        filteredPageSections = CombinationTransformations.combineNullable(favoritesEnabled, recordingsEnabled) { favoritesOn, recordingsOn ->
            getFilteredPageDataSectionsLD(favoritesOn, recordingsOn)
        }
        sectionViewModels = Transformations.map(filteredPageSections) { sections ->
            getSectionViewModels(sections)
        }
        rowViewModels = Transformations.map(asynchronousRowViewModels) { models ->
            rowDaemon.filterRowViewModels(models)
        }

        contentOffset = Transformations.map(filteredPageSections) {
            if(it.getOrNull(0)?.sectionStyle == "GRID") resourceRepository.getDimensionPixelSize(R.dimen.topbar_offset_for_content_grid)
            else resourceRepository.getDimensionPixelSize(R.dimen.topbar_offset_for_content)
        }
    }

    private fun getUnfilteredPageDataSectionsLD(reference: SmartPageReference) = rowPageRepository.getPageSectionsLD(reference)

    private fun getFavoritesEnabledLD(): LiveData<Boolean> = MutableLiveData<Boolean>().also {
        if (Flavor().showFavorites().not()) it.postValue(false)
        else setupDataRepository.loadCredentialsObject { credentials -> it.postValue(credentials?.recommendations ?: false) }
    }

    private fun getRecordingsEnabledLD() = recordingsRepository?.getRecordingsEnabledLD() ?: MutableLiveData<Boolean>().apply { value = false }

    private fun getFilteredPageDataSectionsLD(favoritesOn: Boolean?, recordingsOn: Boolean?): List<SmartPageSection> {
        val unfilteredPageSectionsValue = unfilteredPageSections.value
        return when {
            favoritesOn == null || unfilteredPageSectionsValue == null -> emptyList()
            else -> unfilteredPageSectionsValue.filterNot {
                (favoritesOn.not() && Flavor().isFavoriteRow(it)) || (recordingsOn != true && Flavor().isRecordingRow(it))
            }
        }
    }

    //CombinationTransformations.combineNullable is causing re-assigning when navigated to different activity
    //we need to keep the original sectionViewModels if possible to keep the row updating working
    private fun getSectionViewModels(sections: List<SmartPageSection>): List<SectionViewModel> {
        return if (sectionViewModels.value.isNullOrEmpty())
            sections.mapIndexed { index, section -> SectionViewModel(rowPageRepository, resourceRepository, epgRepository, section, index) }
        else
            sectionViewModels.value!!
    }

    fun updatePageReference(pageReference: SmartPageReference) {
        this.pageReference.value.ifNotEqualTo(pageReference) { this.pageReference.value = pageReference }
    }

    fun updatePageSectionsOnLifecycle(lifecycleOwner: LifecycleOwner) {
        filteredPageSections.observe(lifecycleOwner, Observer { filteredList ->
            if (pageSections.value?.size != filteredList?.size) pageSections.postValue(filteredList?.toMutableList() ?: mutableListOf())
        })
        sectionViewModels.observe(lifecycleOwner, Observer { sectionModels ->
            rowDaemon = RowDaemon(sectionModels.size)
            sectionModels.forEach { sectionModel ->
                if (pageSectionUpdatesStopped && sectionModel.isSectionUpdateActive()){
                    stopPageSectionsUpdates()
                }
                sectionModel.rowViewModels.observe(lifecycleOwner, Observer obs@{ rowModels ->
                    rowDaemon.append(sectionModel.sectionIndex, rowModels.size)
                    val currentRowModels = asynchronousRowViewModels.value ?: mutableListOf()
                    rowModels.ifEmpty {
                        //it is necessary to reapply row models to trigger first row filtering in RowDaemon
                        //when first row is empty, yet its empty collection of rowModels comes as last
                        //issue PRJ1210ENT-2832
                        asynchronousRowViewModels.value = currentRowModels
                        return@obs
                    }
                    rowModels
                        .filterNot { rowModel -> currentRowModels.contains(rowModel) }
                        .also { toBeAddedRowModels ->
                            toBeAddedRowModels.ifEmpty { return@obs }
                            currentRowModels.addAll(toBeAddedRowModels)
                            currentRowModels.sortBy { it.index }
                            asynchronousRowViewModels.value = currentRowModels
                        }
                })
            }
        })
    }

    fun updatePageSectionsOnResume(skipUpdateForRowIndex: Int? = null) {
        sectionViewModels.value?.forEach {
            if (it.sectionIndex != skipUpdateForRowIndex) {
                it.refreshOnResume()
            }
        }
    }

    fun stopPageSectionsUpdates() {
        pageSectionUpdatesStopped = true
        sectionViewModels.value?.forEach { it.stopSectionUpdates() }
    }

    inner class RowDaemon(private val numberOfSections: Int): SparseIntArray() {
        private var firstVisibleRowWasAlreadyProcessed = false

        fun filterRowViewModels(rowViewModels: MutableList<RowViewModel>): MutableList<RowViewModel> {
            firstVisibleRowWasAlreadyProcessed.ifTrue { return rowViewModels }
            (0 until numberOfSections).forEach { indexOfSection ->
                when (get(indexOfSection, NO_INDEX)) {
                    NO_INDEX -> return mutableListOf() //loop break
                    0 -> { } //loop continue
                    else -> {
                        firstVisibleRowWasAlreadyProcessed = true
                        return rowViewModels
                    }
                }
            }
            return mutableListOf()
        }
    }

    companion object {
        private const val NO_INDEX = -1
    }
}