package com.twentyfouri.tvlauncher.viewmodels

import android.util.Log
import android.view.View
import androidx.lifecycle.*
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.ListPickerItem
import com.twentyfouri.tvlauncher.data.RowPageRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import com.twentyfouri.tvlauncher.utils.ViewHolder
import com.twentyfouri.tvlauncher.widgets.leanback.OnChildViewHolderSelectedListener
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.first
import timber.log.Timber

class SectionViewModel(
    private val repository: RowPageRepository,
    private val resourceRepository: ResourceRepository,
    private val epgRepository: EpgRepository?,
    val pageSection: SmartPageSection,
    val sectionIndex: Int
): ViewModel() {

    val isPaged = when (Flavor().getPlaylistType(pageSection)) {
        PlaylistType.PLAY_STORE,
        PlaylistType.RECOMMENDED_APPS,
        PlaylistType.ALL_APPS,
        PlaylistType.ALL_GAMES,
        PlaylistType.MOVIES,
        PlaylistType.LAST_WATCHED,
        PlaylistType.EXTERNAL_CHANNEL,
        PlaylistType.FAVORITE_APPS,
        PlaylistType.ALL_CHANNELS,
        PlaylistType.GRID -> false
        PlaylistType.CATCHUP_PROGRAMS,
        PlaylistType.LAST_WATCHED_CHANNELS,
        PlaylistType.ON_NOW,
        PlaylistType.ALL_CHANNELS_PAGED,
        PlaylistType.DO_NOT_MISS_CHANNELS,
        PlaylistType.FAVORITE_CHANNELS,
        PlaylistType.OFFLINE,
        PlaylistType.RECENT_RECORDINGS,
        PlaylistType.UPCOMING_PROGRAMS,
        PlaylistType.UNKNOWN,
        PlaylistType.MOST_WATCHED,
        PlaylistType.RECORDINGS -> true
    }

    val isGrid: Boolean = when (Flavor().getPlaylistType(pageSection)) {
        PlaylistType.ALL_APPS,
        PlaylistType.ALL_GAMES,
        PlaylistType.GRID -> true
        PlaylistType.PLAY_STORE,
        PlaylistType.RECOMMENDED_APPS,
        PlaylistType.FAVORITE_APPS,
        PlaylistType.MOVIES,
        PlaylistType.ALL_CHANNELS,
        PlaylistType.CATCHUP_PROGRAMS,
        PlaylistType.LAST_WATCHED_CHANNELS,
        PlaylistType.ON_NOW,
        PlaylistType.ALL_CHANNELS_PAGED,
        PlaylistType.DO_NOT_MISS_CHANNELS,
        PlaylistType.EXTERNAL_CHANNEL,
        PlaylistType.FAVORITE_CHANNELS,
        PlaylistType.OFFLINE,
        PlaylistType.RECENT_RECORDINGS,
        PlaylistType.UPCOMING_PROGRAMS,
        PlaylistType.UNKNOWN,
        PlaylistType.LAST_WATCHED,
        PlaylistType.MOST_WATCHED,
        PlaylistType.RECORDINGS -> false
    }

    private val sectionData = MutableLiveData<List<Any>>()
    private val  _sectionDataNextPageReceived = MutableLiveData(false)
    private val sectionDataUpdater = Observer<List<Any>> {
        Timber.tag("Flow").d("sectionDataUpdater size: ${it.size}")
        sectionData.postValue(it)
    }

    val sectionDataNextPageReceived: LiveData<Boolean> = _sectionDataNextPageReceived
    val rowViewModels: LiveData<List<RowViewModel>>
    val onNextPageRequestListener: LiveData<OnChildViewHolderSelectedListener>

    private var currentSelectedPosition = RecyclerView.NO_POSITION
    private var isRowLoadingNext = false
    private var oldRowDataTotalCount: Int = 0

    var horizontalFocusIndex = 0
    var verticalFocusIndex = 0

    var resetRow: Boolean = false
    var sectionLoaded = false
    private val updateBlock = {
        sectionLoaded = false
        when (PlaylistType.GRID) {
            Flavor().getPlaylistType(pageSection) -> repository.updateSectionContentLDWithLoading(pageSection, sectionDataUpdater)
            else -> repository.updateSectionContentLD(pageSection, sectionDataUpdater)
        }
    }
    private var updateJob: Job? = null

    val scrollType: ScrollType

    private val playlistOptionsLD: LiveData<SmartPlaylistOptions>
    private val filtersLD: LiveData<List<SmartPlaylistFilter>>
    private val defaultFiltersLD: LiveData<Set<SmartPlaylistFilterReference>>
    val filterViewHolders: LiveData<List<ViewHolder<FilterViewModel>>>
    private val selectedFiltersReferencesLD: MutableLiveData<Set<SmartPlaylistFilterReference>> = MutableLiveData(emptySet())

    init {
        onNextPageRequestListener = MutableLiveData<OnChildViewHolderSelectedListener>().apply {
            value =
                if (isPaged) {
                    object : OnChildViewHolderSelectedListener() {
                        override fun onChildViewHolderSelected(parent: RecyclerView, child: RecyclerView.ViewHolder?, position: Int, subposition: Int) {
                            currentSelectedPosition = position
                            maybeLoadNextPage()
                        }
                    }
                } else null
        }
        _sectionDataNextPageReceived.observeForever {
            if (it) {
                isRowLoadingNext = false
            }
        }
        scrollType = when (Flavor().getPlaylistType(pageSection)) {
            PlaylistType.PLAY_STORE,
            PlaylistType.ALL_APPS,
            PlaylistType.ALL_GAMES,
            PlaylistType.GRID -> ScrollType.NONE
            PlaylistType.RECOMMENDED_APPS,
            PlaylistType.FAVORITE_APPS,
            PlaylistType.ALL_CHANNELS,
            PlaylistType.ALL_CHANNELS_PAGED -> ScrollType.DEFAULT
            else -> ScrollType.FIXED_LEFT_EDGE
        }

        updateJob = viewModelScope.launchPeriodicJob(
            refreshTime = pageSection.refreshTimeMs,
            block = updateBlock
        )

        rowViewModels = Transformations.map(sectionData) { data ->
            val rowModels = when {
                data.isEmpty() && pageSection.sectionStyle != PlaylistType.OFFLINE.name -> emptyList()
                isGrid -> {
                    data
                        .run {
                            // Sort Live TV by channel numbers
                            if (selectedFiltersReferencesLD.value?.any { Flavor().getFilterContent(it) == SmartMediaType.LIVE_VIDEO } == true) {
                                filterIsInstance<SmartMediaItem>().sortedBy { it.channelNumber }
                            } else {
                                this
                            }
                        }
                        .chunked(
                            if (Flavor().getPlaylistType(pageSection) in arrayOf(PlaylistType.ALL_APPS, PlaylistType.ALL_GAMES)) {
                                NUMBER_OF_COLUMNS_IN_GRID_APP
                            } else {
                                NUMBER_OF_COLUMNS_IN_GRID_CHANNELS
                            }
                        )
                        .mapIndexed { index, subData ->
                            RowViewModel(repository, resourceRepository, this, index).apply {
                                rowData.value = subData.toMutableList()
                            }
                        }
                }
                else -> listOf(RowViewModel(repository, resourceRepository, this, 0).apply {
                    rowData.value = data
                })
            }
            Timber.tag("Rows2").d("SectionViewModel sectionData loaded ... index:$sectionIndex ... sectionDataSize:${data?.size} ... rowModelsSize:${rowModels.size}")
            rowModels
        }

        playlistOptionsLD = Transformations.switchMap(sectionData) {
            repository.getPlaylistOptionsLD(pageSection)
        }

        filtersLD = Transformations.map(playlistOptionsLD) {
            it.filters
        }

        defaultFiltersLD = Transformations.map(filtersLD) {
            it.mapNotNull(blockFilterToDefaultReferenceOrNull).toSet()
        }

        filterViewHolders = CombinationTransformations.combineNonNullable(selectedFiltersReferencesLD, filtersLD) { selectedFiltersReferences, filters ->
            if (filters.isEmpty()) {
                emptyList()
            } else {
                filters
                    .filter { it.isAvailable(selectedFiltersReferences) }
                    .groupBy { Flavor().getFilterType(it.reference) }
                    .map {
                        ViewHolder(
                            R.layout.filter_item,
                            when (it.key) {
                                FilterViewModel.Type.CHANNEL -> FilterChannelViewModel(it.key, it.value, selectedFiltersReferences, resourceRepository, epgRepository)
                                FilterViewModel.Type.DATE -> FilterDateViewModel(it.key, it.value, selectedFiltersReferences, resourceRepository)
                                FilterViewModel.Type.CONTENT -> FilterContentViewModel(it.key, it.value, selectedFiltersReferences, resourceRepository)
                                else -> FilterViewModel(it.key, it.value, selectedFiltersReferences, resourceRepository)
                            }
                        )
                    }
                    .toMutableList()
                    .apply {
                        if (selectedFiltersReferences != defaultFiltersLD.value) {
                            add(
                                ViewHolder(
                                    R.layout.filter_item_spacer,
                                    FilterViewModel(FilterViewModel.Type.NONE, emptyList(), emptyList(), resourceRepository)
                                )
                            )
                            add(
                                ViewHolder(
                                    R.layout.filter_item_reset,
                                    FilterViewModel(FilterViewModel.Type.RESET, emptyList(), emptyList(), resourceRepository)
                                )
                            )
                        }

                        firstOrNull()?.model?.isFirst = true
                    }
            }
        }

        if (isGrid) {
            viewModelScope.launch {
                // We need to wait for data to load, and set default filters after that
                defaultFiltersLD.asFlow().first { it.isNotEmpty() }.also { applySelectedFilters(it) }
            }
        }
    }

    private val blockFilterToDefaultReferenceOrNull: (SmartPlaylistFilter) -> SmartPlaylistFilterReference? = {
        blockFilterReferenceToDefaultReferenceOrNull(it.reference)
    }

    private val blockFilterReferenceToDefaultReferenceOrNull: (SmartPlaylistFilterReference) -> SmartPlaylistFilterReference? = {
        when {
            Flavor().getFilterDate(it) == 0 -> it
            Flavor().getFilterContent(it) == SmartMediaType.LIVE_EVENT -> it
            else -> null
        }
    }

    private fun SmartPlaylistFilter.isAvailable(selectedFiltersReferences: Collection<SmartPlaylistFilterReference>): Boolean {
        when (Flavor().getFilterType(reference)) {
            FilterViewModel.Type.DATE -> {
                if (selectedFiltersReferences.any {
                        val c = Flavor().getFilterContent(it)
                        c == SmartMediaType.LIVE_VIDEO || c == SmartMediaType.LIVE_CHANNEL
                    }) {
                    if ((Flavor().getFilterDate(reference) ?: 0) != 0) return false
                }
                if (selectedFiltersReferences.any { Flavor().getFilterContent(it) == SmartMediaType.LIVE_EVENT }) {
                    return (Flavor().getFilterDate(reference) ?: 0) in -7..0
                }
            }
            FilterViewModel.Type.CONTENT -> {
                return Flavor().getFilterContent(reference) != SmartMediaType.LIVE_CHANNEL
            }
            else -> {}
        }
        return true
    }

    fun refreshOnResume() {
        if (pageSection.refreshOnResume || sectionLoaded.not() || shouldLaunchPeriodicUpdate()) {
            Timber.tag("Flow").d( "refreshOnResume ... index: $sectionIndex")
            if (pageSection.resetOnResume) resetRow = true
            loadRowData()
        }
    }

    private fun loadRowData() {
        updateJob?.cancel()
        updateJob = viewModelScope.launchPeriodicJob(
            refreshTime = pageSection.refreshTimeMs,
            block = updateBlock
        )
    }

    private fun maybeLoadNextPage() {
        if (isRowLoadingNext
            || currentSelectedPosition == RecyclerView.NO_POSITION) {
            return
        }

        if (shouldLoadNextPage()) {
            loadNextPage()
        }
    }

    private fun loadNextPage(){
        oldRowDataTotalCount = sectionData.value?.size ?: 0
        isRowLoadingNext = true
        repository.updateSectionContentNextLD(pageSection, sectionData, _sectionDataNextPageReceived)
        sectionLoaded = false
    }

    private fun shouldLoadNextPage(): Boolean {
        val totalItems = sectionData.value?.size ?: 0
        if (currentSelectedPosition < (totalItems / 2)) {
            Timber.tag("Flow").d( "shouldLoadNextPage = false ... not yet in half")
            return false
        }
        if (isPaged.not()) {
            Timber.tag("Flow").d("shouldLoadNextPage = false ... section is not pageable")
            return false
        }
        //if you want to e.g. artificially limit total amount of items or pages, here is the correct method to override
        if (Flavor().hasAnotherPage(pageSection).not()) {
            Timber.tag("Flow").d("shouldLoadNextPage = false ... does not have another page")
            return false
        }

        Timber.tag("Flow").d("shouldLoadNextPage = true")
        return true
    }

    fun stopSectionUpdates(){
        updateJob?.cancel()
        updateJob = null
    }

    override fun onCleared() {
        repository.clearSection(pageSection, sectionDataUpdater)
        updateJob?.cancel()
        updateJob = null
        super.onCleared()
    }

    fun getOldDataTotalCount(): Int = oldRowDataTotalCount

    private fun CoroutineScope.launchPeriodicJob(
        refreshTime: Int,
        block: () -> Unit
    ) = this.launch (Dispatchers.IO) {
        if (refreshTime > 0) {
            while (isActive) {
                block()
                delay(refreshTime.toLong())
            }
        } else {
            block()
        }
    }

    private fun shouldLaunchPeriodicUpdate() =
        updateJob == null && pageSection.refreshTimeMs > 0

    fun isSectionUpdateActive() = updateJob != null

    fun updateFavoriteAppsPriority(currentList: List<Any>) {
        repository.updateFavoriteAppsPriority(currentList)
    }

    fun selectFilter(listPickerItem: ListPickerItem) {
        val selected = selectedFiltersReferencesLD.value?.toMutableSet() ?: return
        if (listPickerItem.isUnselectAction) {
            selected.unselectFilter(listPickerItem)
            return
        }
        val aFilterToSelect = filtersLD.value?.find { it.reference.hashCode().toString() == listPickerItem.id }
        //don't select anything if the selected filter does not belong to this section
        aFilterToSelect ?: return
        //don't select what is already selected
        selected.any { it == aFilterToSelect.reference }.ifTrue { return }
        //remove selected filters of the same kind
        val selectedFilterType = Flavor().getFilterType(aFilterToSelect.reference)
        selected.removeIf { Flavor().getFilterType(it) == selectedFilterType }
        //remove incompatible Date filters
        selected.removeIf {
            val d = Flavor().getFilterDate(it)
            if (d == null) {
                false
            } else {
                when (Flavor().getFilterContent(aFilterToSelect.reference)) {
                    SmartMediaType.LIVE_EVENT -> d > 0
                    SmartMediaType.LIVE_VIDEO,
                    SmartMediaType.LIVE_CHANNEL -> true
                    else -> false
                }
            }
        }
        selected.add(aFilterToSelect.reference)
        applySelectedFilters(selected)
    }

    fun onLoadMore(v: View) {
        loadNextPage()
    }

    private fun MutableSet<SmartPlaylistFilterReference>.unselectFilter(listPickerItem: ListPickerItem) {
        when (listPickerItem.id) {
            FilterViewModel.Type.RESET.name -> {
                clear()
                defaultFiltersLD.value?.also { addAll(it) }
            }
            else -> removeIf { Flavor().getFilterType(it).name == listPickerItem.id } // one filter should be unselected
        }
        applySelectedFilters(this)
    }

    private fun applySelectedFilters(selected: Set<SmartPlaylistFilterReference>) {
        selectedFiltersReferencesLD.value = selected
        val updatedSelectedFilters = Flavor().applyAdditionalFilters(selected.toList())
        pageSection.selectedOptions = playlistOptionsLD.value?.createSelection()?.also {
            it.setFilters(updatedSelectedFilters)
            it.setPaging(
                offset = 0,
                count = Flavor().getPageSize(pageSection),
                totalCount = 0
            )
        }
        oldRowDataTotalCount = 0
        loadRowData()
    }

    private val ListPickerItem.isUnselectAction: Boolean
        get() = when (id) {
            FilterViewModel.Type.CHANNEL.name,
            FilterViewModel.Type.CONTENT.name,
            FilterViewModel.Type.INITIAL.name,
            FilterViewModel.Type.DATE.name,
            FilterViewModel.Type.RESET.name -> true
            else -> false
        }

    enum class ScrollType {
        DEFAULT,
        NONE,
        FIXED_LEFT_EDGE
    }

    companion object {
        private const val NUMBER_OF_COLUMNS_IN_GRID_APP = 7
        private const val NUMBER_OF_COLUMNS_IN_GRID_CHANNELS = 5
    }
}