package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import androidx.lifecycle.*
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.enlarge
import com.twentyfouri.tvlauncher.common.utils.PurchaseChecker
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.utils.ExpirableGlideImageSpecification

class SubscriptionsViewModel(
    val metadataRepository: MetadataRepository,
    val resourceRepository: ResourceRepository
) : ViewModel() {

    private val subscriptionChannel = MutableLiveData<SmartMediaItem>()
    private val _offerTwoVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val _offerThreeVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val _subscriptionBought = MediatorLiveData<SmartPurchase?>(). apply { value = null }

    val channelIcon: LiveData<ImageSpecification>
    val channelText: LiveData<String>
    private val subscriptionPackages: LiveData<List<SmartSubscription>>
    val offerOneImage: LiveData<ImageSpecification>
    val offerTwoImage: LiveData<ImageSpecification>
    val offerThreeImage: LiveData<ImageSpecification>
    val offerTwoVisibility: LiveData<Int> = _offerTwoVisibility
    val offerThreeVisibility: LiveData<Int> = _offerThreeVisibility
    val subscriptionBought: LiveData<SmartPurchase?> = _subscriptionBought

    init {
        subscriptionPackages = Transformations.switchMap(subscriptionChannel) { metadataRepository.getSubscriptionsListLD(it?.reference) }
        channelIcon = Transformations.map(subscriptionChannel) { mediaItem ->
            Flavor().pickBasedOnFlavor(Flavor().getImageOfType(mediaItem, ImageType.DARK),
                    resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_channel_icon_width),
                    resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_channel_icon_height),
                    SmartImages.UNRESTRICTED)?.let {
                ExpirableGlideImageSpecification(it)
                }
        }
        channelText = Transformations.map(subscriptionChannel) { it.title }
        offerOneImage = Transformations.map(subscriptionPackages) { subsList ->
            when {
                subsList.isNotEmpty() -> Flavor().pickBasedOnFlavor(subsList[0].images,
                    resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_width),
                    resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_height),
                    SmartImages.UNRESTRICTED
                )?.let { ExpirableGlideImageSpecification(it)}
                else -> EmptyImageSpecification()
            }
        }
        offerTwoImage = Transformations.map(subscriptionPackages) { subsList ->
            when {
                subsList.size > 1 -> {
                    _offerTwoVisibility.value = View.VISIBLE
                    Flavor().pickBasedOnFlavor(subsList[1].images,
                        resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_width),
                        resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_height),
                        SmartImages.UNRESTRICTED
                    )?.let { ExpirableGlideImageSpecification(it)}
                }
                else -> EmptyImageSpecification()
            }
        }
        offerThreeImage = Transformations.map(subscriptionPackages) { subsList ->
            when {
                subsList.size > 2 -> {
                    _offerThreeVisibility.value = View.VISIBLE
                    Flavor().pickBasedOnFlavor(subsList[2].images,
                        resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_width),
                        resourceRepository.getDimensionPixelSize(R.dimen.not_subscribed_offer_height),
                        SmartImages.UNRESTRICTED
                    )?.let { ExpirableGlideImageSpecification(it)}
                }
                else -> EmptyImageSpecification()
            }
        }
    }

    fun onFocusChange(v: View, hasFocus: Boolean) {
        if (hasFocus) v.foreground = resourceRepository.getDrawable(com.twentyfouri.tvlauncher.R.color.transparent)
        else v.foreground = resourceRepository.getDrawable(com.twentyfouri.tvlauncher.R.color.transparent_10)
        v.enlarge(hasFocus)
    }

    fun onOfferOneClicked(v: View) { onOfferClicked(v, 0) }

    fun onOfferTwoClicked(v: View) { onOfferClicked(v, 1) }

    fun onOfferThreeClicked(v: View) { onOfferClicked(v, 2) }

    private fun onOfferClicked(v: View, index: Int) {
        val doAfterPinVerified: () -> Unit = {
            _subscriptionBought.addSource(metadataRepository.subscribeProduct(
                subscriptionPackages.value?.get(index)?.reference,
                subscriptionPackages.value?.get(index)?.isAutoRenew,
                null
            )) {
                _subscriptionBought.value = it
            }
        }
        PurchaseChecker.checkPurchaseWithPin(v.context, doAfterPinVerified)
    }

    fun hideOffers() {
        _offerTwoVisibility.value = View.GONE
        _offerThreeVisibility.value = View.GONE
    }
    fun setSubscriptionChannel(channel: SmartMediaItem?) { if (channel != null) subscriptionChannel.value = channel!! }

    fun setSubscriptionBought(purchaseResult: SmartPurchase?) { _subscriptionBought.value = purchaseResult }
}