package com.twentyfouri.tvlauncher.widgets

import android.animation.Animator
import android.animation.AnimatorInflater
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.drawable.BitmapDrawable
import android.os.Handler
import android.util.AttributeSet
import android.util.Log
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.view.animation.AnimationUtils
import android.widget.TextView
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import com.bumptech.glide.Glide
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.databinding.DetailViewBinding
import com.twentyfouri.tvlauncher.viewmodels.MetadataViewModel
import timber.log.Timber

class DetailView @JvmOverloads constructor(
    context: Context, val attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FocusHandleFrameLayout(context, attrs, defStyleAttr) {

    private var nullableBinding: DetailViewBinding? = DataBindingUtil.inflate(
            LayoutInflater.from(context),
            R.layout.detail_view,
            this,
            true
    )
    val binding: DetailViewBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")
    private var doAnimations = false
    private var focusAssigned = false
    private var descriptionOnPreDrawListener: ViewTreeObserver.OnPreDrawListener? = null

    init {
        Timber.tag("leaks").d("DetailView init")
    }

    private fun View.postRequestFocus(delay: Long, buttonSectionVisibility: Boolean = true) {
        focusAssigned = false
        if (buttonSectionVisibility) { this.pivotX = 0f }
        postDelayed(
            {
                if (!focusAssigned) { requestFocus() }
                focusAssigned = true
            },
            delay
        )
    }

    fun setObservers(lifecycleOwner: LifecycleOwner, context: Context, activity: FragmentActivity?) {
        binding.detailDescription.binding.viewModel?.apply {
            recordingState.observe(lifecycleOwner) { binding.viewModel?.setRecordingState(it) }
            imageSpecification.observe(lifecycleOwner) { binding.viewModel?.setDetailImageSpecification(it) }
            userTriggeredRecordingState.observe(binding.lifecycleOwner!!) {
                when (it) {
                    MetadataViewModel.RecordingState.NOTHING -> {
                        activity?.onBackPressed()
                    }
                    else -> binding.viewModel?.setRecordingState(it)
                }
            }
            mediaDetail.observe(lifecycleOwner) {detail -> detail?.also { binding.viewModel?.setMediaDetail(it) } }
            channelReference.observe(lifecycleOwner) { binding.viewModel?.setChannelReference(it) }
            playability.observe(lifecycleOwner) { binding.viewModel?.setPlayability(it) }
            descriptionMaxLines.observe(lifecycleOwner) { maxLines ->
                val descriptionTextView = binding.detailDescription.binding.metadataText
                if (descriptionOnPreDrawListener == null && maxLines != INVALID_VALUE) {
                    descriptionTextView.post {
                        descriptionTextView.viewTreeObserver
                            .addOnPreDrawListener(
                                getDescriptionOnPreDrawListener(
                                    descriptionTextView,
                                    maxLines,
                                    description
                                )
                            )
                    }
                }
            }
        }
        binding.viewModel?.apply {
            playButtonVisibility.observe(lifecycleOwner) {
                if (it == View.VISIBLE && !focusAssigned) binding.detailPlayButton.postRequestFocus(100)
            }
            planRecordingButtonVisibility.observe(lifecycleOwner) {
                if (it == View.VISIBLE && !focusAssigned) binding.startRecordButton.postRequestFocus(125)
            }
            stopRecordingButtonVisibility.observe(lifecycleOwner) {
                if (it == View.VISIBLE && !focusAssigned) binding.stopRecordButton.postRequestFocus(125)
            }
            // default focus to show more button
            anyButtonVisibility.observe(lifecycleOwner) {
                if (it == View.GONE) binding.showMoreButton.postRequestFocus(400, false)
            }
            startStopRecordingClicked.observe(lifecycleOwner) {
                binding.detailDescription.startStopRecording()
            }
            detailProgress.observe(lifecycleOwner) {
                Timber.tag("Bookmarks").d("Detail view progress: $it")
                binding.detailPlayButton.setProgress(it)
            }
            //should not be needed the button observes this directly
//            playButtonProgressBarVisibility.observe(
//                lifecycleOwner,
//                Observer { binding.detailPlayButton.setProgressBarVisibility(it) }
//            )
            val metaDataExpanded: () -> Unit = { binding.viewModel?.setIsExpanded(true) }
            val metaDataCollapsed: () -> Unit = {
                binding.viewModel?.setIsExpanded(false)
                binding.viewModel?.setDetailButtonsVisibility(View.VISIBLE)
            }
            val updateTopMargin: (ValueAnimator, ViewGroup.LayoutParams) -> Unit = { anim, params ->
                (params as MarginLayoutParams).topMargin = anim.animatedValue as Int
            }
            shownMore.observe(lifecycleOwner) {
                if (it) binding.viewModel?.setDetailButtonsVisibility(View.GONE)
                if (doAnimations) {
                    if (it) {
                        Handler().post {
                            binding.overlay.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_up_detail_image
                                )
                            )
                            binding.detailImage.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_up_detail_image
                                )
                            )
                            binding.detailChannelLogo.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_up_detail_channel_logo
                                )
                            )
                            binding.descriptionScrollView.animateLayoutParams(
                                R.animator.detail_fragment_metadata_expand,
                                updateTopMargin,
                                metaDataExpanded)
                        }
                    } else {
                        Handler().post {
                            binding.overlay.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_down_detail_image
                                )
                            )
                            binding.detailImage.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_down_detail_image
                                )
                            )
                            binding.detailChannelLogo.startAnimation(
                                AnimationUtils.loadAnimation(
                                    context,
                                    R.anim.move_down_detail_channel_logo
                                )
                            )
                            binding.descriptionScrollView.animateLayoutParams(
                                R.animator.detail_fragment_metadata_collapse,
                                updateTopMargin,
                                metaDataCollapsed)
                        }
                    }
                } else
                    doAnimations = true
            }
        }
    }

    private fun getDescriptionOnPreDrawListener(
        descriptionTextView: TextView,
        maxLines: Int?,
        description: LiveData<String>
    ): ViewTreeObserver.OnPreDrawListener {
        if (descriptionOnPreDrawListener == null) {
            descriptionOnPreDrawListener = ViewTreeObserver.OnPreDrawListener {
                onDescriptionPreDraw(descriptionTextView, maxLines, description)
            }
        } else {
            //There is already one OnPreDrawListener added.
            descriptionTextView.viewTreeObserver
                .removeOnPreDrawListener(descriptionOnPreDrawListener)
        }
        return descriptionOnPreDrawListener!!
    }

    private fun onDescriptionPreDraw(
        descriptionTextView: TextView,
        maxLines: Int?,
        description: LiveData<String>
    ): Boolean {
        nullableBinding ?: return true
        // fix IllegalStateException: descriptionTextView.layout must not be null
        if (descriptionTextView.layout != null) {
            val lineCount = descriptionTextView.layout.lineCount
            if (maxLines != null && lineCount <= maxLines) {
                if (descriptionTextView.layout.text.toString() == description.value) {
                    // Text is not ellipsized.
                    setOnKeyListenersOfButtons()
                    binding.showMoreButton.visibility = View.GONE
                    if (binding.viewModel?.playButtonVisibility?.value != View.VISIBLE
                        && binding.viewModel?.planRecordingButtonVisibility?.value != View.VISIBLE
                        && binding.viewModel?.stopRecordingButtonVisibility?.value != View.VISIBLE
                    ) focusable = View.FOCUSABLE
                    else focusable = View.NOT_FOCUSABLE
                } else {
                    // Text is ellipsized.
                    binding.showMoreButton.visibility = View.VISIBLE
                    focusable = View.NOT_FOCUSABLE
                }
            } else {
                binding.showMoreButton.visibility = View.VISIBLE
                focusable = View.NOT_FOCUSABLE
            }
            return true
        }
        return false
    }

    private fun View.animateLayoutParams(
        animatorId: Int,
        onUpdate: (ValueAnimator, ViewGroup.LayoutParams) -> Unit,
        onAnimationEnd: () -> Unit
    ) {
        (AnimatorInflater.loadAnimator(context, animatorId) as ValueAnimator).apply {
            addListener(object: Animator.AnimatorListener {
                override fun onAnimationRepeat(animation: Animator) {}
                override fun onAnimationCancel(animation: Animator) {}
                override fun onAnimationStart(animation: Animator) {}

                override fun onAnimationEnd(animation: Animator) = onAnimationEnd.invoke()
            })
            addUpdateListener {
                val params = layoutParams
                onUpdate.invoke(it, params)
                layoutParams = params
            }
            start()
        }
    }

    private fun setOnKeyListenersOfButtons() {
        binding.detailPlayButton.setOnKeyListener { v, keyCode, event ->
            if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
                binding.viewModel?.scrollDown?.call()
                true
            } else {
                false
            }
        }
        binding.deleteRecordButton.setOnKeyListener { v, keyCode, event ->
            if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
                binding.viewModel?.scrollDown?.call()
                true
            } else {
                false
            }
        }
        binding.restartButton.setOnKeyListener { v, keyCode, event ->
            if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
                binding.viewModel?.scrollDown?.call()
                true
            } else {
                false
            }
        }
        binding.startRecordButton.setOnKeyListener { v, keyCode, event ->
            if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
                binding.viewModel?.scrollDown?.call()
                true
            } else {
                false
            }
        }
        binding.stopRecordButton.setOnKeyListener { v, keyCode, event ->
            if (event.action == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
                binding.viewModel?.scrollDown?.call()
                true
            } else {
                false
            }
        }
    }

    fun onPausePrivate(){
        nullableBinding ?: return
        if (descriptionOnPreDrawListener == null) return
        nullableBinding!!.detailDescription.binding.metadataText.viewTreeObserver?.removeOnPreDrawListener(descriptionOnPreDrawListener)
        descriptionOnPreDrawListener = null
        Glide.with(context).clear(binding.detailImage)
    }

    fun onDestroyView() {
        Timber.d("DetailView onDestroyView")
        onPausePrivate()
        binding.detailDescription.onDestroyView()
        binding.detailImage.also {
            it.invalidate()
            (it.getDrawable() as? BitmapDrawable)?.bitmap?.recycle()
        }
        nullableBinding = null
    }

    companion object{
        private const val INVALID_VALUE = -1
    }
}