package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.LayoutInflater
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.databinding.DataBindingUtil
import androidx.lifecycle.LifecycleOwner
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.databinding.MetadataViewBinding
import com.twentyfouri.tvlauncher.viewmodels.MetadataViewModel
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber

class MetadataView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr), KoinComponent {

    private var nullableBinding: MetadataViewBinding? = DataBindingUtil.inflate<MetadataViewBinding>(
        LayoutInflater.from(context),
        R.layout.metadata_view,
        this,
        true
    ).apply {
        lifecycleOwner = context as LifecycleOwner
    }
    val binding: MetadataViewBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")

    init {
        val viewModelFromFactory = get<MetadataViewModel>()
        viewModelFromFactory.setObserversInContext(context)
        binding.viewModel = viewModelFromFactory
        viewModelFromFactory.iconsViewModel = binding.mediaRestrictionIcons.viewModel
    }

    fun bind(
        aMediaReference: SmartMediaReference,
        fallbackMediaReference: SmartMediaReference? = null,
        where: MetadataViewModel.WhereMetadataAreDisplayed = MetadataViewModel.WhereMetadataAreDisplayed.DETAIL
    ) {
        binding.viewModel?.apply {
            // Where must be set first!
            setWhereDisplayed(where)
            setMediaReference(aMediaReference, fallbackMediaReference)
        }
    }

    fun bindOnEpg(aMediaReference: SmartMediaReference, fallbackMediaReference: SmartMediaReference? = null) {
        bind(aMediaReference, fallbackMediaReference, MetadataViewModel.WhereMetadataAreDisplayed.EPG)
    }

    fun bindOnRow(aMediaReference: SmartMediaReference, fallbackMediaReference: SmartMediaReference?) {
        bind(aMediaReference, fallbackMediaReference, MetadataViewModel.WhereMetadataAreDisplayed.ROW)
    }

    fun bindOnChannelRow(aMediaReference: SmartMediaReference, fallbackMediaReference: SmartMediaReference?) {
        bind(aMediaReference, fallbackMediaReference, MetadataViewModel.WhereMetadataAreDisplayed.CHANNEL_ROW)
    }

    fun setExpanded(expanded: Boolean) { binding.viewModel?.setExpanded(expanded) }

    fun startStopRecording() { binding.viewModel?.startStopRecording(context as LifecycleOwner) }

    fun onDestroyView() {
        Timber.tag("leaks").d("MetadataView onDestroyView")
        binding.viewModel?.onDestroyView()
        nullableBinding = null
    }
}