package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.graphics.drawable.Drawable
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.Observer
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.bumptech.glide.signature.ObjectKey
import com.twentyfouri.androidcore.epg.EpgView
import com.twentyfouri.smartexoplayer.ui.StandardControlsHelper
import com.twentyfouri.smartexoplayer.ui.StandardControlsListener
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.databinding.PlayerControlsLiveBinding
import com.twentyfouri.tvlauncher.di.viewModel
import com.twentyfouri.tvlauncher.viewmodels.PlayerUILiveViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.joda.time.DateTime
import timber.log.Timber
import java.lang.IllegalArgumentException

class PlayerUILive constructor(context: Context) : PlayerUIBase(context), StandardControlsListener {

    override val viewModel: PlayerUILiveViewModel by viewModel()

    override val binding = DataBindingUtil.inflate<PlayerControlsLiveBinding>(
            LayoutInflater.from(context),
            R.layout.player_controls_live,
            this,
            true
    )

    //Preload channel icons
    //Unfortunately glide method "preload" do not work as expected. There is still big delay when glide....preload() is used.
    //Seems that only way to force image into memory is really display every icon into view with same params as target view have.
    //For this purpose there is additional view 'playerChannelLogoLoad'. It works also if view is INVISIBLE.
    //After view is layout (have nonzero size) and list of channel icons is loaded (nonzero size) sequential loading of all icons is started
    //In both cases success/fail next icon will be loaded
    private var preLoadIconsIndex = 0
    private var preLoadIconsListSize = 0
    private var preLoadIconsViewReady = false
    private var preLoadIconsListReady = false
    private var preLoadIconsLog = false
    private var preLoadIconsFailed = 0
    private var preLoadIconsStartTime = 0L

    init {
        val lifecycleOwner = context as LifecycleOwner
        binding.lifecycleOwner = lifecycleOwner
        binding.viewModel = viewModel
        viewModel.setEpgFocusedEventStore(context as? EpgView.EpgFocusedEventStore)
        viewModel.setObserversInLifecycle(lifecycleOwner)
        viewModel.ageRatings.observe(lifecycleOwner, Observer { binding.mediaRestrictionIcons.viewModel.setContentIcons(it) })
        hidingHelper.listener = this

        viewModel.channelIcons.observe(lifecycleOwner, Observer {
            if (preLoadIconsLog) Timber.tag("PreLoadChannelIcons").d("Icons: ${it.size}")
            if (it.isNotEmpty()) {
                preLoadIconsListSize = it.size
                preLoadIconsListReady = true
                preLoadIconsStart()
            }
        })
        binding.playerChannelLogoLoad.addOnLayoutChangeListener(object : OnLayoutChangeListener {
            override fun onLayoutChange(v: View?, left: Int, top: Int, right: Int, bottom: Int, oldLeft: Int, oldTop: Int, oldRight: Int, oldBottom: Int) {
                v?.let {
                    if(it.height > 0 && it.width > 0) {
                        preLoadIconsViewReady = true
                        preLoadIconsStart()
                        binding.playerChannelLogoLoad.removeOnLayoutChangeListener(this)
                    }
                }
            }
        })
    }

    private fun preLoadIconsStart() {
        if (preLoadIconsLog) Timber.tag("PreLoadChannelIcons")
            .d("Starting... View ready: $preLoadIconsViewReady, List ready: $preLoadIconsListReady")
        if (preLoadIconsViewReady && preLoadIconsListReady) {
            Timber.tag("PreLoadChannelIcons").d("Started")
            preLoadIconsStartTime = DateTime.now().millis
            preLoadIconsLoad()
        }
    }

    private fun preLoadIconsNext() {
        if(preLoadIconsIndex < preLoadIconsListSize - 1) {
            preLoadIconsIndex++
            CoroutineScope(Dispatchers.Main).launch {
                preLoadIconsLoad()
            }
        } else {
            Timber.tag("PreLoadChannelIcons")
                .d("All $preLoadIconsListSize preloaded in ${DateTime.now().millis - preLoadIconsStartTime}, including $preLoadIconsFailed failed")
        }
    }

    private fun preLoadIconsLoad() {
        if ((context as? FragmentActivity)?.isDestroyed == true) return
        val url = viewModel.channelIcons.value?.getOrNull(preLoadIconsIndex)
        if (preLoadIconsLog) Timber.tag("PreLoadChannelIcons")
            .d("Loading: $preLoadIconsIndex, URL: $url")
        try {
            Glide.with(context)
                .load(url)
                .listener(object : RequestListener<Drawable> {
                    override fun onLoadFailed(
                        e: GlideException?,
                        model: Any?,
                        target: Target<Drawable>?,
                        isFirstResource: Boolean,
                    ): Boolean {
                        if (preLoadIconsLog) Timber.tag("PreLoadChannelIcons").d("fail")
                        preLoadIconsFailed++
                        preLoadIconsNext()
                        return false
                    }

                    override fun onResourceReady(
                        resource: Drawable?,
                        model: Any?,
                        target: Target<Drawable>?,
                        dataSource: DataSource?,
                        isFirstResource: Boolean,
                    ): Boolean {
                        if (preLoadIconsLog) Timber.tag("PreLoadChannelIcons").d("success")
                        preLoadIconsNext()
                        return false
                    }
                })
                .signature(ObjectKey(Flavor().glideCacheExpirationInterval))
                .into(binding.playerChannelLogoLoad)
        } catch (e: IllegalArgumentException) {
            Timber.tag("PlayerUILive").e(e.stackTraceToString())
        }
    }

    override fun onPlayerControlVisible(visible: Boolean) {
        if(!visible) viewModel.cancelSoftZap()
    }

    override val standardHelpers: List<StandardControlsHelper> = listOf(
            timerHelper, hidingHelper
    )

    override fun getToastBottomOffset(): Int {
        return super.getToastBottomOffset() + ((binding.bottomGuideline.layoutParams as? ConstraintLayout.LayoutParams)?.guideEnd ?: 0)
    }
}