package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.graphics.ColorMatrix
import android.graphics.ColorMatrixColorFilter
import android.graphics.Rect
import android.util.AttributeSet
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.cardview.widget.CardView
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.leanback.widget.HorizontalGridView
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.tvlauncher.BR
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.data.AppListItem
import com.twentyfouri.tvlauncher.data.channel.ExternalRowItem
import com.twentyfouri.tvlauncher.utils.ScaleFocusChangeListener
import com.twentyfouri.tvlauncher.viewmodels.RowItemViewModel
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

class RowItemView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : LinearLayout(context, attrs, defStyleAttr), KoinComponent {

    private val cardImage: ImageView
    val card: CardView
    val viewModel: RowItemViewModel by inject()
    var title: TextView? = null //because row_item_google_play does not have card_title
    private var positionRelevantForFocusHandler: Int = 0
    private var markedAsDisabled: Boolean = false

    init {
        requireNotNull(attrs) { "RowItemView does not have layout" }
        val a = context.obtainStyledAttributes(attrs, R.styleable.RowItemView, 0, 0)
        val layoutId = a.getResourceId(R.styleable.RowItemView_layout, 0)
        require(layoutId != 0) { "RowItemView does not have layout" }
        a.recycle()

        val binding = DataBindingUtil.inflate<ViewDataBinding>(
                LayoutInflater.from(context),
                layoutId,
                this,
                true
        ).apply {
            lifecycleOwner = context as LifecycleOwner
        }
        binding.setVariable(BR.viewModel, viewModel)

        cardImage = binding.root.findViewById(R.id.card_image) //because we don't know which binding there will be
        card = binding.root.findViewById(R.id.card)
    }

    private fun setupOnClickListenerObserver() { viewModel.onClickListener.observe(context as LifecycleOwner, Observer { setOnClickListener(it) }) }

    private fun setupOnLongClickListenerObserver() { viewModel.onLongClickListener.observe(context as LifecycleOwner, Observer { setOnLongClickListener(it) }) }

    private fun setupOnKeyEventListenerObserver() { viewModel.onKeyEventListener.observe(context as LifecycleOwner, Observer { setOnKeyListener(it) }) }

    fun onViewDetachedFromWindow() {
        viewModel.onClickListener.removeObservers(context as LifecycleOwner)
        viewModel.onLongClickListener.removeObservers(context as LifecycleOwner)
        viewModel.onKeyEventListener.removeObservers(context as LifecycleOwner)
        viewModel.cardWidth.removeObservers(context as LifecycleOwner)
        onFocusChangeListener = null
    }

    fun onViewAttachedToWindow() {
        setupOnClickListenerObserver()
        setupOnLongClickListenerObserver()
        setupOnKeyEventListenerObserver()
        onFocusChangeListener = ScaleFocusChangeListener(positionRelevantForFocusHandler)
        executeOnFocusChange()
    }

    override fun onWindowVisibilityChanged(visibility: Int) {
        super.onWindowVisibilityChanged(visibility)
        if (visibility == View.VISIBLE) disableItemIfObsolete()
    }

    fun bind(mediaItem: SmartMediaItem, position: Int, isGrid: Boolean = false) {
        viewModel.position = position
        viewModel.isFirst = (position == 0)
        isGrid
            .ifTrue { viewModel.setGridItem(mediaItem) }
            .ifElse { viewModel.setItem(mediaItem) }
        // ScaleFocusChangeListener manipulated visibility of title view
        // so when title is not utilized in any way, let it be set as null
        // (null -> when large scale animation is used with metadata)

        isSmallLayout(mediaItem, isGrid)
            .ifTrue {
                title = findViewById(R.id.card_label)
                positionRelevantForFocusHandler = position
            }
            .ifElse {
                title = null
                positionRelevantForFocusHandler = 0 //position is set to zero, so there is always the left edge animation
            }
        viewModel.cardWidth.observe(context as LifecycleOwner, Observer { aCardWidth ->
            if (position == 0) {
                TypedValue()
                    .also { resources.getValue(R.dimen.small_card_focused_percentage, it, true) }
                    .let { it.getFraction(1f, 1f) }
                    .also { viewModel.setLabelFirstTileShift(aCardWidth.times(it).minus(aCardWidth)) }
            }
        })
        disableItemIfObsolete()
    }

    private fun isSmallLayout(mediaItem: SmartMediaItem, isGrid: Boolean) = when {
        isGrid -> true
        mediaItem.type == SmartMediaType.LIVE_CHANNEL -> true
        else -> false
    }

    fun bind(item: AppListItem, position: Int) {
        viewModel.position = position
        viewModel.setItem(item)
        title = findViewById(R.id.card_label)
        positionRelevantForFocusHandler = position
        viewModel.cardWidth.observe(context as LifecycleOwner, Observer { aCardWidth ->
            if (position == 0) viewModel.setLabelFirstTileShift(aCardWidth.times(1.4f).minus(aCardWidth))
        })
    }

    fun bindGooglePlay(item: AppListItem, position: Int) {
        viewModel.position = position
        viewModel.setItemGooglePlay(item)
        positionRelevantForFocusHandler = position
    }

    private fun executeOnFocusChange() {
        viewModel.appChannelProgram.value.ifNotNull{ return }
        onFocusChangeListener.onFocusChange(this, viewModel.isFocused.value ?: false) //needed to properly draw recycled row
    }

    fun bindAppChannelProgram(item: ExternalRowItem, position: Int) {
        viewModel.position = position
        viewModel.setAppChannelProgram(item)
        positionRelevantForFocusHandler = 0
    }

    override fun getFocusedRect(r: Rect?) {
        super.getFocusedRect(r)
        // this will stretch focusable area of item over whole horizontal grid width -> will be added to focusables during search
        if (parent is HorizontalGridView && !(parent as HorizontalGridView).hasFocus()) {
            r?.left = 0
            r?.right = (parent as HorizontalGridView).width
        }
    }

    private fun disableItemIfObsolete() {
        when (viewModel.getIsItemObsolete()) {
            true -> {
                markedAsDisabled.ifTrue { return }
                val grayscaleMatrix = ColorMatrix()
                grayscaleMatrix.setSaturation(0f)
                cardImage.alpha = 0.4f
                cardImage.colorFilter = ColorMatrixColorFilter(grayscaleMatrix)
                markedAsDisabled = true
            }
            false -> {
                markedAsDisabled.ifFalse { return }
                val grayscaleMatrix = ColorMatrix()
                grayscaleMatrix.setSaturation(1f)
                cardImage.alpha = 1f
                cardImage.colorFilter = ColorMatrixColorFilter(grayscaleMatrix)
                markedAsDisabled = false
            }
            null -> return
        }
    }
}