package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.os.Handler
import android.util.AttributeSet
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.view.ViewTreeObserver
import androidx.core.view.isVisible
import androidx.databinding.DataBindingUtil
import androidx.leanback.widget.BaseGridView
import androidx.leanback.widget.VerticalGridView
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.RecyclerView
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.adapters.RowItemAdapter
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_UI_LOG
import com.twentyfouri.tvlauncher.databinding.RowViewBinding
import com.twentyfouri.tvlauncher.utils.LastItemSpaceDecoration
import com.twentyfouri.tvlauncher.viewmodels.RowItemViewModel
import com.twentyfouri.tvlauncher.viewmodels.RowViewModel
import com.twentyfouri.tvlauncher.viewmodels.SectionViewModel
import timber.log.Timber

class RowView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FocusHandleFrameLayout(context, attrs, defStyleAttr), ViewTreeObserver.OnGlobalFocusChangeListener {

    private var nullableBinding: RowViewBinding?
    val binding: RowViewBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")
    private var nullableRowViewModel: RowViewModel? = null
    private val rowViewModel: RowViewModel get() = nullableRowViewModel ?: throw IllegalStateException("unbinded RowView")
    private val sectionViewModel: SectionViewModel get() = rowViewModel.sectionViewModel
    private var layoutType: Int = ROW_VIEW_LAYOUT_BASIC
    private var onChildAttachStateChangeListener: RecyclerView.OnChildAttachStateChangeListener? = null
    private val rowItemViewModels: MutableList<RowItemViewModel> = mutableListOf()
    private var focusedChildIsMine = false

    init {
        nullableBinding = DataBindingUtil.inflate<RowViewBinding>(
                LayoutInflater.from(context),
                R.layout.row_view,
                this,
                true)
        binding.lifecycleOwner = context as LifecycleOwner
        viewTreeObserver.addOnGlobalFocusChangeListener(this)
    }

    fun bind(rowViewModel: RowViewModel) {
        nullableRowViewModel = rowViewModel
        privateAttach()
    }

    fun onDetached() {
        //do not do anything here, it drastically slows scrolling
    }

    fun onAttached() {
        //do not do anything here, it drastically slows scrolling
    }

    fun onRowAdapterDetached() {
        nullableRowViewModel ?: return
        clearChildrenObserver()
        clearRowDataObservers()
        binding.rowList.adapter = null
        nullableRowViewModel = null
        binding.metadata.onDestroyView()
        binding.rowList.removeAllViews()

    }

    fun onDestroyView() {
        viewTreeObserver.removeOnGlobalFocusChangeListener(this)
        onRowAdapterDetached()
        nullableBinding = null
    }

    private fun privateAttach() {
        binding.rowViewModel = rowViewModel
        binding.sectionViewModel = sectionViewModel

        when (sectionViewModel.scrollType) {
            SectionViewModel.ScrollType.DEFAULT -> setScrollDefault()
            SectionViewModel.ScrollType.NONE -> setScrollNone()
            else -> setScrollFixedLeftEdge()
        }

        val adapter = RowItemAdapter(sectionViewModel.pageSection)
        if (sectionViewModel.isPaged) {
            adapter.setHasStableIds(true)
        } else {
            adapter.setHasStableIds(false)
        }
        binding.rowList.adapter = adapter

        when (layoutType) {
            ROW_VIEW_LAYOUT_ANIMATED -> rowViewModel.setUseAnimatedLayout(true)
            else -> rowViewModel.setUseAnimatedLayout(false)
        }

        setupChildrenObserver(adapter)
        setupRowDataObservers(adapter)
    }

    fun setLayoutStyle(layoutType: Int = ROW_VIEW_LAYOUT_BASIC) { this.layoutType = layoutType }

    private fun clearChildrenObserver() {
        //seems the AdapterDataObserver does not need to be unregistered, but if there will be another memory leak
        //keep this option here
//        binding.rowList.adapter.unregisterAdapterDataObserver()
        (0 until binding.rowList.childCount).forEach { i ->
            onChildAttachStateChangeListener?.onChildViewDetachedFromWindow(binding.rowList.getChildAt(i)) }
        onChildAttachStateChangeListener?.also { binding.rowList.removeOnChildAttachStateChangeListener(it) }
        onChildAttachStateChangeListener = null
    }

    private fun setupChildrenObserver(adapter: RowItemAdapter) {
        //some rows need to be scrolled to first item after their data are refreshed on resume,
        // typically Last Watched watched row where we want to always focus most recent event
        binding.rowList.adapter?.registerAdapterDataObserver(object: RecyclerView.AdapterDataObserver(){
            override fun onChanged() {
                if (sectionViewModel.resetRow) {
                    //scroll to first item is delayed because in some cases this scrolling happens before the new item is even drawn
                    binding.rowList.postDelayed({binding.rowList.setSelectedPositionSmooth(0)},100)
                    //resetting of row is disabled only after section is loaded
                    //(because onChanged() is also triggered upon navigating to home page)
                    if (sectionViewModel.sectionLoaded) sectionViewModel.resetRow = false
                }
                super.onChanged()
            }
        })

        onChildAttachStateChangeListener = object : RecyclerView.OnChildAttachStateChangeListener {
            var pendingFocus: View? = null
            override fun onChildViewAttachedToWindow(view: View) {
                val rowItemViewModel = (view as? RowItemView)?.viewModel ?: return
                rowItemViewModels.add(rowItemViewModel)
                rowViewModel.isFocused.addSource(rowItemViewModel.isFocused) { rowItemIsFocused ->
                    if (rowItemIsFocused == null) return@addSource
                    rowViewModel.isFocused.value = findFocus() != null
                    rowViewModel.isFilterFocused.value = binding.filters.findFocus() != null
                }
                rowItemViewModel.isFocused.observe(context as LifecycleOwner, Observer { rowItemIsFocused ->
                    if (rowItemIsFocused == true) {
                        if(sectionViewModel.isGrid) handleGridVerticalFocus(rowItemViewModel)
                        rowItemViewModel.mediaItem.value?.also {
                            if (adapter.isChannelRow) binding.metadata.bindOnChannelRow(it.reference, it.channelReference)
                            else binding.metadata.bindOnRow(it.reference, it.channelReference)
                        }
                        rowItemViewModel.appChannelProgram.value?.also {
                            rowItemViewModel.isFocused.value.ifTrue { binding.metadataExternalChannelProgram.bind(it) }
                        }
                    }
                })
                //refresh of section data
                rowItemViewModel.mediaItem.observe(context as LifecycleOwner, Observer {
                    rowItemViewModel.isFocused.value.ifTrue {
                        if (adapter.isChannelRow) binding.metadata.bindOnChannelRow(it.reference, it.channelReference)
                        else binding.metadata.bindOnRow(it.reference, it.channelReference)
                    }
                })
                rowItemViewModel.appChannelProgram.observe(context as LifecycleOwner, Observer {
                    rowItemViewModel.isFocused.value.ifTrue { binding.metadataExternalChannelProgram.bind(it) }
                })
            }

            private fun handleGridVerticalFocus(rowItemViewModel: RowItemViewModel) {
                val verticalIndex = sectionViewModel.rowViewModels.value?.indexOf(rowViewModel)
                if (sectionViewModel.verticalFocusIndex != verticalIndex) {
                    //different row in section was focused so force focus correct item
                    minOf(sectionViewModel.horizontalFocusIndex, binding.rowList.childCount - 1)
                        .let { binding.rowList.getChildAt(it) }
                        ?.also {
                            pendingFocus = it
                            //immediate requestFocus causing troubles to ScaleFocusChangeListener and could leave multiple items enlarged
                            //pendingFocus is prevention from focusing obsolete items in case of quick scroll or holding Up/Down button
                            it.post {
                                if (it == pendingFocus) {
                                    it.requestFocus()
                                    pendingFocus = null
                                }
                            }
                        }
                    sectionViewModel.verticalFocusIndex = verticalIndex ?: 0
                }

                (rowItemViewModel.mediaItem.value ?: rowItemViewModel.appListItem.value)?.also {
                    val horizontalIndex = rowViewModel.rowData.value?.indexOf(it)
                    //store horizontal focus index if we are still on the same row
                    if (sectionViewModel.verticalFocusIndex == verticalIndex) sectionViewModel.horizontalFocusIndex = horizontalIndex ?: 0
                }
            }

            override fun onChildViewDetachedFromWindow(view: View) {
//                Log.d("Rows3", "detach child ${sectionViewModel.pageSection.label}")
                val rowItemViewModel = (view as? RowItemView)?.viewModel ?: return
                rowItemViewModels.remove(rowItemViewModel)
                rowItemViewModel.setIsFocused(null)
                rowViewModel.isFocused.removeSource(rowItemViewModel.isFocused)
                rowItemViewModel.isFocused.removeObservers(context as LifecycleOwner)
                rowItemViewModel.mediaItem.removeObservers(context as LifecycleOwner)
                rowItemViewModel.appChannelProgram.removeObservers(context as LifecycleOwner)
            }
        }
        binding.rowList.addOnChildAttachStateChangeListener(onChildAttachStateChangeListener!!)
    }

    private fun clearRowDataObservers() {
        //TODO there should be either ViewLifecycleOwner from the fragment, thus the observers will be dropped when the fragment's wiew is destroyed
        //TODO or the observers can be dropped onDetach of the View
        nullableRowViewModel ?: return //this might happen if the detach is called before bind (e.g. goToDetail+back quickly)
        sectionViewModel.sectionDataNextPageReceived.removeObservers(context as LifecycleOwner)
        sectionViewModel.rowViewModels.removeObservers(context as LifecycleOwner)
        sectionViewModel.filterViewHolders.removeObservers(context as LifecycleOwner)
        rowViewModel.rowData.removeObservers(context as LifecycleOwner)

        //TODO the nVidia rows functionality is disabled for a moment
//        nullableBinding?.rowViewModel?.sectionOptionsRowFocused?.removeObservers(context as LifecycleOwner)
//        nullableBinding?.rowViewModel?.sectionTitleRowFocused?.removeObservers(context as LifecycleOwner)
    }

    private fun setupRowDataObservers(adapter: RowItemAdapter) {
        //this notifies row adapter each time next page data is loaded.
        //notifyDataSetChanged is here necessary to update view and redraw LastItemSpaceDecoration()to actual new last item.

        //TODO the nVidia rows functionality is disabled for a moment
//        binding.rowViewModel?.sectionOptionsRowFocused?.observe(context as LifecycleOwner, Observer {
//            var wasTranslated = false
//            if (it) {
//                RowPageAnimationHelper.doLeftOptionsFocusedAnimation(binding)
//                wasTranslated = true
//            } else {
//                if (wasTranslated) {
//                    RowPageAnimationHelper.doLeftOptionsNotFocusedAnimation(binding)
//                    wasTranslated = false
//                }
//            }
//        })
//
//        if (layoutType == ROW_VIEW_LAYOUT_ANIMATED) {
//            binding.rowViewModel?.sectionTitleRowFocused?.observe(context as LifecycleOwner, Observer {
//                if (it) {
//                    RowPageAnimationHelper.doCategoryRowFocusedAnimation(binding)
//                } else {
//                    RowPageAnimationHelper.doCategoryContentFocusedAnimation(binding)
//                }
//
//            })
//        }

        sectionViewModel.sectionDataNextPageReceived.observe(context as LifecycleOwner, Observer {
            if (it) {
                val oldItemsCount = binding.sectionViewModel?.getOldDataTotalCount()
                if (oldItemsCount != null && adapter.itemCount <= oldItemsCount) return@Observer
                binding.rowList.post {
                    if (oldItemsCount != null) {
                        adapter.notifyItemRangeInserted(oldItemsCount, adapter.itemCount - oldItemsCount) //add new page items
                        adapter.notifyItemChanged(oldItemsCount-1) // redraw last item of previous page
                    } else {
                        adapter.notifyDataSetChanged()
                    }
                }
                keepFocusedItemScaled()
            }
        })

        sectionViewModel.rowViewModels.observe(context as LifecycleOwner, Observer {
            binding.rowViewModel?.updateFromNewRowModels(it)
            binding.narrowFilterButton.let { button ->
                button.isVisible.ifTrue {
                    button.setOnClickListener {
                        binding.rowList.requestFocus()
                        (findAncestorVerticalGridView(binding.rowList) as? HomeListView)?.scrollToPosition(0)
                        //TODO: tryFocusFilters() is not working (PRJ1010YOU-2760) uncomment if it gets fixed
//                        ((findAncestorVerticalGridView(binding.rowList) as? HomeListView)?.adapter as? RowAdapter)?.tryFocusFilters()
                    }
                }
            }
        })

        //TODO unify this data observer with the one in RowViewViewModel
        rowViewModel.rowData.observe(context as LifecycleOwner, Observer { result ->
//            Log.d("Rows3", "RowView rowData loaded ... index: ${binding.rowViewModel?.index} ... size:${result.size}")
            adapter.submitList(result)
            if (sectionViewModel.isPaged.not()) {
                    // notifyItemRangeChanged results in smoother animations in recycler without stable IDs
                 adapter.notifyItemRangeChanged(0, result.size)
            } else {
                adapter.notifyDataSetChanged()
            }
            //TODO try to find better solution
            //unfortunately internal implementation is trying to maintain last selected item even after dataSetChanged and forcing selected item.
            //Setting selectedPosition to 0 is useless even in onLayoutChange and also in post{}. Everything is too early
            //and internal code override those attempts and set selected position to position where is last item newly positioned.
            //It happens usually around 100ms after notifyDataSetChanged is called. This is why 300ms delay for reset is used.
            if(shouldResetSelection(adapter.currentList, result)) {
                Handler().postDelayed({
                    binding.rowList.selectedPosition = 0
                }, 300)
            }
            sectionViewModel.sectionLoaded = true
            keepFocusedItemScaled()
            if (result.isNotEmpty() && sectionViewModel.scrollType != SectionViewModel.ScrollType.NONE) {
                val screenWidth = context.resources.displayMetrics.widthPixels
                val itemWidth: Int? = context.resources.getDimensionPixelSize(R.dimen.small_card_width)
                val itemMarginWidth = context.resources.getDimensionPixelSize(R.dimen.row_space_between_tiles)
                /* When we scroll to the right after we reach the center of the screen,
                   if the row has enough items, will scroll them to the left.
                   But if we don't have enough items it will scroll the free remaining space.
                   This space is the offset. */
                val offset = 2
                itemWidth?.let {
                    val remainderSpace: Int = screenWidth.rem(it + itemMarginWidth) + offset
                    if (remainderSpace < (itemWidth + itemMarginWidth)) {
                        val decorationSpace = (((2 * itemMarginWidth) + itemWidth) - remainderSpace)
                        binding.rowList.addItemDecoration(
                            LastItemSpaceDecoration(
                                TypedValue.applyDimension(
                                    TypedValue.COMPLEX_UNIT_DIP,
                                    decorationSpace.toFloat(),
                                    resources.displayMetrics
                                ).toInt()
                            )
                        )
                    }
                }
            }
        })
    }

    private fun shouldResetSelection(old: List<Any>, new: List<Any>): Boolean {
        //reset currently should happen only for LAST_WATCHED_ROW and in case count or order of channels is changed
        //when only events changed reset should not be performed
        if(Flavor().getPlaylistType(binding.sectionViewModel?.pageSection) != PlaylistType.LAST_WATCHED_CHANNELS) return false
        if(old.size != new.size) return true
        val oldOrder = old.map { (it as? SmartMediaItem)?.channelNumber }
        val newOrder = new.map { (it as? SmartMediaItem)?.channelNumber }
        if(oldOrder != newOrder) return true
        return false
    }

    private fun setScrollFixedLeftEdge() {
        setScrollDefault()
        binding.rowList.apply {
            when (layoutType) {
                ROW_VIEW_LAYOUT_ANIMATED -> {
                    itemAlignmentOffset = 150
                    itemAlignmentOffsetPercent = VerticalGridView.ITEM_ALIGN_OFFSET_PERCENT_DISABLED
                    windowAlignment = BaseGridView.WINDOW_ALIGN_NO_EDGE
                    windowAlignmentOffsetPercent = VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED
                }
                else -> {
                    itemAlignmentOffsetPercent = VerticalGridView.ITEM_ALIGN_OFFSET_PERCENT_DISABLED
                    windowAlignment = BaseGridView.WINDOW_ALIGN_NO_EDGE
                    windowAlignmentOffsetPercent = VerticalGridView.WINDOW_ALIGN_OFFSET_PERCENT_DISABLED
                }
            }
        }
    }

    private fun setScrollDefault() {
        binding.rowList.apply {
            when (layoutType) {
                ROW_VIEW_LAYOUT_ANIMATED -> {
                    itemAlignmentOffset = 150
                    itemAlignmentOffsetPercent = 40f
                    isItemAlignmentOffsetWithPadding = true
                    windowAlignment = BaseGridView.WINDOW_ALIGN_BOTH_EDGE
                    windowAlignmentOffset = resources.getDimensionPixelSize(R.dimen.row_first_item)
                    windowAlignmentOffsetPercent = 40f
                }
                else -> {
                    itemAlignmentOffset = 0
                    itemAlignmentOffsetPercent = 40f
                    isItemAlignmentOffsetWithPadding = true
                    windowAlignment = BaseGridView.WINDOW_ALIGN_BOTH_EDGE
                    windowAlignmentOffset = resources.getDimensionPixelSize(R.dimen.row_first_item)
                    windowAlignmentOffsetPercent = 40f
                }
            }
        }
    }

    private fun setScrollNone() {
        binding.rowList.apply {
            when (layoutType) {
                ROW_VIEW_LAYOUT_ANIMATED -> {
                    itemAlignmentOffset = 150
                    isItemAlignmentOffsetWithPadding = true
                    windowAlignment = BaseGridView.WINDOW_ALIGN_BOTH_EDGE
                    windowAlignmentOffsetPercent = 100f
                }
                else -> {
                    itemAlignmentOffset = 0
                    isItemAlignmentOffsetWithPadding = true
                    windowAlignment = BaseGridView.WINDOW_ALIGN_BOTH_EDGE
                    windowAlignmentOffsetPercent = 100f
                }
            }
        }
    }

    private fun keepFocusedItemScaled(){
        if (binding.rowList.focusedChild != null) {
            binding.rowList.focusedChild.onFocusChangeListener?.onFocusChange(focusedChild, true)
        }
    }

    fun moveRowItem(from: Int, to: Int) {
        var offset = 1
        val adapter = (binding.rowList.adapter as RowItemAdapter)
        val allowedMoveInterval = Flavor().getBannedMoveIntervalForSection(sectionViewModel.pageSection)
        if (from >= allowedMoveInterval.first && from <= adapter.itemCount - 1
            && to >= allowedMoveInterval.second && to <= adapter.itemCount - 2) {
            val currentList = adapter.currentList.toMutableList()
            val fromItem = currentList[from]
            currentList[from] = currentList[to]
            currentList[to] = fromItem
            adapter.submitList(currentList)
            val positionDifference = to - from
            if (positionDifference > 0) {
                offset = -1
            }
            binding.sectionViewModel?.updateFavoriteAppsPriority(currentList.dropLast(1))
            adapter.notifyItemMoved(to + offset, from)
        }
    }

    private fun findAncestorVerticalGridView(v: View): View? {
        var target = v.parent
        while (target != null) {
            if (target is HomeListView) {
                return target
            }
            target = target.parent
        }
        return null
    }

    companion object {
        const val ROW_VIEW_LAYOUT_BASIC = 0
        const val ROW_VIEW_LAYOUT_ANIMATED = 1
    }

    override fun onGlobalFocusChanged(oldFocus: View?, newFocus: View?) {
        if(newFocus is RowItemView && rowItemViewModels.contains(newFocus.viewModel)) {
            if(!focusedChildIsMine) {
                Timber.tag(TAG_UI_LOG).d("Row ${sectionViewModel.sectionIndex}-${rowViewModel.rowIndex} focused: ${rowViewModel.headerText.value}")
            }
            focusedChildIsMine = true
        } else {
            focusedChildIsMine = false
        }
    }
}