package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.databinding.DataBindingUtil
import androidx.lifecycle.LifecycleOwner
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.androidcore.utils.ResourceImageSpecification
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.viewmodels.SettingsItemViewModel
import com.twentyfouri.tvlauncher.databinding.SettingsItemBinding
import timber.log.Timber

enum class SettingsItemType(val id: Int) {
    //this list must be synchronized with the <attr name="SettingsItemViewType" format="enum"> in attrs.xml
    SWITCH(1),
    LABEL(2),
    ICON(3),
    UNKNOWN(4),
    SECTION_LABEL(5),
    TITLE_WITH_ICON(6),
    TITLE_SUBTITLE(7);
}

class SettingsItemView: ConstraintLayout {
    constructor(context: Context) : super(context) { init(null) }
    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) { init(attrs) }
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) { init(attrs) }

    lateinit var binding : SettingsItemBinding

    private fun init(attrs: AttributeSet?) {
        lateinit var itemType: SettingsItemType
        var titleText: String? = null
        var subtitleText: String? = null
        var labelText: String? = null
        var itemIcon: ImageSpecification? = null

        if (attrs != null) {
            val a = context.obtainStyledAttributes(attrs, R.styleable.SettingsItemView, 0, 0)
            itemType = SettingsItemType.values().find { it.id == a.getInt(R.styleable.SettingsItemView_type, 0) } ?: SettingsItemType.UNKNOWN
            titleText = a.getText(R.styleable.SettingsItemView_titleText)?.toString()
            subtitleText = a.getText(R.styleable.SettingsItemView_subtitleText)?.toString()
            labelText = a.getText(R.styleable.SettingsItemView_labelText)?.toString() ?: if (itemType == SettingsItemType.UNKNOWN) itemType.toString() else null
            val iconResource = a.getResourceId(R.styleable.SettingsItemView_itemIcon, 0)
            if(iconResource != 0) itemIcon = ResourceImageSpecification(iconResource)
            a.recycle()
        }

        binding = DataBindingUtil.inflate<SettingsItemBinding>(
            LayoutInflater.from(context),
            R.layout.settings_item,
            this,
            true
        ).apply {
            viewModel = SettingsItemViewModel(resources)
            lifecycleOwner = context as LifecycleOwner
        }

        binding.viewModel?.setFromData(itemType, labelText, titleText, subtitleText, itemIcon)
    }

    fun setSwitch(checked: Boolean) {
        Timber.tag("GDPR").d("setSwitch.checked: $checked")
        binding.viewModel?.setChecked(checked)
    }

    fun bindLabel(label: String) {
        focusable = View.NOT_FOCUSABLE
        binding.viewModel?.setFromData(itemType = SettingsItemType.SECTION_LABEL, labelText = label)
    }
}