package com.twentyfouri.tvlauncher.common.analytics

import android.app.Activity
import android.app.Service
import android.content.Context
import android.net.NetworkCapabilities
import android.net.wifi.WifiManager
import android.os.Bundle
import com.google.android.exoplayer2.trackselection.MappingTrackSelector
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.npaw.youbora.lib6.YouboraLog
import com.npaw.youbora.lib6.plugin.Options
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.error.*
import com.twentyfouri.smartmodel.model.media.SmartMediaStream
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.common.BuildConfig
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.data.StreamingType
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import timber.log.Timber
import java.io.InterruptedIOException
import java.net.SocketTimeoutException
import java.util.Timer
import javax.net.ssl.SSLException
import kotlin.concurrent.schedule

class YouboraAnalytics(
        context: Context
) {
    private val options = Options().apply {
        accountCode = Flavor().youboraAccountCode
    }
    val plugin = YouboraPlugin(options, context.applicationContext).apply {
        options.contentMetrics = Bundle()
    }
    private var appRecreated: Boolean = true
    private var deviceColdBoot = false
    private val youboraEnabled = Flavor().isYouboraEnabled
    private val deviceInfo = DeviceInfo(context, object : InfoUpdated {
        override suspend fun onInfoUpdated() {
            youboraEnabled.ifTrue {
                setPlaybackRelatedMetrics()
            }
        }
    }, Flavor().deviceInfoConfig)
    private var isUserSet: Boolean = false
    private val appContext = context.applicationContext
    private val delayedEvents: MutableList<AnalyticsEvent> = mutableListOf()

    init {
        setAppVersion()
        if (BuildConfig.IS_DEBUG) {
            YouboraLog.setDebugLevel(YouboraLog.Level.VERBOSE)
        }
    }

    //Client setup region

    private fun setAppVersion() {
        plugin.options.contentCustomDimension3 = deviceInfo?.launcherVersion
        plugin.options.contentCustomDimension4 = deviceInfo?.wizardVersion
    }

    fun setupUser(smartApi: FlowSmartApi?) {
        options.isEnabled = Flavor().isYouboraEnabled
        if (options.isEnabled) {
            options.contentCustomDimension5 = DeviceId.getDeviceId(appContext)
            val userProfile = runBlocking {
                try {
                    smartApi?.getProfiles()?.last()?.lastOrNull()
                } catch (e: Exception) {
                    Timber.w(e.stackTraceToString())
                    null
                }
            }
            val setupData = runBlocking { SetupDataRepository.getInstance(appContext).getSetupData() }
            userProfile?.let { profile ->
                options.username = Flavor().getUserProfileId(profile) ?: Flavor().getUsername(setupData) ?: "anonymous"
                val userType = profile.extras.get("customerProfileType").getString() ?: Flavor().getUserType(setupData)
                setYouboraUserType(userType)
                val householdId = profile.extras.get("externalHouseholdId").getString()
                options.sessionMetrics = Bundle().apply { putString("externalId", householdId) }
            } ?: setupUserFromSetupData(setupData)

            isUserSet = true
            beginSession()
            setNetworkRelatedDimensions(NetworkInfo.getActiveConnectionInfo(appContext))
            executePendingReports()
        }
    }

    private fun checkPluginSetup() {
        if (!youboraEnabled) return
        if (plugin.username.isNullOrBlank() || plugin.infinity.flags.isStarted.not()){
            setupUser(null)
        }
        if (isSessionExpired()){
            endSession() //End expired session
            beginSession() //Begin new session
        }
    }

    private fun setupUserFromSetupData(setupData: String?) {
        options.username = Flavor().getUsername(setupData) ?: "anonymous"
        setYouboraUserType(Flavor().getUserType(setupData))
    }

    private fun setYouboraUserType(userType: String?) = when (userType){
        "1" -> options.contentCustomDimension1 = "FTTH"
        "2" -> options.contentCustomDimension1 = "OTT"
        else -> {}
    }

    fun setActivity(activity: Activity?){
        activity?.let { plugin.activity = activity }
    }

    //endregion

    //Player region

    fun detachPlayer() {
        deviceInfo.stopAutoUpdate()
        plugin.removeAdapter()
    }

    fun attachPlayer(
        player: SmartPlayer,
        streamingType: StreamingType,
        isInStartoverMode: Boolean,
        stream: SmartMediaStream,
        contentChannel: String?,
        activity: Activity?,
        reference: SmartMediaReference?
    ) {
        checkPluginSetup()
        deviceInfo.startAutoUpdate(appContext, DEVICE_INFO_SEND_INTERVAL)
        if (plugin.options.isEnabled && activity != null) {
            when (streamingType){
                StreamingType.CATCHUP_IN_LIVE_PLAYER,
                StreamingType.EVENT,
                StreamingType.VOD -> {
                    plugin.options.contentIsLive = false
                    plugin.options.contentType = "Catch Up"
                }
                StreamingType.CHANNEL -> {
                    if (isInStartoverMode){
                        plugin.options.contentIsLive = false
                        plugin.options.contentType = "Start Over"
                    } else {
                        plugin.options.contentIsLive = true
                        plugin.options.contentType = ""

                    }
                }
            }
            plugin.options.contentResource = stream.primaryUrl
            plugin.options.contentId = "${reference.toString()}/${stream.reference}" //AssetId/FileId
            plugin.options.contentChannel = contentChannel ?: ""
            val adapter = YouboraAdapter(player)
            adapter.setCustomEventLogger(player.trackSelector as? MappingTrackSelector)
            adapter.bandwidthMeter = player.bandwidthMeter
            plugin.adapter = adapter
        }
    }

    //endregion

    // Session events region

    fun resolveResumeState() {
        if (!youboraEnabled) return
        CoroutineScope(Dispatchers.IO).launch { deviceInfo?.update(appContext) }
        if (!appRecreated) return
        appRecreated = false
        awaitForBootBroadcast {
            SharedPreferencesUtils.getAppCrashData().let {
                if (it.isNotEmpty()) {
                    Timber.d("Report event device crashed")
                    fireOrDelayEvent(AnalyticsEvent(appContext.getString(R.string.analytics_launcher_crash)))
                    return@awaitForBootBroadcast
                }
            }
            SharedPreferencesUtils.getLastShutDownTime().let {
                if (it + REBOOT_TIME_INTERVAL.toLong() > System.currentTimeMillis()) {
                    Timber.d("Report event device rebooted")
                    fireOrDelayEvent(AnalyticsEvent(appContext.getString(R.string.analytics_device_reboot)))

                } else if (deviceColdBoot) {
                    Timber.d("Report event device cold booted")
                    fireOrDelayEvent(AnalyticsEvent(appContext.getString(R.string.analytics_device_coldboot)))
                    return@awaitForBootBroadcast
                }
                if (it <= -1) {
                    Timber.d("Report event launcher restarted")
                    fireOrDelayEvent(AnalyticsEvent(appContext.getString(R.string.analytics_launcher_restart)))
                    return@awaitForBootBroadcast
                }
            }
        }
    }

    fun resolveConnectionState(state: NetworkConnectionState.State?) {
        when (state) {
            NetworkConnectionState.State.ONLINE -> {
                val errorState = SharedPreferencesUtils.getConnectivityErrorState()
                if (errorState.isNotEmpty()) {
                    Timber.d("Youbora send event Network connectivity lost")
                    val event = AnalyticsEvent(
                        name = appContext.getString(R.string.analytics_network_connectivity_lost),
                        dimensions = mutableMapOf(
                            appContext.getString(R.string.network_state) to errorState
                        )
                    )
                    fireOrDelayEvent(event)
                }
            }
            null -> {}
            else -> {
                SharedPreferencesUtils.putConnectivityErrorState(state.name)
            }
        }
    }

    private fun setPlaybackRelatedMetrics() {
        setDeviceInfoMetricsForPlayback(appContext)
        setNetworkDimensionsForPlayback()
    }

    private fun setDeviceInfoMetricsForPlayback(context: Context) {
        if (!plugin.infinity.flags.isStarted) return
        deviceInfo?.let { deviceInfo ->
            plugin.options.contentMetrics?.apply {
                //Memory
                putDouble(context.getString(R.string.analytics_memory_total), deviceInfo.memTotal.toDouble() / (1024.0 * 1024.0))
                putDouble(context.getString(R.string.analytics_memory_available), deviceInfo.memAvailable.toDouble() / (1024.0 * 1024.0))
                putDouble(context.getString(R.string.analytics_memory_percent), deviceInfo.memAvailable.toDouble() / deviceInfo.memTotal.toDouble() * 100.0)
                //CPU
                putInt(context.getString(R.string.analytics_number_of_cpu_cores), deviceInfo.numCores)
                putLong(context.getString(R.string.analytics_cpu_frequency), deviceInfo.maxFreq)
            }
        }
    }

    private fun setNetworkDimensionsForPlayback() {
        val networkInfo = NetworkInfo.getActiveConnectionInfo(appContext)
        when (networkInfo.type) {
            InterfaceType.WIFI -> {
                val wifiInfo = networkInfo as? ActiveConnectionInfoWifi
                plugin.options.contentCustomDimension2 = appContext.getString(R.string.di_wifi)
                wifiInfo?.let { info ->
                    plugin.options.contentCustomDimension7 = if (info.is5GHz) "5GHz" else "2.4GHz"
                    plugin.options.contentCustomDimension8 = info.rssi.toString()
                }
            }
            else -> {
                plugin.options.contentCustomDimension2 = appContext.getString(R.string.di_ethernet)
                plugin.options.contentCustomDimension7 = ""
                plugin.options.contentCustomDimension8 = ""
            }
        }
    }

    fun reportYouboraNetworkConnection(context: Context, info: ActiveConnectionInfo, streamProblem: String? = null) {
        setNetworkRelatedDimensions(info)
        if(!plugin.infinity.flags.isStarted) return
        if(!streamProblem.isNullOrEmpty()) {
            val eventNameStreamProblem = context.getString(R.string.analytics_nq_stream_problem)
            plugin.infinity.fireEvent(eventNameStreamProblem, hashMapOf(Pair(eventNameStreamProblem, streamProblem)))
        }
    }

    private fun setNetworkRelatedDimensions(info: ActiveConnectionInfo){
        val eventNameWifiFreq = appContext.getString(R.string.analytics_nq_wifi_freq)
        val wifiManager = appContext.getSystemService(Service.WIFI_SERVICE) as WifiManager
        val connectionInfo = wifiManager.connectionInfo
        val wifiConfiguration = wifiManager.configuredNetworks.find { it.networkId == connectionInfo.networkId }

        when (info.type) {
            InterfaceType.WIFI -> {
                val activeConnectionInfoWifi = info as ActiveConnectionInfoWifi
                plugin.options.contentCustomDimension6 = wifiConfiguration?.SSID ?: ""
                plugin.options.contentCustomDimension7 = if(activeConnectionInfoWifi.is5GHz) "5GHz" else "2.4GHz"
                plugin.options.contentCustomDimension2 = appContext.getString(R.string.di_wifi)
                plugin.options.contentCustomDimension8 = connectionInfo.rssi.toString()
                plugin.options.contentMetrics?.apply {
                    putInt(eventNameWifiFreq, connectionInfo.frequency)
                }
            }
            else -> {
                plugin.options.contentCustomDimension6 = ""
                plugin.options.contentCustomDimension7 = ""
                plugin.options.contentCustomDimension8 = ""
                plugin.options.contentCustomDimension2 = appContext.getString(R.string.di_ethernet)
                plugin.options.contentMetrics?.apply {
                    putInt(eventNameWifiFreq, 0)
                }
            }
        }
    }

    fun reportScreenOffEvent(context: Context) {
        if (youboraEnabled) {
            Timber.d("Report screen off event")
            fireOrDelayEvent(AnalyticsEvent(context.getString(R.string.analytics_enter_standby)))
            endSession()
        }
    }

    fun reportScreenOnEvent(context: Context) {
        if (youboraEnabled) {
            Timber.d("Report screen on event")
            fireOrDelayEvent(AnalyticsEvent(context.getString(R.string.analytics_leave_standby)))
            beginSession()
        }
    }

    fun reportApiException(
        e: Exception,
        code: String? = null,
        isOnnetService: Boolean = false,
        urlPath: String? = null,
    ) {
        if (!youboraEnabled || plugin.adapter == null) return
        Timber.d("Report API error: $e")
        when {
            e is AppCannotRunException -> {
                plugin.adapter.fireError(
                    code = code,
                    msg = appContext.getString(R.string.analytics_app_cannot_run),
                    exception = e
                )
            }
            e is AppCanRunWithLimitationsException -> {
                plugin.adapter.fireError(
                    code = code,
                    msg = appContext.getString(R.string.analytics_device_out_of_home),
                    exception = e
                )
            }
            e is WrongInputException || e is InvalidCredentialsException -> {
                plugin.adapter.fireError(
                    code = code,
                    msg = appContext.getString(R.string.analytics_login_failure),
                    exception = e
                )
            }
            e is SocketTimeoutException || e.cause is SocketTimeoutException
                    || e is InterruptedIOException || e.cause is InterruptedIOException -> {
                if (urlPath?.isNotEmpty() == true || isOnnetService) {
                    val eventName = if (isOnnetService) {
                        appContext.getString(R.string.analytics_onnet_timeout)
                    } else {
                        appContext.getString(R.string.analytics_api_timeout)
                    }
                    val errorMetadata = JsonObject().apply { addProperty("urlPath", urlPath) }
                    plugin.adapter.fireError(
                        code = code,
                        msg = eventName,
                        errorMetadata = errorMetadata.toString(),
                        exception = e
                    )
                }
            }
            e is JsonParseException -> {
                plugin.adapter.fireError(
                    code = code,
                    msg = appContext.getString(R.string.analytics_api_format_error),
                    exception = e
                )
            }
            e is AnonymousSessionException || e is NotFoundException || e is UnsubscribedChannelException
                    || e is PlaybackRestrictedException || e is SuspendedException
                    || e is InvalidSessionException || e is SSLException -> {}
            e is GeneralApiException -> {
                val errorMetadata = JsonObject().apply { addProperty("urlPath", e.serviceUrl ?: "") }
                plugin.adapter.fireError(
                    code = e.code,
                    msg = appContext.getString(R.string.analytics_general_exception),
                    errorMetadata = errorMetadata.asString,
                    exception = e
                )
            }
        }
    }

    fun reportApiEvent(event: AnalyticsEvent){
        plugin.infinity.fireEvent(event.name)
    }

    fun reportOpenScreen(screen: String) {
        plugin.infinity.fireNav(screen)
    }

    fun reportEpgDatabaseCleared(success: Boolean){
        val eventName = if (success) {
            appContext.getString(R.string.analytics_epg_database_cleared)
        } else {
            appContext.getString(R.string.analytics_epg_database_fail_to_clear)
        }
        fireOrDelayEvent(AnalyticsEvent(eventName))
    }

    fun beginSession() {
        if (youboraEnabled) {
            Timber.d("Report start session fire event")
            plugin.infinity.begin("")
        }
    }

    fun endSession() {
        if (youboraEnabled) {
            Timber.d("Report end session")
            plugin.infinity.end()
        }
    }

    private fun isSessionExpired(): Boolean{
        var expired = false
        if (plugin.fastDataConfig.expirationTime != null
            && plugin.infinity != null
            && plugin.infinity.getLastSent() != null
        ) {
            expired =
                (plugin.infinity.getLastSent()!! + plugin.fastDataConfig.expirationTime * 1000
                        < TimeProvider.nowMs())
        }
        return expired
    }

    fun stopDeviceInfoReporting(){
        deviceInfo?.stopAutoUpdate()
    }

    private fun fireOrDelayEvent(event: AnalyticsEvent) {
        if (isUserSet) {
            plugin.infinity.fireEvent(event.name, event.dimensions, event.values)
        } else {
            delayedEvents.add(event)
        }
    }

    private fun executePendingReports(){
        delayedEvents.forEach{ event ->
            plugin.infinity.fireEvent(event.name, event.dimensions, event.values)
        }
    }

    fun setDeviceIsBooted() {
        deviceColdBoot = true
    }

    fun setNetworkType(context: Context, networkCapabilities: NetworkCapabilities) {
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)){
            plugin.options.contentCustomDimension2 = context.getString(R.string.di_ethernet)
            return
        }
        if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)){
            plugin.options.contentCustomDimension2 = context.getString(R.string.di_wifi)
            return
        }
    }

    private fun awaitForBootBroadcast(timeout: Long = BOOT_COMPLETED_WAIT, callback: () -> Unit) {
        val t0 = System.currentTimeMillis()
        Timer().schedule(0L, 1000L) {
            if (System.currentTimeMillis() - t0 > timeout || deviceColdBoot) {
                callback.invoke()
                cancel()
                return@schedule
            }
        }
    }

    fun reportPlayerError(exception: Exception) {
        when (exception){
            is UnsubscribedChannelException -> plugin.fireError(exception.message, "${exception.cause}, ${exception.code}", exception.stackTraceToString())
            is ConcurrencyLimitationException -> plugin.fireError(exception.message, "${exception.cause}, ${exception.code}", exception.stackTraceToString())
            is SuspendedException -> plugin.fireError(exception.message, "${exception.cause}, ${exception.code}", exception.stackTraceToString())
            is GeneralApiException -> plugin.fireError(exception.message, "${exception.cause}, ${exception.code}", exception.stackTraceToString())
        }
    }

    companion object {
        private const val REBOOT_TIME_INTERVAL = 60/*seconds*/ * 1000/*millis*/
        private const val BOOT_COMPLETED_WAIT: Long = 10/*seconds*/ * 1000/*millis*/
        private const val DEVICE_INFO_SEND_INTERVAL: Long = 10/*seconds*/ * 1000/*millis*/

        @Volatile
        private var instance: YouboraAnalytics? = null
        fun getInstance(context: Context): YouboraAnalytics =
                instance ?: synchronized(this) {
                    instance ?: YouboraAnalytics(context).also {
                        instance = it
                    }
                }
        fun removeInstance() {
            instance = null
        }
    }
}