package com.twentyfouri.tvlauncher.common.data

import android.content.Context
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.ErrorStringMapper
import com.twentyfouri.tvlauncher.common.utils.NetworkConnectionState
import timber.log.Timber
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import java.net.ConnectException

object SmartApiRepository {

    // For Singleton instantiation
    @Volatile
    private var instance: FlowSmartApi? = null
    private var alreadyFailedDoNotTryAgain = false
    private var exception: Exception? = null
    var isServerBusy: Boolean = false

    fun getSmartApi(context: Context?, setupDataRepo: SetupDataRepository, oselLogger: OselToggleableLogger) =
        instance ?: synchronized(this) {
            Timber.d("Getting smart api.")
            context?.applicationContext ?: throw IllegalStateException("Context null when creating SmartApi")
            instance ?: Flavor().getSmartApi(context.applicationContext, setupDataRepo, oselLogger).also {
                Timber.d("Getting smart api success.")
                instance = it
            }
        }

    @Suppress("FunctionName", "unused")
    fun getSmartApi_safe(context: Context?, setupDataRepo: SetupDataRepository, oselLogger: OselToggleableLogger): FlowSmartApi? {
        if (context?.applicationContext == null || alreadyFailedDoNotTryAgain)
            return null
        return try {
            if (NetworkConnectionState.instance.state == NetworkConnectionState.State.ONLINE) {
                getSmartApi(context, setupDataRepo, oselLogger)
            } else {
                Timber.d("Getting smart api safe null, because not connectivity")
                exception = ConnectException()
                null
            }
        } catch (e: Exception) {
            exception = e
            Timber.d("Getting smart api safe exception - ${e}")
            alreadyFailedDoNotTryAgain = true
            null
        }
    }

    @Suppress("unused")
    fun reset() {
        isServerBusy = false
        alreadyFailedDoNotTryAgain = false
        instance = null
        exception = null
        TimeProvider.refreshUsingOffset(0)
    }

    fun getError(context: Context):String? {
       return exception?.let { ErrorStringMapper.getErrorString(it,context) }
    }
}