package com.twentyfouri.tvlauncher.common.data.apihandler

import android.content.Context
import android.util.Log
import com.twentyfouri.smartmodel.model.error.*
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.utils.ErrorStringMapper
import com.twentyfouri.tvlauncher.common.utils.NavigatorCommon
import kotlinx.coroutines.*
import org.koin.core.component.KoinComponent
import java.io.InterruptedIOException
import java.net.SocketTimeoutException
import java.util.concurrent.ConcurrentHashMap
import javax.net.ssl.SSLException

class ApiHandler(
    internal val resourceRepository: ResourceRepository,
    context: Context,
    internal val navigatorCommon: NavigatorCommon,
): KoinComponent {

    internal val appContext: Context = context.applicationContext
    internal lateinit var retryBlock: suspend CoroutineScope.() -> Unit
    private val synchronizedJobs: ConcurrentHashMap<String, Job> = ConcurrentHashMap()

    private fun isUserLockedException(e: Exception): Boolean =
        e is GeneralApiException && e.code.isNotEmpty() && USER_LOCKED_ERROR_CODES.contains(e.code.toInt())

    /** used to launch a suspending block
     * this method prevents spamming of same block by joining all duplicate requests to the one which is still running - if there is any
     * that means that all launches of the block will get the result of the first started block
     * the duplicate calls are dropped
     * */
    fun joinPreviousOrLaunchNew(
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
        showDialog: Boolean = Flavor().showDialogs(),
        synchronizedJobName: String,
        block: suspend CoroutineScope.() -> Unit,
    ) {
        val previousOrNewJob: Job = synchronizedJobs[synchronizedJobName]
                ?: launchAndGetJob(catchBlock, showDialog, synchronizedJobName, block)
                        .also { aJob -> synchronizedJobs[synchronizedJobName] = aJob }
        CoroutineScope(Dispatchers.Default).launch {
            previousOrNewJob.join()
            synchronizedJobs.remove(synchronizedJobName)
            Log.d(TAG, "job $synchronizedJobName was removed by join")
        }
    }

    /** used to launch a suspending block
     * this method prevents spamming of same block by cancelling any previous duplicate block - if there is any
     * that means that from all launches of the same block only the last one will get finished
     * */
    fun cancelPreviousAndLaunchNew(
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
        showDialog: Boolean = Flavor().showDialogs(),
        synchronizedJobName: String,
        block: suspend CoroutineScope.() -> Unit,
    ) {
        synchronizedJobs[synchronizedJobName]?.also { previousJob ->
            previousJob.cancel()
            synchronizedJobs.remove(synchronizedJobName)
            Log.d(TAG, "job $synchronizedJobName was removed by cancel")
        }
        val newJob = launchAndGetJob(catchBlock, showDialog, synchronizedJobName, block)
        synchronizedJobs[synchronizedJobName] = newJob
        CoroutineScope(Dispatchers.Default).launch {
            newJob.join()
            synchronizedJobs.remove(synchronizedJobName)
            Log.d(TAG, "job $synchronizedJobName was removed by join")
        }
    }

    fun cancelFilteredJobsAndJoinPreviousOrLaunchNew(
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
        showDialog: Boolean = Flavor().showDialogs(),
        shouldCancelJobBlock: (jobName: String) -> Boolean, //determines whether a job with this name should be cancelled
        nameOfJobToBeJoined: String,
        block: suspend CoroutineScope.() -> Unit,
    ) {
        synchronizedJobs.forEach { (jobName, job) ->
            if (shouldCancelJobBlock(jobName)) {
                job.cancel()
                synchronizedJobs.remove(jobName)
                Log.d(TAG, "job $jobName was removed by cancel")
            }
        }
        joinPreviousOrLaunchNew(catchBlock, showDialog, nameOfJobToBeJoined, block)
    }

    /** used to launch a suspending block
     * this method DOES NOT prevents spamming of same block
     * */
    fun launchNew(
            catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
            showDialog: Boolean = Flavor().showDialogs(),
            synchronizedJobName: String? = null,
            block: suspend CoroutineScope.() -> Unit
    ) {
        launchAndGetJob(catchBlock, showDialog, synchronizedJobName, block)
    }

    private fun launchAndGetJob(
            catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
            showDialog: Boolean = Flavor().showDialogs(),
            synchronizedJobName: String? = null,
            block: suspend CoroutineScope.() -> Unit
    ): Job {
        var exceptionCaught = false
        return CoroutineScope(Dispatchers.Default).launch {
            try {
                Log.d(TAG, "start of block $synchronizedJobName")
                block()
                Log.d(TAG, "end of block $synchronizedJobName")
            } catch (e: Exception) {
                exceptionCaught = true
                retryBlock = block
                e.printStackTrace()
                when {
                    e is SuspendedException -> processException(e, showDialog) //catchBlock is auto-retrying for errors in player, but we don't want to auto-retry for suspended user errors
                    catchBlock(e) -> return@launch
                    e is InternetConnectionException -> mainActivityAction?.checkAndMaybeRestartActivity()
                    else -> processException(e, showDialog)
                }
            }
            // in case of exceptions, hide progress anyway
            if (exceptionCaught) navigatorCommon.navigateLoader(visible = false, handleSplash = true)
        }
    }

    fun launchWithLoading(
        allowProgressHide: Boolean = false,
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = { false },
        showDialog: Boolean = false,
        delay: Long = 0,
        block: suspend CoroutineScope.() -> Unit,
    ) {
        var exceptionCaught = false
        CoroutineScope(Dispatchers.Main).launch {
            navigatorCommon.navigateLoader(
                visible = true,
                handleSplash = false,
                delay = delay
            )
            val task = async(Dispatchers.IO) {
                try {
                    block()
                } catch (e: Exception) {
                    exceptionCaught = true
                    retryBlock = block
                    e.printStackTrace()
                    if (catchBlock(e)) return@async //error handled
                    processException(e, showDialog)
                }
            }
            task.await()
            if (allowProgressHide || exceptionCaught) {
                navigatorCommon.navigateLoader(
                    visible = false,
                    handleSplash = true
                )
            }
        }
    }

    private suspend fun processException(e: Exception, showDialog: Boolean) {
        withContext(Dispatchers.Default) {
            mainActivityAction?.reportApiException(e)
        }
        withContext(Dispatchers.Main){
            when {
                e is AnonymousSessionException -> {
                    if (Flavor().dialogAllowedForException(e)) {
                        catchServiceForbiddenWithDialog(R.string.service_forbidden)
                    } else {
                        catchExceptionWithToast(resourceRepository.getString(R.string.service_forbidden))
                    }
                }
                e is NotFoundException -> {
                    if (Flavor().dialogAllowedForException(e)) {
                        catchNotFoundExceptionWithDialog(e)
                    } else {
                        catchExceptionWithToast(Flavor().getErrorMessage(e, appContext))
                    }
                }
                e is AppCannotRunException ->
                    catchAppCannotRunExceptionWithDialog(R.string.app_cannot_run)
                e is AppCanRunWithLimitationsException ->
                    catchOtherApiExceptionWithDialog(R.string.out_of_home)
                e is WrongInputException || e is InvalidCredentialsException ->
                    catchLoginFailureExceptionWithDialog(R.string.authentication_failure_dialog_message)
                isUserLockedException(e) ->
                    catchLoginFailureExceptionWithDialog(R.string.authentication_failure_user_locked_dialog_message)
                e is MissingSessionException ->
                    catchMissingSessionExceptionWithDialog()
                e is InvalidSessionException ->
                    catchInvalidSessionExceptionWithDialog()
                e is SocketTimeoutException || e.cause is SocketTimeoutException || e.cause is InterruptedIOException -> {
                    catchExceptionWithToast(ErrorStringMapper.getErrorString(e,appContext)) }
                e is UnsubscribedChannelException ->
                    catchUnsubscribedExceptionWithDialog(R.string.not_entitled_temporary)
                e is PlaybackRestrictedException ->
                    catchOtherApiExceptionWithDialog(R.string.parental_restricted)
                e is SuspendedException ->
                    catchSuspendedExceptionWithDialog(e)
                e is GeneralApiException && showDialog -> {
                    if (Flavor().dialogAllowedForException(e)) {
                        catchGeneralApiExceptionWithDialog(e)
                    } else {
                        catchExceptionWithToast(Flavor().getErrorMessage(e, appContext))
                    }
                }
                e is SSLException -> catchExceptionWithToast(ErrorStringMapper.getErrorString(e,appContext))
                e is ConcurrentModificationException -> catchExceptionWithToast("EPG loading")
                else -> {
                    if(Flavor().handleProjectSpecificException(e).not()) catchExceptionWithToast(ErrorStringMapper.getErrorString(e,appContext))
                }
            }
        }
    }

    internal val mainActivityAction: MainActivityAction?
        get() = with(getKoin()) {
            try {
                getScope(getProperty(MAIN_ACTIVITY_SCOPE_ID)!!).get()
            } catch (t: Throwable) {
                Log.w(TAG, t)
                null
            }
        }

    companion object {
        private val USER_LOCKED_ERROR_CODES = setOf(157021008, 157021011)
        internal const val TAG = "ApiHandler"
        const val MAIN_ACTIVITY_SCOPE_ID = "MAIN_ACTIVITY_SCOPE_ID"
    }
}