package com.twentyfouri.tvlauncher.common.ui.messagedialog

import android.app.Activity
import android.content.Context
import android.content.DialogInterface
import android.os.Bundle
import android.view.KeyEvent
import android.view.View
import android.widget.Button
import androidx.lifecycle.ViewModelProvider
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.databinding.DialogMessageBinding
import com.twentyfouri.tvlauncher.common.extensions.enlarge
import com.twentyfouri.tvlauncher.common.ui.BaseDialogFragment
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import timber.log.Timber

class MessageDialogFragment private constructor() : BaseDialogFragment<DialogMessageBinding>(), MessageDialogView {
    override val layoutRes = R.layout.dialog_message
    var mListener: MessageDialogFragmentListener? = null
    var mDismissListener: MessageDialogDismissListener? = null
    private lateinit var viewModel: MessageDialogViewModel
    lateinit var tagExt: String
    private var mDismissPending = false
    private var mAttached = false

    override fun onBindViewModel(binding: DialogMessageBinding) {
        viewModel = ViewModelProvider(this)[MessageDialogViewModel::class.java]
        viewModel.view = this
        viewModel.model = arguments?.getParcelable(PARAM_MODEL)
        binding.viewModel = viewModel
    }

    companion object {
        private const val PARAM_MODEL = "model"

        fun newInstance(model: MessageDialogModel) = MessageDialogFragment().apply {
            tagExt = model.code
            arguments = Bundle().apply {
                putParcelable(PARAM_MODEL, model)
            }
        }
    }

    // region Binded

    override fun onResult(answer: MessageDialogAction) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Dialog ${viewModel.code.value} answer: ${answer} ")
        mListener?.onResult(answer)
        this.dismiss()
    }

    // endregion

    override fun onResume() {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Dialog ${viewModel.code.value} resumed: ${viewModel.message.value}, ${getButtonsInfo()} ")
        super.onResume()
    }

    override fun onPause() {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Dialog ${viewModel.code.value} paused: ${viewModel.message.value}, ${getButtonsInfo()} ")
        super.onPause()
    }

    private fun getButtonsInfo(): String {
        with(viewModel) {
            val a = if(optionAVisiblity.value == View.VISIBLE) optionAButtonText.value ?: "" else ""
            val b = if(optionBVisiblity.value == View.VISIBLE) optionBButtonText.value ?: "" else ""
            val c = if(optionCVisiblity.value == View.VISIBLE) optionCButtonText.value ?: "" else ""
            val canc = if(cancelVisiblity.value == View.VISIBLE) cancelButtonText.value ?: "" else ""
            return "[$canc] [$c] [$b] [$a]" //same order as in UI
        }
    }

    override fun onStart() {
        super.onStart()

        val width = resources.getDimensionPixelSize(R.dimen.dialog_width)
        val height = resources.getDimensionPixelSize(R.dimen.dialog_height)

        dialog?.window?.setLayout(width, height)

        when {
            viewModel.optionAVisiblity.value == View.VISIBLE -> preSetFocus(mBinding.optionAButton)
            viewModel.optionBVisiblity.value == View.VISIBLE -> preSetFocus(mBinding.optionBButton)
            viewModel.optionCVisiblity.value == View.VISIBLE -> preSetFocus(mBinding.optionCButton)
            viewModel.cancelVisiblity.value == View.VISIBLE -> preSetFocus(mBinding.cancelButton)
        }

        dialog?.setOnKeyListener { _, _, event ->
            if (event.isNavigationEvent) return@setOnKeyListener false
            if (mListener?.onResult(MessageDialogAction.Event(event)) == true) {
                true //key event consumed by dialog
            } else {
                //key event not consumed by dialog - dispatch to RemoteKeyManager if needed
                if (event.action == KeyEvent.ACTION_UP && event.keyCode == KeyEvent.KEYCODE_GUIDE) {
                    dismiss()
                }
                if (event.action == KeyEvent.ACTION_UP && event.keyCode == KeyEvent.KEYCODE_GUIDE) {
                    (context as? Activity)?.dispatchKeyEvent(event)
                }
                false
            }
        }
    }

    private fun preSetFocus(view: View) {
        view.requestFocus()
        view.enlarge(true)
    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        mAttached = true
        //WARNING this can override the listener if it was set from outside previously
        if (context is MessageDialogFragmentListener) mListener = context
        if(mDismissPending) {
            mDismissPending = false
            dismiss()
        }
    }

    override fun onDetach() {
        super.onDetach()
        mAttached = false
    }

    override fun onDismiss(dialog: DialogInterface) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Dialog ${viewModel.code.value} dismissed: ${viewModel.message.value}, ${getButtonsInfo()} ")
        super.onDismiss(dialog)
        this.mDismissListener?.onDismiss()
    }

    override fun dismiss() {
        if (!mAttached) {
            // dismiss() before getFragmentManager() is set cause NPE in dismissInternal().
            // FragmentManager is set when a fragment is used in a transaction,
            // so wait here until we can dismiss safely.
            mDismissPending = true
        } else {
            super.dismiss()
        }
    }

    override fun onFocusChange(v: View?, hasFocus: Boolean) {
        if (v is Button) {
            v.enlarge(hasFocus)
        }
    }

    fun setDialogText(message: String) {
        mBinding.messageDialogTitle.text = message
    }
    
    private val KeyEvent.isNavigationEvent
        get() = when (this.keyCode) {
            KeyEvent.KEYCODE_BUTTON_A,
            KeyEvent.KEYCODE_DPAD_LEFT,
            KeyEvent.KEYCODE_DPAD_RIGHT,
            KeyEvent.KEYCODE_DPAD_CENTER,
            KeyEvent.KEYCODE_DPAD_UP,
            KeyEvent.KEYCODE_DPAD_DOWN -> true
            else -> false
        }
}