package com.twentyfouri.tvlauncher.common.utils

import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.wifi.WifiManager
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.getSystemService
import androidx.lifecycle.*
import androidx.lifecycle.Observer
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.joda.time.DateTime
import timber.log.Timber
import java.lang.RuntimeException
import java.net.InetAddress
import java.net.InetSocketAddress
import java.net.Socket
import java.util.*
import kotlin.concurrent.schedule

class NetworkConnectionState private constructor() {

    enum class State {
        ONLINE,
        NO_INTERNET,
        OFFLINE
    }

    enum class NetworkQualityCheckReason {
        NEW_USER,
        NETWORK_CHANGE
    }

    private object HOLDER {
        val INSTANCE = NetworkConnectionState()
    }

    companion object {
        const val PROBE_HOST ="connectivitycheck.gstatic.com"
        const val PROBE_TIMEOUT_ALL = 2000 //maximum time of all attempts within one Probe
        const val PROBE_TIMEOUT_SINGLE = 350 //maximum time of one attempt
        private const val DEFAULT_REPEAT_DELAY = 60000
        private const val CONNECTIVITY_INITIAL_WAIT: Long = 20000
        private const val SAFE_TIMEOUT_AFTER_WAKE_UP: Long = 2000

        val instance: NetworkConnectionState by lazy { HOLDER.INSTANCE }
    }

    private var networkCallback: ConnectivityManager.NetworkCallback? = null

    private var currentlyProbing = false
    private var interestedInProbeResult = false
    private var lastProbeState = State.OFFLINE

    private val repeatDelaysOnline = arrayOf(1000, 1000, 1000, 5000, 15000, 35000, DEFAULT_REPEAT_DELAY)
    private val repeatDelaysNoInternet = arrayOf(1000, 1000, 1000, 5000, 10000)
    private var repeatDelayIndex = -1
    private var repeatTimerTask: TimerTask? = null

    var state = lastProbeState
        private set(newState) {
            field = newState
            Timber.tag("PROBE").d("Network state $state")
        }

    private val stateLive: MutableLiveData<State> = MutableLiveData(state)
    val stateLD: LiveData<State> = stateLive

    fun register(context: Context, fallback: () -> Unit = {}, networkApprove: () -> Unit) {
        networkCallback = LauncherNetworkCallback(context, networkApprove)
        try {
            context.getSystemService<ConnectivityManager>()?.registerDefaultNetworkCallback(
                networkCallback!!
            )
        } catch (e: RuntimeException) {
            // something went wrong during registration, restart application
            fallback()
        }
    }

    fun unregister(context: Context) {
        interestedInProbeResult = false
        stopProbeRepeating()
        context.getSystemService<ConnectivityManager>()?.unregisterNetworkCallback(networkCallback!!)
        networkCallback = null
    }

    private fun probe() {
        if(currentlyProbing) return
        currentlyProbing = true
        Probe(PROBE_HOST, PROBE_TIMEOUT_ALL, PROBE_TIMEOUT_SINGLE) { success, time ->
            currentlyProbing = false
            if(interestedInProbeResult) {

                if((success && lastProbeState == State.ONLINE) || (!success && lastProbeState == State.NO_INTERNET)) {
                    //same result as before
                    lastProbeState = if(success) State.ONLINE else State.NO_INTERNET
                    if(state != lastProbeState) {
                        state = lastProbeState
                        stateLive.postValue(lastProbeState)
                    }

                    //send next probe
                    if(++repeatDelayIndex >= getRepeatDelayArray(lastProbeState).size) repeatDelayIndex = getRepeatDelayArray(lastProbeState).size - 1
                    repeatTimerTask = Timer().schedule(getDelayByIndex(repeatDelayIndex, lastProbeState).toLong()) {
                        if(interestedInProbeResult) probe()
                    }
                } else {
                    //different result as before
                    lastProbeState = if(success) State.ONLINE else State.NO_INTERNET
                    stopProbeRepeating()
                    probe()
                }
            } else {
                stopProbeRepeating()
            }
        }
    }

    private fun getDelayByIndex(index: Int, state: State): Int {
        return if(index < 0 || index >= getRepeatDelayArray(state).size) DEFAULT_REPEAT_DELAY
        else getRepeatDelayArray(state)[index]
    }

    private fun stopProbeRepeating() {
        repeatTimerTask?.cancel()
        repeatDelayIndex = -1
    }

    private fun getRepeatDelayArray(state: State): Array<Int> {
        return when (state) {
            State.OFFLINE -> emptyArray()
            State.NO_INTERNET -> repeatDelaysNoInternet
            State.ONLINE -> repeatDelaysOnline
        }
    }

    private inner class LauncherNetworkCallback(private val context: Context, private val networkApprove: () -> Unit): ConnectivityManager.NetworkCallback() {
        override fun onLost(network: Network) {
            super.onLost(network)
            stopProbeRepeating()
            interestedInProbeResult = false
            lastProbeState = State.OFFLINE
            state = State.OFFLINE
            stateLive.postValue(state)
        }

        override fun onAvailable(network: Network) {
            super.onAvailable(network)
            if (Flavor().useSafeListOfApprovedNetworks) {
                val wifiInfo = context.getSystemService<WifiManager>()?.connectionInfo
                val pendingSaveInitialNetworkOfCurrentUser =
                    SharedPreferencesUtils.isFirstTimeDeviceUsage().not()
                            && SharedPreferencesUtils.getShouldCheckNetworkQuality().not()
                            && SharedPreferencesUtils.getApprovedNetworks() == null
                            && SharedPreferencesUtils.getIsInitialUsersNetworkSaved().not()

                if (NetworkInfo.getActiveConnectionInfo(context).type == InterfaceType.LAN) {
                    if (pendingSaveInitialNetworkOfCurrentUser) {
                        //current user is connected via ethernet during first feature launch.
                        // Do not add any WiFi network to the approved networks list
                        SharedPreferencesUtils.putApprovedNetworks(setOf("LAN"))
                        SharedPreferencesUtils.putIsInitialUsersNetworkSaved(true)
                    }
                } else {
                    if (pendingSaveInitialNetworkOfCurrentUser) {
                        //current user is connected via WiFi during first feature launch.
                        //Add current WiFi network to the approved networks list even though it may not
                        //meet approval criteria.
                        SharedPreferencesUtils.putApprovedNetworks(setOf(wifiInfo?.networkId.toString()))
                        SharedPreferencesUtils.putIsInitialUsersNetworkSaved(true)
                    }

                    SharedPreferencesUtils.putNetworkQualityCheckReason(NetworkQualityCheckReason.NETWORK_CHANGE.ordinal)
                    SharedPreferencesUtils.putShouldCheckNetworkQuality(true)
                    try {
                        if (lifecycleStateAtLeastResumed()) {
                            CoroutineScope(Dispatchers.Main).launch { networkApprove.invoke() }
                        }
                    } catch (e: Exception) {
                        Log.w("LauncherNetworkCallback", "Network was not approved")
                    }
                }
            }
            interestedInProbeResult = true
            probe()
        }

        override fun onCapabilitiesChanged(network: Network, networkCapabilities: NetworkCapabilities) {
            super.onCapabilitiesChanged(network, networkCapabilities)
            CoroutineScope(Dispatchers.Main).launch {
                YouboraAnalytics.getInstance(context).setNetworkType(context, networkCapabilities)
            }
        }

        private fun lifecycleStateAtLeastResumed() =
            (context as? AppCompatActivity)?.lifecycle?.currentState?.isAtLeast(Lifecycle.State.RESUMED) == true
    }

    fun waitForConnection(lifecycleOwner: LifecycleOwner, timeout: Long = CONNECTIVITY_INITIAL_WAIT, skipWaiting: Boolean = false, callback: () -> Unit) {
        //this is just convenient method so we don't need to create extra method if we are currently not interested in waiting for connection
        /**
         * currentState lifecycle:
         * INITIALIZED ->
         * CREATED (after onCreate) ->
         * STARTED (after onStart) ->
         * RESUMED (after onResume) ->
         * STARTED (right before onPause) ->
         * CREATED (right before onStop) ->
         * DESTROYED (right before onDestroy)
         *
         * with condition "lifecycleOwner.lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).not" we want to cancel waiting for connection for
         * all states that are NOT between CREATED (after onCreate) and CREATED (right before onStop)
         */
        if(skipWaiting) {
            if (lifecycleOwner.lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).not()) return
            callback()
            return
        }

        Timer().schedule(SAFE_TIMEOUT_AFTER_WAKE_UP) {
            CoroutineScope(Dispatchers.Main).launch {
                if (lifecycleOwner.lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).not()) return@launch
                if (state == State.ONLINE) {
                    callback()
                } else {
                    var waitingFinished = false
                    var connectionWaitTask: TimerTask? = null

                    val connectionObserver = object : Observer<State> {
                        override fun onChanged(state: State?) {
                            if (waitingFinished) {
                                stateLD.removeObserver(this)
                                return
                            }

                            when (state) {
                                State.ONLINE -> {
                                    waitingFinished = true
                                    connectionWaitTask?.cancel()
                                    stateLD.removeObserver(this)
                                    if (lifecycleOwner.lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).not()) return
                                    callback()
                                }
                                State.NO_INTERNET,
                                State.OFFLINE,
                                null -> {
                                    //ignore it, we are waiting for ONLINE
                                }
                            }
                        }
                    }

                    connectionWaitTask = Timer().schedule(timeout) {
                        if (waitingFinished) return@schedule
                        waitingFinished = true
                        CoroutineScope(Dispatchers.Main).launch {
                            stateLD.removeObserver(connectionObserver)
                            if (lifecycleOwner.lifecycle.currentState.isAtLeast(Lifecycle.State.CREATED).not()) return@launch
                            callback()
                        }
                    }

                    stateLD.observe(lifecycleOwner, connectionObserver)
                }
            }
        }
    }
}

class Probe(
        private val host: String,
        private val timeoutAll: Int,
        private val timeoutSingle: Int,
        private val resultInterface: (result: Boolean, time: Long) -> Unit
) {
    private var resultSent = false
    private val start = DateTime.now().millis

    init {
        Timber.tag("PROBE").d("Probing $host with ${timeoutAll}ms timeout")
        Timer("probeTimeout", false).schedule(timeoutAll.toLong()) {
            sendResult(false)
        }

        CoroutineScope(Dispatchers.IO).launch {
            val inetAddress = try {
                InetAddress.getByName(host)
            } catch (e: Exception) {
                if (!resultSent) Timber.tag("PROBE")
                    .d("Unable to get IP from DNS -> Probing $host failed with ${e.message} -> going to check connectivity via Socket")
                null
            }
            inetAddress?.also {
                val isReachable = try {
                    it.isReachable(timeoutSingle)
                } catch (e: Exception) {
                    if (!resultSent) Timber.tag("PROBE")
                        .d("Probing $host failed with ${e.message} -> going to check connectivity via Socket")
                    false
                }
                if (isReachable) {
                    sendResult(true)
                    return@launch
                }
                if (!resultSent) Timber.tag("PROBE")
                    .d("Probing $host failed -> going to check connectivity via Socket")
            }

            listOf(22, 80, 443, 25).forEach { port ->
                if (isReachableViaSocket(host, port, timeoutSingle)) {
                    sendResult(true)
                    return@launch
                } else {
                    if (!resultSent) Timber.tag("PROBE")
                        .d("Probing $host on port: $port failed -> going to check connectivity via Socket")
                }
            }
            sendResult(false)
        }
    }

    /** Any Open port on other machine
     * openPort =  22 - ssh, 80 or 443 - webserver, 25 - mailserver etc.
     * */
    private fun isReachableViaSocket(address: String, openPort: Int, timeOutMillis: Int): Boolean = try {
        Socket().use { soc -> soc.connect(InetSocketAddress(address, openPort), timeOutMillis) }
        true
    } catch (ex: Exception) {
        false
    }

    private fun sendResult(result: Boolean) {
        if (!resultSent) {
            resultSent = true
            val textResult = if (result) "REACHABLE" else "NOT REACHABLE"
            val elapsedTime = DateTime.now().millis - start
            Timber.tag("PROBE")
                .d("Probing finished, $host is $textResult within ${timeoutAll}ms. Time elapsed $elapsedTime")
            resultInterface(result, elapsedTime)
        }
    }
}
