package com.twentyfouri.tvlauncher.common.utils

import android.annotation.SuppressLint
import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.pm.PackageInfo
import android.net.ConnectivityManager
import android.net.wifi.WifiConfiguration
import java.lang.reflect.Field
import java.lang.reflect.Method

/*

!! Methods here can easily crash, handle exceptions on you own !!

Currently goal of following methods is to determine if IP of interface is assigned by DHCP or is STATIC
But principles can be used also in other cases

 */

object ReflectionUtils {

    //double reflection is needed because system will return ClassNotFoundException if you try directly
    //first prepare class (even if class itself is not hidden) and then its method and/or field
    //static methods/fields can be invoked directly on prepared class, otherwise invoke them on suitable object
    //for private fields use getDeclaredField and set isAccessible = true on it before invoke
    //be careful when preparing method to set proper amount of params and their type
    //if param type is not primitive and is also hidden you can prepare class and use it there

    //example of access: "public void setConfiguration(String iface, IpConfiguration config)" on EthernetManager class
    // config param is type IpConfiguration which is hidden
    // prepare needed class: val ipConfigurationClass = getClass.invoke(null, "android.net.IpConfiguration") as Class<*>
    // and then prepare method with correct second param type:
    // val setIfMethod = getMethod.invoke(ethernetManagerClass, "setConfiguration", arrayOf<Class<*>>(String::class.java, ipConfigurationClass)) as Method

    private val getClass = Class::class.java.getMethod("forName", String::class.java)
    private val getMethod = Class::class.java.getMethod("getMethod", String::class.java, arrayOf<Class<*>>()::class.java)
    private val getField = Class::class.java.getMethod("getField", String::class.java)
    private val getDeclaredField = Class::class.java.getMethod("getDeclaredField", String::class.java)

    //returned value is type IpConfiguration (which is hidden)
    fun getIpConfiguration(wifiConfiguration: WifiConfiguration): Any? {
        val wifiConfigurationClass = getClass.invoke(null, "android.net.wifi.WifiConfiguration") as Class<*>
        val getIpConfigurationMethod = getMethod.invoke(wifiConfigurationClass, "getIpConfiguration", arrayOf<Class<*>>()) as Method
        return getIpConfigurationMethod.invoke(wifiConfiguration)
    }

    //param is type IpConfiguration (which is hidden)
    //returned value is enum type IpAssignment (which is hidden)
    fun getIpAssignment(ipConfiguration: Any?): Any? {
        ipConfiguration ?: return null
        val ipConfigurationClass = getClass.invoke(null, "android.net.IpConfiguration") as Class<*>
        try {
            val ipAssignmentProperty = getField.invoke(ipConfigurationClass, "ipAssignment" ) as Field
            return ipAssignmentProperty.get(ipConfiguration)
        } catch (e: Exception) {
            try {
                val ipAssignmentMethod = getMethod.invoke(ipConfigurationClass, "getIpAssignment", arrayOf<Class<*>>()) as Method
                return ipAssignmentMethod.invoke(ipConfiguration)
            } catch (e: Exception) {
                return null
            }
        }
    }

    //return all network interfaces from
    //tried to acquire all network interfaces and then read ipConfiguration of ethernet interface
    //DEAD END because of "android.permission.CONNECTIVITY_INTERNAL" which is not granted, even for launcher is not granted
    fun getNetworkInterfaces(context: Context): Any? {
        val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        val onNetworkActiveListener = ConnectivityManager.OnNetworkActiveListener {}
        connectivityManager.addDefaultNetworkActiveListener(onNetworkActiveListener) //needed to attach mNMService in ConnectivityManager
        connectivityManager.removeDefaultNetworkActiveListener(onNetworkActiveListener)

        //INetworkManagementService is private in ConnectivityManager
        val connectivityManagerClass = getClass.invoke(null, "android.net.ConnectivityManager") as Class<*>
        val inmsField = getDeclaredField.invoke(connectivityManagerClass, "mNMService") as Field
        inmsField.isAccessible = true
        val inms = inmsField.get(connectivityManager)

        val iNetworkManagementServiceClass = getClass.invoke(null, "android.os.INetworkManagementService") as Class<*>
        val listInterfacesMethod = getMethod.invoke(iNetworkManagementServiceClass, "listInterfaces", arrayOf<Class<*>>()) as Method
        //this call fail because of "android.permission.CONNECTIVITY_INTERNAL"
        val items = listInterfacesMethod.invoke(inms)
        return items
    }

    //DEAD END EthernetManager return no interfaces
    //DEAD END "eth0" configuration is fake one (UNASSIGNED)
    fun getEthernetIpAssignment(context: Context): Any? {
        val ethernetManagerClass = getClass.invoke(null, "android.net.EthernetManager") as Class<*>
        val getConfigurationMethod = getMethod.invoke(ethernetManagerClass, "getConfiguration", arrayOf<Class<*>>(String::class.java)) as Method
        val getAvailableInterfacesMethod = getMethod.invoke(ethernetManagerClass, "getAvailableInterfaces", arrayOf<Class<*>>()) as Method

        //get EthernetManager which is hidden
        //hidden public static final String ETHERNET_SERVICE = "ethernet";
        @SuppressLint("WrongConstant")
        val ethernetManager = context.applicationContext.getSystemService("ethernet")

        //for some reason it returns empty array, no interfaces
        val interfaces = getAvailableInterfacesMethod.invoke(ethernetManager)

        //blind shot for "eth0" but it return UNASSIGNED
        val ipConfiguration = getConfigurationMethod.invoke(ethernetManager, "eth0")
        val ipAssignment = getIpAssignment(ipConfiguration)
        return ipAssignment
    }

    //other dead ends

    // "android.server." classes are not available in android.jar
    // val EthernetConfigStoreClass = getClass.invoke(null, "android.server.ethernet.EthernetConfigStore") as Class<*>
    // val readMethod = getMethod.invoke(ethernetConfigStoreClass, "read", arrayOf<Class<*>>()) as Method
    // readMethod.invoke(ethernetConfigStoreClass)

    // settings don't seem to include required information
    // Settings.Global Settings.Secure Settings.System

    //getting info about running application from outside of it (from library) without using context
    fun getPackageInfo(): PackageInfo? {
        return try {
            val activityThreadClass = getClass.invoke(null, "android.app.ActivityThread") as Class<*>
            val packageNameMethod = getMethod.invoke(activityThreadClass, "currentPackageName", arrayOf<Class<*>>()) as Method
            val packageName = packageNameMethod.invoke(activityThreadClass) as String
            val currentApplicationMethod = getMethod.invoke(activityThreadClass, "currentApplication", arrayOf<Class<*>>()) as Method
            val app = currentApplicationMethod.invoke(activityThreadClass) as Application
            app.packageManager.getPackageInfo(packageName, 0)
        } catch (e: Exception) {
            //not crucial, don't care if fail
            null
        }
    }

    fun getPackageInfo(packageName: String): PackageInfo? {
        return try {
            val activityThreadClass = getClass.invoke(null, "android.app.ActivityThread") as Class<*>
            val currentApplicationMethod = getMethod.invoke(activityThreadClass, "currentApplication", arrayOf<Class<*>>()) as Method
            val app = currentApplicationMethod.invoke(activityThreadClass) as Application
            app.packageManager.getPackageInfo(packageName, 0)
        } catch (e: Exception) {
            //not crucial, don't care if fail
            null
        }
    }

    fun getApplicationContext(): Context? {
        return try {
            val activityThreadClass = getClass.invoke(null, "android.app.ActivityThread") as Class<*>
            val currentApplicationMethod = getMethod.invoke(activityThreadClass, "currentApplication", arrayOf<Class<*>>()) as Method
            currentApplicationMethod.invoke(activityThreadClass) as Context
        } catch (e: Exception) {
            null
        }
    }
}