package com.twentyfouri.tvlauncher.common.utils.logging

import android.annotation.SuppressLint
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.SharedPreferencesUtils
import com.twentyfouri.tvlauncher.common.utils.TextFileTree
import com.twentyfouri.tvlauncher.common.utils.TextFileTree.Companion.TAG_FILE_LOG
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLoggerPermissions.PermissionRequestResult
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import okhttp3.logging.HttpLoggingInterceptor
import timber.log.Timber
import java.io.*
import java.util.*
import kotlin.concurrent.schedule

@Suppress("SpellCheckingInspection")
class OselToggleableLogger(
    val permissions: OselToggleableLoggerPermissions
) : HttpLoggingInterceptor.Logger {
    var detailedLoggingStatus: String = STATUS_NOT_LOGGING
        private set

    var loggingStatus: Status = Status.DISABLED
        private set

    val loggingEnabled: Boolean
        get() = loggingStatus == Status.ENABLED

    private var waitingForPermission = false
    private var oselTextFileTree: TextFileTree? = null
    private var limitTimer: TimerTask? = null

    init {
        permissions.onPermissionRequestComplete = ::onPermissionRequestCompleted
    }

    fun toggleLogging(enable: Boolean, isAutomaticCall: Boolean = false) {
        if (enable) {
            if (loggingEnabled && oselTextFileTree != null) {
                // Stop current logging and start new one (time/storage limit may change)
                disableLogging(true)
            }

            if (permissions.hasPermissions) {
                GlobalScope.launch { enableLogging() }
            } else {
                when {
                    isAutomaticCall -> {
                        loggingStatus = Status.ENABLING_FAILED
                        detailedLoggingStatus = STATUS_PERMISSION_MISSING_ON_RESTART
                    }
                    permissions.requestPermissionFromUser() -> {
                        waitingForPermission = true
                        loggingStatus = Status.ENABLING_IN_PROGRESS
                        detailedLoggingStatus = STATUS_REQUESTING_PERMISSION
                    }
                    else -> {
                        loggingStatus = Status.ENABLING_FAILED
                        detailedLoggingStatus = STATUS_CANNOT_REQUEST_PERMISSION
                    }
                }
            }
        } else {
            if (loggingEnabled.not() && oselTextFileTree == null) return
            disableLogging(false)
        }
    }

    override fun log(message: String) {
        Timber.tag(TAG_HTTP_LOG).d(message)
    }

    private fun onPermissionRequestCompleted(result: PermissionRequestResult) {
        if (waitingForPermission) {
            waitingForPermission = false
            when (result) {
                PermissionRequestResult.GRANTED -> {
                    toggleLogging(true)
                }
                PermissionRequestResult.DENIED -> {
                    loggingStatus = Status.ENABLING_FAILED
                    detailedLoggingStatus = STATUS_PERMISSION_DENIED
                }
                PermissionRequestResult.ACTIVITY_CLOSED -> {
                    loggingStatus = Status.ENABLING_FAILED
                    detailedLoggingStatus = STATUS_PERMISSION_NOT_GRANTED
                }
            }
        }
    }

    private suspend fun enableLogging() {
        if (TextFileTree.isStorageWritable(PATH_NAME, LOG_FILE_NAME) && isTimeLimitOK()) {
            val storedStorageLimit = SharedPreferencesUtils.getRemoteLoggingStorageLimit()
            TextFileTree(
                pathName = PATH_NAME,
                baseFileName = LOG_FILE_NAME,
                tagSelection = TAG_SELECTION,
                maxFileSize = FILE_SIZE_LIMIT_MB,
                storageLimit = if (storedStorageLimit <= 0) DEFAULT_STORAGE_MAX_THRESHOLD_MB else storedStorageLimit,
                requestToStop = ::disableLogging
            ).also {
                oselTextFileTree = it
                Timber.plant(it)
            }

            val storedStopDateLimit = SharedPreferencesUtils.getRemoteLoggingStopDate()
            if (storedStopDateLimit != 0L) {
                val timeLimit = storedStopDateLimit - TimeProvider.nowMs()
                limitTimer = Timer("oselLogTimeLimit", true).schedule(timeLimit) {
                    cancel()
                    limitTimer = null
                    Timber.tag(TAG_RUN_LOG).i("OSEL ENDED because required time for run is up")
                    disableLogging(false)
                }
            }

            Timber.tag(TAG_RUN_LOG).i("OSEL STARTED")
            loggingStatus = Status.ENABLED
            detailedLoggingStatus = STATUS_LOGGING
            waitingForPermission = false // Make sure we don't get into an inconsistent state
        } else {
            loggingStatus = Status.ENABLING_FAILED
            detailedLoggingStatus = STATUS_FILE_ERROR
        }
    }

    private fun isTimeLimitOK(): Boolean {
        val storedStopDateLimit = SharedPreferencesUtils.getRemoteLoggingStopDate()
        if (storedStopDateLimit == 0L) return true // No limit set
        return storedStopDateLimit - TimeProvider.nowMs() > 0 // Return false is we are behind stop timestamp
    }

    private fun disableLogging(becauseOfRestart: Boolean = false) {
        Timber.tag(TAG_RUN_LOG).i("OSEL STOPPED")
        oselTextFileTree?.also {
            if (loggingEnabled) it.disableLogging()
            Timber.uproot(it)
            oselTextFileTree = null
        }

        loggingStatus = Status.DISABLED
        detailedLoggingStatus = STATUS_NOT_LOGGING

        limitTimer?.cancel()
        limitTimer = null

        if (!becauseOfRestart) {
            // Reset shared prefs
            SharedPreferencesUtils.putRemoteLoggingEnabled(false)
            SharedPreferencesUtils.putRemoteLoggingStopDate(0)
            SharedPreferencesUtils.putRemoteLoggingStorageLimit(0)
        }
    }

    enum class Status {
        ENABLED, DISABLED, ENABLING_IN_PROGRESS, ENABLING_FAILED
    }

    companion object {
        @SuppressLint("SdCardPath")
        private const val PATH_NAME = "/sdcard/Download/Entel"
        private const val LOG_FILE_NAME = "logFile"

        const val TAG_HTTP_LOG = "OSEL_HTTP"
        const val TAG_RUN_LOG = "OSEL_RUN"
        const val TAG_PLAYER_LOG = "OSEL_PLAYER_EVENT"
        const val TAG_PLAYER_STREAM_LOG = "OSEL_PLAYER_STREAM"
        const val TAG_PLAYER_ERROR_LOG = "OSEL_PLAYER_ERROR"
        const val TAG_KEY_EVENT_LOG = "OSEL_KEY"
        const val TAG_UI_LOG = "OSEL_UI"
        private const val DEFAULT_STORAGE_MAX_THRESHOLD_MB = 300
        private const val FILE_SIZE_LIMIT_MB = 50

        private const val STATUS_NOT_LOGGING = "Logging disabled"
        private const val STATUS_LOGGING = "Logging enabled"
        private const val STATUS_REQUESTING_PERMISSION = "Enabling logging, requesting storage permission from the user"
        private const val STATUS_CANNOT_REQUEST_PERMISSION = "Error enabling logging: need to request storage permission from the user, but the launcher needs to be open in the foreground in order to do this"
        private const val STATUS_PERMISSION_MISSING_ON_RESTART = "Error enabling logging: tried to automatically (re-)enable on restart, but missing storage permissions"
        private const val STATUS_PERMISSION_DENIED = "Error enabling logging: storage permission denied by user"
        private const val STATUS_PERMISSION_NOT_GRANTED = "Error enabling logging: storage permission not granted by user (request closed without granting or denying)"
        private const val STATUS_FILE_ERROR = "Error enabling logging: failed to create or open log file"

        private val TAG_SELECTION = listOf(
            TAG_HTTP_LOG,
            TAG_PLAYER_LOG,
            TAG_PLAYER_STREAM_LOG,
            TAG_PLAYER_ERROR_LOG,
            TAG_RUN_LOG,
            TAG_KEY_EVENT_LOG,
            TAG_FILE_LOG,
            TAG_UI_LOG
        )
    }
}