package com.twentyfouri.tvlauncher.utils

import com.twentyfouri.tvlauncher.common.utils.ZipFileTree
import io.kotest.assertions.timing.EventuallyConfig
import io.kotest.assertions.timing.eventually
import io.kotest.assertions.until.fixed
import io.kotest.core.spec.style.StringSpec
import io.kotest.engine.spec.tempfile
import io.kotest.matchers.booleans.shouldBeTrue
import io.kotest.matchers.file.shouldBeEmpty
import io.kotest.matchers.file.shouldNotBeEmpty
import io.kotest.matchers.ints.shouldBeExactly
import io.kotest.matchers.ints.shouldBeZero
import io.kotest.matchers.longs.shouldBeLessThanOrEqual
import io.kotest.matchers.nulls.shouldBeNull
import io.kotest.matchers.nulls.shouldNotBeNull
import io.kotest.matchers.shouldBe
import io.kotest.matchers.string.shouldContain
import io.kotest.matchers.string.shouldEndWith
import io.kotest.matchers.string.shouldHaveLineCount
import io.kotest.matchers.types.shouldBeInstanceOf
import io.kotest.property.Arb
import io.kotest.property.arbitrary.string
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import timber.log.Timber
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.util.zip.ZipFile
import java.util.zip.ZipInputStream
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.ExperimentalTime

@ExperimentalTime
class ZipFileTreeTest : StringSpec({
    val zipFile = tempfile()

    beforeEach {
        Timber.uprootAll()
        Timber.treeCount.shouldBeZero()
        zipFile.shouldBeEmpty()
        Timber.plant(ZipFileTree(zipFile, LIMIT))
        Timber.treeCount shouldBeExactly 1
    }

    afterEach {
        FileOutputStream(zipFile).use {
            it.write(ByteArray(0))
        }
        zipFile.shouldBeEmpty()
    }

    "Simple log" {
        Timber.d(LINE)

        eventually(EVENTUALLY_CONFIG) {
            zipFile.shouldNotBeEmpty()
            ZipInputStream(FileInputStream(zipFile)).use { zip ->
                zip.nextEntry.shouldNotBeNull().name shouldBe ZipFileTree.FILE_NAME
                zip.readBytes().decodeToString() shouldEndWith LINE
                zip.nextEntry.shouldBeNull()
            }
        }
    }

    "Multiple lines" {
        val lines = 10

        repeat(lines) {
            Timber.d("$LINE $it")
        }
        eventually(EVENTUALLY_CONFIG) {
            zipFile.shouldNotBeEmpty()
            ZipInputStream(FileInputStream(zipFile)).use { zip ->
                zip.nextEntry.shouldNotBeNull().name shouldBe ZipFileTree.FILE_NAME
                zip.readBytes().decodeToString().shouldHaveLineCount(lines)
                zip.nextEntry.shouldBeNull()
            }
        }
    }

    "Multiline" {
        val log1 = """
            A
            B
        """.trimIndent()
        val log2 = """
            C
            D
        """.trimIndent()

        Timber.d(log1)
        Timber.d(log2)

        eventually(EVENTUALLY_CONFIG) {
            with (ZipFile(zipFile)) {
                getInputStream(getEntry(ZipFileTree.FILE_NAME))
                    .use { it.readBytes() }
                    .decodeToString().shouldHaveLineCount(4)
                    .lines()
                    .map { it.split(' ').last() }
                    .shouldBe(log2.lines() + log1.lines())
            }
        }
    }

    "Multiple coroutines" {
        val lines = 10
        val coroutines = List(5) { c ->
            launch {
                repeat(lines) { l ->
                    Timber.d("$c: $LINE $l")
                }
            }
        }.apply {
            joinAll()
        }

        eventually(EVENTUALLY_CONFIG) {
            zipFile.shouldNotBeEmpty()
            ZipInputStream(FileInputStream(zipFile)).use { zip ->
                zip.nextEntry.shouldNotBeNull().name shouldBe ZipFileTree.FILE_NAME
                zip.readBytes().decodeToString().shouldHaveLineCount(lines * coroutines.size)
                zip.nextEntry.shouldBeNull()
            }
        }
    }

    "Size limit" {
        // 100 samples gives about 4 times the LIMIT
        Arb.string().samples().take(100).forEach {
            Timber.d(it.value)
        }

        eventually(EVENTUALLY_CONFIG) {
            Timber.forest().first().shouldBeInstanceOf<ZipFileTree>().actorIsEmpty.shouldBeTrue()
            zipFile.shouldNotBeEmpty()
        }

        ZipFile(zipFile).getEntry(ZipFileTree.FILE_NAME).compressedSize shouldBeLessThanOrEqual LIMIT.toLong()
    }

    "Exception" {
        Timber.w(Throwable(LINE))

        eventually(EVENTUALLY_CONFIG) {
            with (ZipFile(zipFile)) {
                getInputStream(getEntry(ZipFileTree.FILE_NAME)).use { it.readBytes() }.decodeToString() shouldContain LINE
            }
        }
    }
}) {
    private companion object {
        private const val LIMIT = 1000
        private const val LINE = "Lorem ipsum"
        private val EVENTUALLY_CONFIG = EventuallyConfig(
            duration = 200.milliseconds,
            interval = 10.milliseconds.fixed(),
            exceptionClass = IOException::class
        )
    }
}